﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Log;

namespace EffectMaker.BusinessLogic.ViewerMessages
{
    /// <summary>
    /// A queue that holds viewer messages and send them out in the exact order
    /// they are pushed to the queue.
    /// This class can be used for viewer messages that are created asynchronously
    /// to make sure messages are sent sequentially.
    /// E.g. When SendBinaryMessage of an emitter set is created, the shader binary
    /// is done in another thread, and the following preview and visibility messages
    /// have to wait until the emitter set binary is sent.
    /// In this case, all the messages are created and pushed to this queue in the
    /// correct order, while the shader binary is being converted asynchronously.
    /// Once the shader binary is done, the emitter set binary message is sent,
    /// and the rest of the messages being sent afterwards.
    /// </summary>
    public static class SendMessageQueue
    {
        /// <summary>
        /// メッセージキューです。
        /// </summary>
        private static readonly Queue<MessageBase> MessageQueue = new Queue<MessageBase>();

        /// <summary>
        /// メッセージドロップ区間のネスト数です。
        /// </summary>
        private static int messageDropNestCount { get; set; }

        /// <summary>
        /// Enqueue the given message and send after all previous message are sent.
        /// </summary>
        /// <param name="message">The message to enqueue.</param>
        public static void Enqueue(MessageBase message)
        {
            // メッセージドロップフラグを設定
            if (messageDropNestCount > 0)
            {
                message.ShouldDropMessage = true;
            }

            if (message.Status == MessageStatus.Canceled)
            {
                // Just skip the message if it's already being canceled.
            }
            else if (MessageQueue.Count <= 0 && message.Status == MessageStatus.ReadyToSend)
            {
                message.SendEnqueuedMessage();
            }
            else
            {
                MessageQueue.Enqueue(message);
            }
        }

        /// <summary>
        /// Cancel all the messages that matches the conditions.
        /// </summary>
        /// <param name="match">The function to determine if the message should be canceled.</param>
        public static void CancelAll(Func<MessageBase, bool> match)
        {
            var tmp = MessageQueue.Where(match).ToArray();
            tmp.ForEach(m => m.CancelMessage());
        }

        /// <summary>
        /// メッセージドロップ区間の開始位置を指定します。
        /// </summary>
        public static void BeginMessageDropSection()
        {
            ++messageDropNestCount;
        }

        /// <summary>
        /// メッセージドロップ区間の終了位置を指定します。
        /// </summary>
        public static void EndMessageDropSection()
        {
            System.Diagnostics.Debug.Assert(messageDropNestCount > 0, "メッセージドロップ区間の指定が不正");

            --messageDropNestCount;
        }

        /// <summary>
        /// Notify the queue that the status of the specified message has changed.
        /// If the status is ready and the message is the first on the queue, the
        /// message will be sent out.
        /// </summary>
        /// <param name="message">The message.</param>
        public static void NotifyMessageStatusChanged(MessageBase message)
        {
            if (MessageQueue.Count > 0 && MessageQueue.Peek() == message)
            {
                ProcessMessageQueue();
            }
        }

        /// <summary>
        /// Process the messages in the queue.
        /// Pop out the messages if they are ready or canceled.
        /// </summary>
        private static void ProcessMessageQueue()
        {
            while (MessageQueue.Count > 0)
            {
                MessageBase message = MessageQueue.Peek();
                switch (message.Status)
                {
                    // The message is not ready yet, skip the rest of the messages.
                    // We will keep process them once this message's status has changed.
                    case MessageStatus.Awaiting:
                        return;

                    // The message is ready to be sent, pop it out and send it.
                    case MessageStatus.ReadyToSend:
                        MessageQueue.Dequeue();
                        message.SendEnqueuedMessage();
                        break;

                    // The message is either canceled or at a unrecognized status,
                    // just pop it out and discard it.
                    default:
                        MessageQueue.Dequeue();
                        break;
                }
            }
        }
    }
}
