﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Runtime.CompilerServices;

using EffectMaker.BusinessLogic.IO;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Manager;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.Foundation.Compiler;
using EffectMaker.Foundation.Debugging.Profiling;
using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Serialization;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.BusinessLogic.UserData
{
    /// <summary>
    /// The user data manager.
    /// </summary>
    public static class UserDataManager
    {
        /// <summary>Set this constant flag to true if you don't want to use the cache.</summary>
        private const bool AlwaysConvertObsoleteUserData = true;

        /// <summary>The mapping of names and user data model owner types.</summary>
        private static readonly Dictionary<string, Type> OwnerNameTypeMap = new Dictionary<string, Type>()
        {
            { "Emitter", typeof(EmitterData) },
            { "Preview", typeof(PreviewData) },
            { "StripeComplex", typeof(StripeComplexData) },
            { "StripeHistory", typeof(StripeHistoryData) },
            { "StripeSuper", typeof(StripeSuperData) },
            { "Model", typeof(ModelData) },
            { "CustomAction", typeof(CustomActionData) },
            { "CustomShader", typeof(EmitterCustomShaderData) },
            { "ReservedShader", typeof(ReservedShaderNodeData) },
            { "EmitterExtParams", typeof(EmitterExtParams) },
        };

        /// <summary>
        /// The system assemblies to reference when compiling data models.
        /// </summary>
        private static readonly string[] ReferencedSystemAssemblyPaths = new string[]
        {
            "System.Core.dll",
            "System.dll",
            "System.Windows.Forms.dll",
            "System.Xml.dll",
        };

        /// <summary>
        /// Referenced application assembly paths for compiling the user data models.
        /// </summary>
        private static readonly string[] ReferencedApplicationAssemblyPaths = new string[]
        {
            "EffectMaker.DataModel.Specific.dll",
            "EffectMaker.DataModel.dll",
            "EffectMaker.DataModelLogic.dll",
            "EffectMaker.Foundation.dll",
        };

        /// <summary>The base user data model type.</summary>
        private static readonly Type UserDataModelType = typeof(UserDataBase);

        /// <summary>The base binary conversion info type.</summary>
        private static Type conversionInfoType = null;

        /// <summary>All the referenced assembly paths for compiling the user data models.</summary>
        private static string[] referencedAssemblyPaths;

        /// <summary>Maps the loaded user data instances with their name.</summary>
        private static Dictionary<string, UserDataInfo> userDataNameMap =
            new Dictionary<string, UserDataInfo>();

        /// <summary>Maps the user data types to their user data info.</summary>
        private static Dictionary<Type, UserDataInfo> userDataTypeMap =
            new Dictionary<Type, UserDataInfo>();

        /// <summary>Maps the owner type of the user data instances to attach to.</summary>
        private static Dictionary<Type, List<UserDataInfo>> userDataOwnerMap =
            new Dictionary<Type, List<UserDataInfo>>();

        /// <summary>The collection of cached obsolete user data conversion info.</summary>
        private static ObsoleteUserDataConversionInfoCollection cachedUserDataConversionInfo;

        /// <summary>
        /// Event triggered when user data updated.
        /// </summary>
        public static event EventHandler<UserDataUpdatedEventArgs> UserDataUpdated;

        /// <summary>
        /// Enumerates the loaded binary conversion info class types.
        /// </summary>
        public static IEnumerable<Type> ImportedBinaryConversionInfoTypes
        {
            get { return userDataNameMap.Select(p => p.Value.BinaryConversionInfoType); }
        }

        /// <summary>
        /// Enumerates the loaded user data model class types.
        /// </summary>
        public static IEnumerable<Type> ImportedUserDataModelTypes
        {
            get { return userDataNameMap.SelectMany(p => p.Value.ExportedUserDataTypes); }
        }

        /// <summary>
        /// Initialize the user data manager.
        /// </summary>
        /// <param name="binaryConversionInfoType">The binary conversion info class type.</param>
        public static void Initialize(Type binaryConversionInfoType)
        {
            // Set binary conversion info type.
            conversionInfoType = binaryConversionInfoType;

            // Process referenced assembly paths.
            int totalNumRefAssemblies =
                ReferencedSystemAssemblyPaths.Length + ReferencedApplicationAssemblyPaths.Length;

            referencedAssemblyPaths = new string[totalNumRefAssemblies];

            int index = 0;
            foreach (string path in ReferencedSystemAssemblyPaths)
            {
                referencedAssemblyPaths[index] = path;
                ++index;
            }

            foreach (string path in ReferencedApplicationAssemblyPaths)
            {
                referencedAssemblyPaths[index] =
                    Path.Combine(IOConstants.CoreModulesPath, path);
                ++index;
            }

            // Load the cached user data conversion info collection.
            var obsoleteUserDataConversionInfoPath = Path.Combine(
                IOConstants.ObsoleteUserDataConvertFolderPath,
                "CachedUserDataConversionInfo.xml");

            if (File.Exists(obsoleteUserDataConversionInfoPath) == true)
            {
                cachedUserDataConversionInfo = SerializationHelper.LoadXmlDocSerializable<ObsoleteUserDataConversionInfoCollection>(
                    obsoleteUserDataConversionInfoPath);
            }
            else
            {
                cachedUserDataConversionInfo = new ObsoleteUserDataConversionInfoCollection();
            }

            // Register event to provide user data types in the loaded assemblies.
            SerializationHelper.RequestExtraTypes += (s, e) =>
            {
                // Add all the exported user data types for serialization.
                userDataNameMap.Values.ForEach(
                    info => e.AddExtraTypes(info.ExportedUserDataTypes));
            };
        }

        /// <summary>
        /// Clear the specified user data.
        /// </summary>
        /// <param name="name">The name to identify the user data.</param>
        public static void ClearUserData(string name)
        {
            UserDataInfo info;
            if (userDataNameMap.TryGetValue(name, out info) == false)
            {
                // There is no user data registered with the given name.
                return;
            }

            // Remove the user data.
            userDataNameMap.Remove(name);
            userDataTypeMap.Remove(info.DataModelType);

            List<UserDataInfo> userDataListForOwner;
            if (userDataOwnerMap.TryGetValue(info.OwnerType, out userDataListForOwner) == true)
            {
                userDataListForOwner.RemoveAll(it => it != null && it.Name == name);
            }
        }

        /// <summary>
        /// Clear the user data with the owner name.
        /// </summary>
        /// <param name="ownerName">The name of the user data owner.</param>
        public static void ClearUserDataByOwner(string ownerName)
        {
            // Find the owner type.
            Type ownerType;
            if (OwnerNameTypeMap.TryGetValue(ownerName, out ownerType) == false)
            {
                // There is no owner being registered with the name.
                return;
            }

            // Find the user data registered under the owner.
            List<UserDataInfo> userDataListForOwner;
            if (userDataOwnerMap.TryGetValue(ownerType, out userDataListForOwner) == false)
            {
                // There is no user data registered under the owner.
                return;
            }

            foreach (UserDataInfo info in userDataListForOwner)
            {
                if (info == null)
                {
                    continue;
                }

                // Remove the user data.
                userDataNameMap.Remove(info.Name);
                userDataTypeMap.Remove(info.DataModelType);
            }

            userDataOwnerMap.Remove(ownerType);
        }

        /// <summary>
        /// Add an empty user data to the owner.
        /// </summary>
        /// <param name="ownerName">The name of the owner to attach the user data to.</param>
        public static void AddEmptyUserDataToOwner(string ownerName)
        {
            // Find the owner type.
            Type ownerType;
            if (OwnerNameTypeMap.TryGetValue(ownerName, out ownerType) == false)
            {
                Logger.Log(
                    LogLevels.Warning,
                    "UserDataManager.LoadUserData : The assigned owner name '{0}' does not exist.",
                    ownerName);
                return;
            }

            // Add an null entry to the owner's user data list.
            List<UserDataInfo> userDataListForOwner;
            if (userDataOwnerMap.TryGetValue(ownerType, out userDataListForOwner) == false)
            {
                userDataListForOwner = new List<UserDataInfo>();
                userDataOwnerMap[ownerType] = userDataListForOwner;
            }

            userDataListForOwner.Add(null);
        }

        /// <summary>
        /// Load and compile user data from UDD2.0
        /// </summary>
        /// <param name="name">The name to identify the user data.</param>
        /// <param name="ownerName">The name of the owner to attach the user data to.</param>
        /// <param name="definitionFilePath">UDD2.0の定義ファイルパス</param>
        /// <param name="additionalRefencedAssemblyPaths">Additional assembly paths the user data references.</param>
        /// <returns>True on success.</returns>
        public static bool LoadUdd2(
            string name,
            string ownerName,
            string definitionFilePath,
            IEnumerable<string> additionalRefencedAssemblyPaths = null)
        {
            // Find the owner type and the user data info.
            Type ownerType;
            UserDataInfo info;
            if (!GetTypeAndInfo(ownerName, name, out ownerType, out info))
            {
                return false;
            }

            if (!UserDataHelper.Instance.Udd2Converter.LoadDefinition(definitionFilePath))
            {
                Logger.Log(
                    "LogView",
                    LogLevels.Error,
                    "UserDataManager.LoadUserDataFromUdd2 : Failed to load a definition at '{0}'.",
                    UserDataHelper.Instance.Udd2Converter.ErrorReport.Replace(Environment.NewLine, "/"));
                return false;
            }

            string warnings = UserDataHelper.Instance.Udd2Converter.ErrorReport;
            if (!string.IsNullOrEmpty(warnings))
            {
                // ロードには成功したが警告がある場合はログビューにずらずら出す
                Logger.Log(
                    "LogView",
                    LogLevels.Warning,
                    "UserDataManager.LoadUserDataFromUdd2 : Warnins on load a definition at '{0}'.",
                    definitionFilePath);
                foreach (var w in warnings.Split(new[] { Environment.NewLine }, StringSplitOptions.RemoveEmptyEntries))
                {
                    Logger.Log("LogView", LogLevels.Warning, w);
                }
            }

            if (UserDataHelper.Instance.Udd2Converter.UserDataType != ownerName)
            {
                Logger.Log(
                    "LogView",
                    LogLevels.Warning,
                    "UserDataManager.LoadUserDataFromUdd2 : The loaded definition is not matched data type '{0}'.",
                    ownerName);
                return false;
            }

            if (!UserDataHelper.Instance.Udd2Converter.Convert())
            {
                Logger.Log(
                    "LogView",
                    LogLevels.Error,
                    "UserDataManager.LoadUserDataFromUdd2 : Failed to generate codes at '{0}'.",
                    UserDataHelper.Instance.Udd2Converter.ErrorReport.Replace(Environment.NewLine, "/"));
                return false;
            }

            info.XamlFilePath = definitionFilePath;
            info.XamlCode = UserDataHelper.Instance.Udd2Converter.XamlCode;
            info.XPathTable = UserDataHelper.Instance.Udd2Converter.XPathTable;

            // Prepare data for compiling the user data.
            var sourceCodes = new string[]
            {
                UserDataHelper.Instance.Udd2Converter.DataModelCode,
                UserDataHelper.Instance.Udd2Converter.BinaryConversionInfoCode
            };

            var refAssemblies = referencedAssemblyPaths;
            if (additionalRefencedAssemblyPaths != null)
            {
                refAssemblies.Concat(additionalRefencedAssemblyPaths);
            }

            // Compile the data model and the binary conversion info.
            Assembly assembly = RuntimeCompiler.CompileFromMemory(
                name,
                new [] { definitionFilePath },
                refAssemblies,
                sourceCodes);

            if (assembly == null)
            {
                Logger.Log(
                    "LogView",
                    LogLevels.Error,
                    "UserDataManager.LoadUserDataFromUdd2 : Failed compiling user data.");
                Logger.Log(LogLevels.Warning, "UDD2.0 : " + definitionFilePath);
                return false;
            }

            return CheckAssemblyContents(assembly, info, name, ownerType);
        }

        /// <summary>
        /// Load and compile user data.
        /// </summary>
        /// <param name="name">The name to identify the user data.</param>
        /// <param name="ownerName">The name of the owner to attach the user data to.</param>
        /// <param name="dataModelPath">The source file path of the data model.</param>
        /// <param name="conversionInfoPath">The source file path of the binary conversion info.</param>
        /// <param name="xamlFilePath">The XAML file path.</param>
        /// <param name="additionalRefencedAssemblyPaths">Additional assembly paths the user data references.</param>
        /// <returns>True on success.</returns>
        public static bool LoadUserData(
            string name,
            string ownerName,
            string dataModelPath,
            string conversionInfoPath,
            string xamlFilePath,
            IEnumerable<string> additionalRefencedAssemblyPaths = null)
        {
            // Find the owner type and the user data info.
            Type ownerType;
            UserDataInfo info;
            if (!GetTypeAndInfo(ownerName, name, out ownerType, out info))
            {
                return false;
            }

            info.DataModelSourceFilePath = dataModelPath;
            info.BinaryConversionInfoPath = conversionInfoPath;
            info.XamlFilePath = xamlFilePath;

            // Prepare data for compiling the user data.
            var sourcePaths = new string[] { dataModelPath, conversionInfoPath };

            var refAssemblies = referencedAssemblyPaths;
            if (additionalRefencedAssemblyPaths != null)
            {
                refAssemblies.Concat(additionalRefencedAssemblyPaths);
            }

            // Compile the data model and the binary conversion info.
            Assembly assembly = RuntimeCompiler.CompileFromFile(
                name,
                sourcePaths,
                refAssemblies,
                sourcePaths);

            if (assembly == null)
            {
                Logger.Log(
                    LogLevels.Warning,
                    "UserDataManager.LoadUserData : Failed compiling user data.");
                Logger.Log(LogLevels.Warning, "User data model : " + dataModelPath);
                Logger.Log(LogLevels.Warning, "Binary conversion info : " + conversionInfoPath);
                return false;
            }

            return CheckAssemblyContents(assembly, info, name, ownerType);
        }

        /// <summary>
        /// Trigger user data changed event.
        /// </summary>
        /// <param name="userDataType">The updated user data type.</param>
        public static void TriggerUserDataChangedEvent(UpdatedUserDataTypes userDataType)
        {
            // Fire event to notify that user data being updated.
            if (UserDataUpdated != null)
            {
                UserDataUpdated(null, new UserDataUpdatedEventArgs(userDataType));
            }
        }

        /// <summary>
        /// Find the user data info by its name.
        /// </summary>
        /// <param name="name">The name of the user data.</param>
        /// <returns>The user data info, or null if the name is not found.</returns>
        public static UserDataInfo FindUserDataByName(string name)
        {
            UserDataInfo info;
            if (userDataNameMap.TryGetValue(name, out info) == false)
            {
                return null;
            }

            return info;
        }

        /// <summary>
        /// Create user data instances from the given user data info.
        /// </summary>
        /// <param name="info">The user data info.</param>
        /// <returns>The created user data instances.</returns>
        public static UserDataBase CreateUserDataFromInfo(UserDataInfo info)
        {
            if (info == null ||
                info.DataModelType == null ||
                info.DataModelType.GetConstructor(Type.EmptyTypes) == null)
            {
                return null;
            }

            return Activator.CreateInstance(info.DataModelType) as UserDataBase;
        }

        /// <summary>
        /// Create user data instances for the owner of the given type.
        /// </summary>
        /// <param name="ownerType">The owner type.</param>
        /// <returns>The created user data instances.</returns>
        public static IEnumerable<UserDataBase> CreateUserDataForOwner(Type ownerType)
        {
            if (ownerType == null)
            {
                return Enumerable.Empty<UserDataBase>();
            }

            List<UserDataInfo> childUserDataInfoList;
            if (userDataOwnerMap.TryGetValue(ownerType, out childUserDataInfoList) == false)
            {
                return Enumerable.Empty<UserDataBase>();
            }

            return from info in childUserDataInfoList
                   select info == null ? null : Activator.CreateInstance(info.DataModelType) as UserDataBase;
        }

        /// <summary>
        /// Enumerate user data information instances for the owner of the given type.
        /// </summary>
        /// <param name="ownerName">The owner name.</param>
        /// <returns>The user data information instances.</returns>
        public static IEnumerable<UserDataInfo> EnumerateUserDataInfoForOwner(string ownerName)
        {
            // Find the owner type.
            Type ownerType;
            if (OwnerNameTypeMap.TryGetValue(ownerName, out ownerType) == false)
            {
                return Enumerable.Empty<UserDataInfo>();
            }

            if (ownerType == null)
            {
                return Enumerable.Empty<UserDataInfo>();
            }

            List<UserDataInfo> childUserDataInfoList;
            if (userDataOwnerMap.TryGetValue(ownerType, out childUserDataInfoList) == false)
            {
                return Enumerable.Empty<UserDataInfo>();
            }

            return childUserDataInfoList;
        }

        /// <summary>
        /// Get the XAML file path for the given user data type.
        /// </summary>
        /// <param name="userDataType">The type of the user data.</param>
        /// <returns>The XAML file path.</returns>
        public static string GetUserDataXamlPath(Type userDataType)
        {
            if (userDataType == null)
            {
                return null;
            }

            UserDataInfo info;
            if (userDataTypeMap.TryGetValue(userDataType, out info) == true)
            {
                return info.XamlFilePath;
            }
            else
            {
                return null;
            }
        }

        /// <summary>
        /// Get the output folder path of the converted usd/udd files.
        /// </summary>
        /// <param name="filePath">The udd/usd file path.</param>
        /// <param name="dataModelName">The user data model class name.</param>
        /// <returns>The output folder path.</returns>
        public static string GetConvertedObsoleteUserDataOutputFolder(
            string filePath,
            string dataModelName)
        {
            var cachedInfo = cachedUserDataConversionInfo[filePath];
            if (cachedInfo != null)
            {
                return cachedInfo.OutputFolderPath;
            }
            else
            {
                return Path.Combine(
                    IOConstants.ObsoleteUserDataConvertFolderPath,
                    dataModelName + "_" + MakeUniqueHashString(filePath));
            }
        }

        /// <summary>
        /// 旧式のユーザー定義ファイル(.usd, .udd)を新形式(.xml)にコンバートします。
        /// </summary>
        /// <param name="infos">コンバートするユーザー定義ファイルの情報</param>
        /// <param name="updateCachedInfo">キャッシュ情報を更新するかどうか</param>
        /// <returns>処理が成功したときTrueを返します。</returns>
        public static bool ConvertObsoleteUserData(IEnumerable<ObsoleteUserDataConversionInfo> infos, bool updateCachedInfo)
        {
            if (UserDataHelper.Instance.ObsoleteUserDataConverter == null)
            {
                return false;
            }

            bool converted = false;  // コンバート処理を行ったか

            // コンバートを実行
            foreach (ObsoleteUserDataConversionInfo info in infos)
            {
                // ユーザー定義ファイルに対応するキャッシュ情報を取得
                var cachedInfo = cachedUserDataConversionInfo[info.FilePath];

                info.OutputFolderPath = Path.Combine(
                    IOConstants.ObsoleteUserDataConvertFolderPath,
                    info.DataModelName + "_" + MakeUniqueHashString(info.FilePath));

                if (cachedInfo != null)
                {
                    // 対応するキャッシュ情報があったとき、ファイルの変更をチェック
                    var lastModifyTime = File.GetLastWriteTimeUtc(cachedInfo.FilePath);
                    ////if (AlwaysConvertObsoleteUserData == false &&
                    ////    cachedInfo.ModifyTime == lastModifyTime &&
                    ////    cachedInfo.DataModelName == info.DataModelName &&
                    ////    cachedInfo.ConversionCommandName == info.ConversionCommandName &&
                    ////    cachedInfo.Language == info.Language &&
                    ////    cachedInfo.OutputFolderPath == info.OutputFolderPath)
                    ////{
                    ////    // ユーザー定義ファイルに変更がなかったとき、コマンド引数への追加をスキップ
                    ////    continue;
                    ////}
                    ////else
                    {
                        // ユーザー定義ファイルに変更があったとき、キャッシュ情報を更新
                        cachedInfo.ModifyTime = lastModifyTime;
                        cachedInfo.DataModelName = info.DataModelName;
                        cachedInfo.ConversionCommandName = info.ConversionCommandName;
                        cachedInfo.Language = info.Language;
                        cachedInfo.OutputFolderPath = info.OutputFolderPath;
                    }
                }
                else
                {
                    // 対応するキャッシュ情報がなかったとき、新たにキャッシュ情報を追加する
                    info.ModifyTime = File.GetLastWriteTimeUtc(info.FilePath);

                    if (updateCachedInfo == true)
                    {
                        cachedUserDataConversionInfo[info.FilePath] = info;
                    }
                }

                bool resConvert = false;

                // コンバートを実行
                if (info.ConversionCommandName == "usd")
                {
                    resConvert = UserDataHelper.Instance.ObsoleteUserDataConverter.ConvertUsd(info.FilePath, info.DataModelName, info.OutputFolderPath, info.Language);
                }
                else
                {
                    resConvert = UserDataHelper.Instance.ObsoleteUserDataConverter.ConvertUdd(info.FilePath, info.DataModelName, info.OutputFolderPath, info.Language);
                }

                if (resConvert == false)
                {
                    Logger.Log("LogView", LogLevels.Error, Properties.Resources.ErrorFailedLoadUddUsd);
                }

                converted = true;
            }

            // コンバートしたファイルがあり、フラグが立っていたときキャッシュ情報を保存する
            if (converted && updateCachedInfo)
            {
                using (new ProfileTimer("Saving .udd/.usd conversion cache"))
                {
                    // Save the user data conversion info to file.
                    var path = Path.Combine(
                        IOConstants.ObsoleteUserDataConvertFolderPath,
                        "CachedUserDataConversionInfo.xml");

                    // In case the directory hasn't already existed, create it.
                    if (Directory.Exists(IOConstants.ObsoleteUserDataConvertFolderPath) == false)
                    {
                        Directory.CreateDirectory(IOConstants.ObsoleteUserDataConvertFolderPath);
                    }

                    SerializationHelper.SaveXmlDocSerializable(cachedUserDataConversionInfo, path);
                }
            }

            return true;
        }

        /// <summary>
        /// USD/UDDからコンバートしたファイルを格納するフォルダ名に付加する文字列を取得します。
        /// </summary>
        /// <param name="filePath">USD/UDDのファイルパス</param>
        /// <returns>ユニークな文字列</returns>
        private static string MakeUniqueHashString(string filePath)
        {
            // ユーザー定義ファイルのファイルパスをもとにキャッシュファイル名を生成……していたが、
            // 並列コンバート時に事故るので、アプリごとにユニークなハッシュを付けたフォルダに入れる。
            // ユニークにした代わりに、終了時にTheApp.CleanupTemporaryFoldersできちんと消すようにする。

            ////int filePathHash = filePath.ToLower().GetHashCode();
            ////var uniqueHash = filePathHash.ToString("X");
            var uniqueHash = IOConstants.ApplicationInstanceHash;

            return uniqueHash;
        }

        /// <summary>
        /// ユーザーデータオーナーの型名と名前から、型情報とユーザーデータ情報を取得します。
        /// </summary>
        /// <param name="ownerName">ユーザーデータオーナー名</param>
        /// <param name="name">ユーザーデータ名</param>
        /// <param name="ownerType">ユーザーデータオーナー型</param>
        /// <param name="info">ユーザーデータ情報</param>
        /// <returns>該当情報が見つかればtrue,どうでなければfalse.</returns>
        private static bool GetTypeAndInfo(string ownerName, string name, out Type ownerType, out UserDataInfo info)
        {

            // Find the owner type.
            if (OwnerNameTypeMap.TryGetValue(ownerName, out ownerType) == false)
            {
                Logger.Log(
                    LogLevels.Warning,
                    "UserDataManager.LoadUserData : The assigned owner name '{0}' does not exist.",
                    ownerName);
                info = null;
                return false;
            }

            // Find the user data info.
            if (userDataNameMap.TryGetValue(name, out info) == false)
            {
                info = new UserDataInfo()
                {
                    Name = name,
                    OwnerType = ownerType,
                };
            }

            return true;
        }

        /// <summary>
        /// コンパイルしたアセンブリの内容を検証します。
        /// </summary>
        /// <param name="assembly">アセンブリ</param>
        /// <param name="info">ユーザーデータ情報</param>
        /// <param name="name">ユーザーデータ名</param>
        /// <param name="ownerType">ユーザーデータを保持する型</param>
        /// <returns>正しいアセンブリであればtrue,異なるデータ型が混入しているなど不正があったらfalse.</returns>
        private static bool CheckAssemblyContents(Assembly assembly, UserDataInfo info, string name, Type ownerType)
        {
            if (assembly == info.Assembly)
            {
                // The assembly is not changed, just return.
                return true;
            }

            // Find the type of the user data model in the compiled assembly.
            info.DataModelType =
                assembly.ExportedTypes.FirstOrDefault(t => t.IsSubclassOf(UserDataModelType));
            if (info.DataModelType == null)
            {
                Logger.Log(
                    LogLevels.Warning,
                    "UserDataManager.LoadUserData : There is no user data model found in the compiled assembly.");
                return false;
            }

            // Check if any other user data already uses the same user data model.
            UserDataInfo userDataWithSameDataModelType;
            if (userDataTypeMap.TryGetValue(info.DataModelType, out userDataWithSameDataModelType) == true &&
                userDataWithSameDataModelType.Name != name)
            {
                Logger.Log(
                    LogLevels.Warning,
                    "UserDataManager.LoadUserData : The user data model '{0}' is already used in {1}.",
                    info.DataModelType.Name,
                    userDataWithSameDataModelType.Name);
                return false;
            }

            // Find the type of the binary conversion info in the compiled assembly.
            info.BinaryConversionInfoType =
                assembly.ExportedTypes.FirstOrDefault(t => t.IsSubclassOf(conversionInfoType));
            if (info.BinaryConversionInfoType == null)
            {
                // Show warning but don't bail out, the user data can still be used
                // but could not be sent to the viewer.
                Logger.Log(
                    LogLevels.Warning,
                    "UserDataManager.LoadUserData : There is no binary conversion info found in the compiled assembly.");
            }

            // Run static constructor of the binary conversion info to manually initialize it.
            RuntimeHelpers.RunClassConstructor(info.BinaryConversionInfoType.TypeHandle);

            // Collect all the exported user data types.
            var types = from t in assembly.GetExportedTypes()
                        where t.IsClass && t.IsSubclassOf(UserDataModelType)
                        select t;

            info.ExportedUserDataTypes = types.ToList();

            // Save the user data.
            userDataNameMap[name] = info;
            userDataTypeMap[info.DataModelType] = info;

            List<UserDataInfo> userDataListForOwner;
            if (userDataOwnerMap.TryGetValue(ownerType, out userDataListForOwner) == false)
            {
                userDataListForOwner = new List<UserDataInfo>();
                userDataOwnerMap[ownerType] = userDataListForOwner;
            }

            userDataListForOwner.Add(info);

            return true;
        }
    }
}
