﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;

using EffectMaker.BusinessLogic.IO;

using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Serialization;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.BusinessLogic.UserData
{
    /// <summary>
    /// Helper class to load the Emitter Extension Parameters definitions.
    /// </summary>
    public class EmitterExtParamsDefinitionLoader
    {
        /// <summary>The owner name for the Emitter Extension Parameters.</summary>
        public const string EmitterExtParamsOwnerName = "EmitterExtParams";

        /// <summary>
        /// Load the Emitter Extension Parameters definitions from the specified file.
        /// </summary>
        /// <remarks>
        /// This method ONLY loads the definitions.
        /// It DOES NOT compile the data model and binary conversion
        /// info source files, and it DOES NOT load the XAML files.
        /// </remarks>
        /// <param name="filePath">The file path.</param>
        /// <returns>The loaded definitions.</returns>
        public EmitterExtParamsDefinition LoadDefinition(string filePath)
        {
            if (string.IsNullOrEmpty(filePath) == true || File.Exists(filePath) == false)
            {
                return null;
            }

            // Load the definitions.
            var defList = SerializationHelper.LoadXmlDocSerializable<EmitterExtParamsDefinitionList>(filePath);
            if (defList == null)
            {
                return null;
            }

            var def = defList.EmitterExtParamsDefinition;
            if (def == null)
            {
                return null;
            }

            // Set up the base folder path, because the paths in the definitions
            // are usually relative paths or even just file names.
            def.BaseFolderPath = Path.GetDirectoryName(filePath);

            return def;
        }

        /// <summary>
        /// Set up the custom actions.
        /// Compiles the data model and binary conversion info, and saves
        /// the XAML file path for later inquiry.
        /// </summary>
        /// <param name="defList">The custom action definitions.</param>
        /// <returns>True on success.</returns>
        public bool SetupUserData(EmitterExtParamsDefinition def)
        {
            UserDataManager.ClearUserDataByOwner(EmitterExtParamsOwnerName);

            if (def == null)
            {
                UserDataManager.AddEmptyUserDataToOwner(EmitterExtParamsOwnerName);
            }
            else if (def.UsingUdd2)
            {
                // Load the user data.
                bool result = UserDataManager.LoadUdd2(
                    EmitterExtParamsOwnerName,
                    EmitterExtParamsOwnerName,
                    def.Udd2FullFilePath);

                if (result == false)
                {
                    Logger.Log(LogLevels.Warning, "Failed setting up Emitter Extension Parameters.");
                    return false;
                }
            }
            else
            {
                // Load the user data.
                // The data model and binary conversion info will be compiled to assembly
                // and the XAML file path is saved in the user data manager.
                bool result = UserDataManager.LoadUserData(
                    EmitterExtParamsOwnerName,
                    EmitterExtParamsOwnerName,
                    def.DataModelFullFilePath,
                    def.BinaryConversionInfoFullPath,
                    def.XamlFullFilePath);

                if (result == false)
                {
                    Logger.Log(LogLevels.Warning, "Failed setting up Emitter Extension Parameters.");
                    return false;
                }
            }

            EmitterExtParamsUserDataManager.Definition = def;

            return true;
        }
    }
}
