﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;

using EffectMaker.BusinessLogic.IO;

using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Serialization;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.BusinessLogic.UserData
{
    /// <summary>
    /// Helper class to load the custom shader definitions.
    /// </summary>
    public class CustomShaderDefinitionLoader
    {
        /// <summary>The owner name for the custom shader.</summary>
        public const string CustomShaderOwnerName = "CustomShader";

        /// <summary>The string to format the custom shader names.</summary>
        public const string CustomShaderNamePattern = "CustomShader_{0}";

        /// <summary>
        /// Constructor.
        /// </summary>
        public CustomShaderDefinitionLoader()
        {
        }

        /// <summary>
        /// Load the custom shader definitions from the specified file.
        /// </summary>
        /// <remarks>
        /// This method ONLY loads the definitions.
        /// It DOES NOT compile the data model and binary conversion
        /// info source files, and it DOES NOT load the XAML files.
        /// </remarks>
        /// <param name="filePath">The file path.</param>
        /// <returns>The loaded definitions.</returns>
        public CustomShaderDefinitionList LoadDefinition(string filePath)
        {
            if (string.IsNullOrEmpty(filePath) == true || File.Exists(filePath) == false)
            {
                return null;
            }

            // Load the definitions.
            CustomShaderDefinitionList defList =
                SerializationHelper.LoadXmlDocSerializable<CustomShaderDefinitionList>(filePath);
            if (defList == null)
            {
                return null;
            }

            // Set up the file path, because the paths in the definitions
            // are usually relative paths or even just file names.
            defList.FilePath = filePath;

            // Load custom shader definitions.
            defList.LoadCustomShaderDefinitionFiles();

            return defList;
        }

        /// <summary>
        /// Set up the custom shaders.
        /// Compiles the data model and binary conversion info, and saves
        /// the XAML file path for later inquiry.
        /// </summary>
        /// <param name="defList">The custom action definitions.</param>
        /// <returns>True on success.</returns>
        public bool SetupUserData(CustomShaderDefinitionList defList)
        {
            UserDataManager.ClearUserDataByOwner(CustomShaderOwnerName);

            if (defList == null)
            {
                CustomShaderUserDataManager.Definition = null;
                return true;
            }

            int index = 0;
            foreach (CustomShaderDefinition def in defList.CustomShaderDefinitions)
            {
                bool result;
                if (def.UsingUdd2)
                {
                    result = UserDataManager.LoadUdd2(
                        string.Format(CustomShaderNamePattern, index),
                        CustomShaderOwnerName,
                        def.Udd2FullFilePath);
                }
                else
                {
                    // Load the user data.
                    // The data model and binary conversion info will be compiled to assembly
                    // and the XAML file path is saved in the user data manager.
                    result = UserDataManager.LoadUserData(
                        string.Format(CustomShaderNamePattern, index),
                        CustomShaderOwnerName,
                        def.DataModelFilePath,
                        def.BinaryConversionInfoPath,
                        def.XamlFilePath);
                }

                if (result == false)
                {
                    Logger.Log(LogLevels.Warning, "Failed setting up custom shader {0}.", index);
                    return false;
                }

                ++index;
                if (index >= 8)
                {
                    // There can only be 8 custom shader definitions at most.
                    break;
                }
            }

            CustomShaderUserDataManager.Definition = defList;

            return true;
        }

        /// <summary>
        /// Collect obsolete user data (e.g. .usd definitions) for conversion.
        /// </summary>
        /// <param name="defList">The custom shader definitions.</param>
        /// <returns>The collected information.</returns>
        public List<ObsoleteUserDataConversionInfo> CollectObsoleteUserDataToConvert(CustomShaderDefinitionList defList)
        {
            var infos = new List<ObsoleteUserDataConversionInfo>();

            if (defList == null)
            {
                return infos;
            }

            string language = "jp";

            var culture = System.Threading.Thread.CurrentThread.CurrentUICulture;
            var cultureEnUs = System.Globalization.CultureInfo.CreateSpecificCulture("en-US");
            if (culture.Equals(cultureEnUs) == true)
            {
                language = "en";
            }

            foreach (ObsoleteCustomShaderDefinition info in defList.UsdDefinitions)
            {
                var convertionInfo = new ObsoleteUserDataConversionInfo()
                {
                    FilePath = info.UsdFilePath,
                    DataModelName = Path.GetFileNameWithoutExtension(info.UsdFilePath),
                    ConversionCommandName = "usd",
                    Language = language,
                };

                infos.Add(convertionInfo);
            }

            return infos;
        }

        /// <summary>
        /// Prepare the converted .udd file definitions for use.
        /// </summary>
        /// <param name="defList">The custom shader definitions.</param>
        public void PrepareConvertedObsoleteUserData(CustomShaderDefinitionList defList)
        {
            if (defList == null)
            {
                return;
            }

            foreach (ObsoleteCustomShaderDefinition info in defList.UsdDefinitions)
            {
                string dataModelName = Path.GetFileNameWithoutExtension(info.UsdFilePath);

                string outputPath = UserDataManager.GetConvertedObsoleteUserDataOutputFolder(
                    info.UsdFilePath,
                    dataModelName);

                info.DefinitionPath = Path.Combine(outputPath, dataModelName + ".xml");
                info.LoadCustomShaderDefinitionFile();
            }
        }
    }
}
