﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Serializer;
using EffectMaker.Foundation.Log;

namespace EffectMaker.BusinessLogic.Serializer
{
    /// <summary>
    /// データモデルのシリアライザです。
    /// </summary>
    public abstract class DataModelSerializer
    {
        /// <summary>
        /// シリアライズを行います。
        /// </summary>
        /// <param name="dataModel">シリアライズする値</param>
        /// <param name="stream">出力ストリーム</param>
        /// <returns>シリアライズ結果を返します。</returns>
        public abstract SerializeResult Serialize(DataModelBase dataModel, Stream stream);

        /// <summary>
        /// デシリアライズを行います。
        /// </summary>
        /// <param name="value">デシリアライズする値</param>
        /// <returns>デシリアライズ結果を返します。</returns>
        public abstract DeserializeResult Deserialize(Stream stream);

        /// <summary>
        /// シリアライズ結果です。
        /// </summary>
        public class SerializeResult
        {
            /// <summary>
            /// コンストラクタです。
            /// </summary>
            public SerializeResult()
            {
            }

            /// <summary>
            /// コンストラクタです。
            /// </summary>
            /// <param name="obj">初期値</param>
            public SerializeResult(SerializeResult obj)
            {
                this.IsSuccess = obj.IsSuccess;
                this.Exception = obj.Exception;
            }

            /// <summary>
            /// シリアライズが成功したかどうか取得または設定します。
            /// </summary>
            public bool IsSuccess { get; set; }

            /// <summary>
            /// シリアライズ中に発生した例外を取得または設定します。
            /// </summary>
            public Exception Exception { get; set; }
        }

        /// <summary>
        /// デシリアライズ結果です。
        /// </summary>
        public class DeserializeResult
        {
            /// <summary>
            /// コンストラクタです。
            /// </summary>
            public DeserializeResult()
            {
            }

            /// <summary>
            /// コンストラクタです。
            /// </summary>
            /// <param name="obj">初期値</param>
            public DeserializeResult(DeserializeResult obj)
            {
                this.IsSuccess = obj.IsSuccess;
                this.Exception = obj.Exception;
                this.IsVersionMismatch = obj.IsVersionMismatch;
                this.DataModel = obj.DataModel;
            }

            /// <summary>
            /// デシリアライズが成功したかどうか取得または設定します。
            /// </summary>
            public bool IsSuccess { get; set; }

            /// <summary>
            /// デシリアライズ中に発生した例外を取得または設定します。
            /// </summary>
            public Exception Exception { get; set; }

            /// <summary>
            /// バージョン不一致のエラーが発生したかどうか取得または設定します。
            /// </summary>
            public bool IsVersionMismatch { get; set; }

            /// <summary>
            /// デシリアライズされたデータモデルを取得または設定します。
            /// </summary>
            public DataModelBase DataModel { get; set; }
        }
    }

    /// <summary>
    /// データモデルのシリアライザです。
    /// </summary>
    /// <typeparam name="T">データモデルタイプ</typeparam>
    public class DataModelSerializer<T> : DataModelSerializer
        where T : DataModelBase
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public DataModelSerializer()
        {
        }

        /// <summary>
        /// シリアライズを行います。
        /// </summary>
        /// <param name="dataModel">シリアライズする値</param>
        /// <param name="stream">出力ストリーム</param>
        /// <returns>シリアライズ結果を返します。</returns>
        public override SerializeResult Serialize(DataModelBase dataModel, Stream stream)
        {
            SerializeResult result = new SerializeResult();

            // 値がnullでないかチェック
            if (dataModel == null)
            {
                result.Exception = new ArgumentException("dataModel is null", "dataModel");

                return result;
            }

            // 値の型がシリアライズ可能かチェック
            if (dataModel is T == false)
            {
                result.Exception = new ArgumentException(string.Format("{0} is not supported.", dataModel.GetType().Name), "dataModel");

                return result;
            }

            // シリアライズを実行
            try
            {
                DataModelSerializerManager.Serialize(dataModel, stream);
            }
            catch (Exception e)
            {
                result.Exception = e;
            }

            // シリアライズ中に発生した例外を処理
            if (result.Exception != null)
            {
                Exception e =result.Exception;

                string errorMessage = new StringReader(e.ToString()).ReadLine();

                Logger.Log(LogLevels.Error, errorMessage);

                return result;
            }

            result.IsSuccess = true;

            return result;
        }

        /// <summary>
        /// デシリアライズを行います。
        /// </summary>
        /// <param name="value">デシリアライズする値</param>
        /// <returns>デシリアライズ結果を返します。</returns>
        public override DataModelSerializer.DeserializeResult Deserialize(Stream stream)
        {
            DeserializeResult result = new DeserializeResult();

            // ジェネリックで指定された型の情報を取得
            Type type = typeof(T);

            // デシリアライズを実行
            try
            {
                result.DataModel = (T)DataModelSerializerManager.Deserialize(stream, type);
            }
            catch (Exception e)
            {
                result.Exception = e;
            }

            // デシリアライズ中に発生した例外を処理
            if (result.Exception != null)
            {
                Exception e = result.Exception;

                if (e is IOException && e.Message == "Version mismatch.")
                {
                    result.IsVersionMismatch = true;
                }

                string errorMessage = new StringReader(e.ToString()).ReadLine();

                Logger.Log(LogLevels.Error, errorMessage);

                return result;
            }

            result.IsSuccess = true;

            return result;
        }

        /// <summary>
        /// デシリアライズ結果です。
        /// </summary>
        public new class DeserializeResult : DataModelSerializer.DeserializeResult
        {
            /// <summary>
            /// コンストラクタです。
            /// </summary>
            public DeserializeResult()
            {
            }

            /// <summary>
            /// コンストラクタです。
            /// </summary>
            /// <param name="obj">初期値</param>
            public DeserializeResult(DataModelSerializer.DeserializeResult obj)
                : base(obj)
            {
            }

            /// <summary>
            /// デシリアライズされたデータモデルを取得または設定します。
            /// </summary>
            public new T DataModel
            {
                get { return (T)base.DataModel; }
                set { base.DataModel = value; }
            }
        }
    }
}
