﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

// ビューアの初期化待ちを一時キャンセル
// キャンセルを取り消すとき、MessageManager.connectWaitMsec = 3000を0に
#define DISABLE_WAIT_APP

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.BusinessLogic.ProjectConfig;
using EffectMaker.BusinessLogic.ViewerMessages;
using EffectMaker.Communicator;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Primitives;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.BusinessLogic.Protocol
{
    /// <summary>
    /// ビューア制御クラスです.
    /// </summary>
    public class ViewerController : IConnectionStateListener, IMessageListener
    {
        /// <summary>
        /// ViewerMessage用のGUID定義です。
        /// ViewerMessageTypeの数と一致させます。
        /// </summary>
        private static readonly Guid[] MessageGuids =
        {
            Guid.NewGuid(),  // Config
            Guid.NewGuid(),  // ResetFrame
            Guid.NewGuid(),  // Play
            Guid.NewGuid(),  // Pause
            Guid.NewGuid(),  // Stop
            Guid.NewGuid(),  // StepFrame
            Guid.NewGuid(),  // ForceFade
            Guid.NewGuid(),  // SetRange
            Guid.NewGuid(),  // Camera
            Guid.NewGuid(),  // BackgroundImage
            Guid.NewGuid(),  // RequestToEditor
            Guid.NewGuid(),  // Visibility
            Guid.NewGuid(),  // AppInitialized
            Guid.NewGuid(),  // DrawOrder
            Guid.NewGuid(),  // Capture
        };

        /// <summary>
        /// シングルトンインスタンスです.
        /// </summary>
        private static ViewerController instance = new ViewerController();

        /// <summary>
        /// 接続状態通知インタフェースの連想コンテナです.
        /// </summary>
        private Dictionary<uint, IConnectionStateListener> connStateListeners = new Dictionary<uint, IConnectionStateListener>();

        /// <summary>The flag indicating whether connected to the viewer.</summary>
        private bool isConnected = false;

        /// <summary>
        /// コンストラクタです.
        /// </summary>
        private ViewerController()
        {
            this.State = new InnerState();
            this.State.StartFrame = 0;
            this.State.EndFrame = OptionStore.RootOptions.Viewer.LoopFrame;
            this.State.IsLoop = OptionStore.RootOptions.Viewer.LoopMode;

            MessageManager.Instance.AddStateNotifier(0, this);
            MessageManager.Instance.AddMessageNotifier(this);
        }

        /// <summary>
        /// 唯一のインスタンを取得します.
        /// </summary>
        public static ViewerController Instance
        {
            get { return instance; }
        }

        /// <summary>
        /// 接続状態を取得します.
        /// </summary>
        public bool IsConnected
        {
            get { return this.isConnected; }
            private set { this.isConnected = value; }
        }

        /// <summary>
        /// 内部状態を設定/取得します.
        /// </summary>
        private InnerState State { get; set; }

        /// <summary>
        /// 接続状態通知インタフェースを追加します.
        /// </summary>
        /// <param name="id">登録IDです</param>
        /// <param name="notifier">登録する接続状態通知インタフェースです</param>
        public void AddStateNotifier(uint id, IConnectionStateListener notifier)
        {
            // 連想コンテナに追加.
            this.connStateListeners.Add(id, notifier);
        }

        /// <summary>
        /// 接続状態通知インタフェースを削除します.
        /// </summary>
        /// <param name="id">登録IDです.</param>
        public void RemoveNotifier(uint id)
        {
            // 連想コンテナから削除.
            this.connStateListeners.Remove(id);
        }

        /// <summary>
        /// 初期設定のパケットを送信します.
        /// </summary>
        /// <param name="frameRate">フレームレート.</param>
        /// <param name="resolution">解像度.</param>
        /// <param name="clearColor">クリアカラー.</param>
        /// <param name="worldScaleRange">ワールドスケール 位置・速度・サイズ範囲.</param>
        /// <param name="worldScaleTime">ワールドスケール 時間</param>
        /// <param name="gridScale">グリッドスケール グリッドと床の表示スケール.</param>
        /// <param name="isLinearEditMode">リニア編集モードのフラグ(true:リニア編集モード, false：非リニア編集モード)</param>
        public void SendConfig(
            int frameRate,
            int resolution,
            Vector4f clearColor,
            float worldScaleRange,
            int worldScaleTime,
            float gridScale,
            bool isLinearEditMode)
        {
            // 内部変数に格納.
            this.State.FrameRate        = frameRate;
            this.State.Resolution       = resolution;
            this.State.ClearColor       = clearColor;
            this.State.WorldScaleRange  = worldScaleRange;
            this.State.WorldScaleTime   = worldScaleTime;
            this.State.GridScale        = gridScale;
            this.State.IsLinearEditMode = isLinearEditMode;

            // 転送処理.
            this.InnerSendConfing();
        }

        /// <summary>
        /// フレーム数をリセットするパケットを送信します.
        /// </summary>
        public void SendResetFrame()
        {
            if (this.IsConnected)
            {
                byte[] buffer = BinaryConversionUtility.ForProtocol.Convert((int)ViewerMessageType.ResetFrame);
                var msg = new SendBinaryMessage(
                    AssetTypes.ViewerMessage,
                    MessageGuids[(int)ViewerMessageType.ResetFrame],
                    buffer,
                    buffer.Length);

                // 送信.
                msg.Send();
            }
        }

        /// <summary>
        /// 再生パケットを送信します.
        /// </summary>
        public void SendPlay()
        {
            if (this.IsConnected)
            {
                byte[] buffer = BinaryConversionUtility.ForProtocol.Convert((int)ViewerMessageType.Play);
                var msg = new SendBinaryMessage(
                    AssetTypes.ViewerMessage,
                    MessageGuids[(int)ViewerMessageType.Play],
                    buffer,
                    buffer.Length);

                // 送信.
                msg.Send();
            }
        }

        /// <summary>
        /// 一時停止パケットを送信します.
        /// </summary>
        public void SendPause()
        {
            if (this.IsConnected)
            {
                byte[] buffer = BinaryConversionUtility.ForProtocol.Convert((int)ViewerMessageType.Pause);
                var msg = new SendBinaryMessage(
                    AssetTypes.ViewerMessage,
                    MessageGuids[(int)ViewerMessageType.Pause],
                    buffer,
                    buffer.Length);

                // 送信.
                msg.Send();
            }
        }

        /// <summary>
        /// 停止パケットを送信します.
        /// </summary>
        public void SendStop()
        {
            if (this.IsConnected)
            {
                byte[] buffer = BinaryConversionUtility.ForProtocol.Convert((int)ViewerMessageType.Stop);
                var msg = new SendBinaryMessage(
                    AssetTypes.ViewerMessage,
                    MessageGuids[(int)ViewerMessageType.Stop],
                    buffer,
                    buffer.Length);

                // 送信.
                msg.Send();
            }
        }

        /// <summary>
        /// 1フレーム前に進めるパケットを送信します.
        /// </summary>
        public void SendStepFrame()
        {
            if (this.IsConnected)
            {
                byte[] buffer = BinaryConversionUtility.ForProtocol.Convert((int)ViewerMessageType.StepFrame);
                var msg = new SendBinaryMessage(
                    AssetTypes.ViewerMessage,
                    MessageGuids[(int)ViewerMessageType.StepFrame],
                    buffer,
                    buffer.Length);

                // 送信.
                msg.Send();
            }
        }

        /// <summary>
        /// 再生を強制フェードするパケットを送信します.
        /// </summary>
        public void SendForceFade()
        {
            if (this.IsConnected)
            {
                byte[] buffer = BinaryConversionUtility.ForProtocol.Convert((int)ViewerMessageType.ForceFade);
                var msg = new SendBinaryMessage(
                    AssetTypes.ViewerMessage,
                    MessageGuids[(int)ViewerMessageType.ForceFade],
                    buffer,
                    buffer.Length);

                // 送信.
                msg.Send();
            }
        }

        /// <summary>
        /// フレーム再生範囲を設定するパケットを送信します.
        /// </summary>
        /// <param name="startFrame">再生開始フレーム.</param>
        /// <param name="endFrame">再生終了フレーム.</param>
        /// <param name="isLoop">ループ再生する場合はtrueを設定.</param>
        public void SendSetFrameRange(int startFrame, int endFrame, bool isLoop)
        {
            // 内部変数に格納.
            this.State.StartFrame = startFrame;
            this.State.EndFrame   = endFrame;
            this.State.IsLoop     = isLoop;

            // 転送処理.
            this.InnerSendSetFrameRange();
        }

        /// <summary>
        /// Send visibility message to the viewer.
        /// </summary>
        /// <param name="data">The data model to send the visibility.</param>
        /// <param name="forceInvisible">強制的に非表示にする場合はtrue,本来の設定値に従う場合はfalse.</param>
        public void SendVisibility(DisplayableDataBase data, bool forceInvisible = false)
        {
            if (this.IsConnected)
            {
                if (data == null)
                {
                    return;
                }

                var assetType = MessageUtility.GetDataModelAssetType(data);
                if (assetType == AssetTypes.Unknown)
                {
                    return;
                }

                string name = string.Empty;
                DataModelBase parent = null;
                if (data is EmitterSetData)
                {
                    parent = data;
                    name = ((EmitterSetData)data).Name;
                }
                else if (data is EmitterData)
                {
                    parent = data.FindParentOfType<EmitterSetData>();
                    name = ((EmitterData)data).Name;
                    foreach (var childEmitter in data.Children.OfType<EmitterData>())
                    {
                        this.SendVisibility(childEmitter, !data.Displayed);
                    }
                }
                else if (data is PreviewData)
                {
                    parent = data;
                    name = ((PreviewData)data).Name;
                }
                else if (data is ModelData)
                {
                    parent = data;
                    name = ((ModelData)data).Name;
                }
                else
                {
                    return;
                }

                if (parent == null)
                {
                    return;
                }

                List<byte> byteList = new List<byte>();

                // コンバート.
                byteList.AddRange(BinaryConversionUtility.ForProtocol.Convert((int)ViewerMessageType.Visibility));
                byteList.AddRange(BinaryConversionUtility.ForProtocol.Convert((uint)assetType));
                byteList.AddRange(BinaryConversionUtility.ForProtocol.Convert(parent.Guid));
                byteList.AddRange(BinaryConversionUtility.ForProtocol.Convert(ConvertString(name, 64, Encoding.ASCII)));
                byteList.AddRange(BinaryConversionUtility.ForProtocol.Convert(!forceInvisible && data.Displayed));

                byte[] buffer = byteList.ToArray();

                var msg = new SendBinaryMessage(
                    AssetTypes.ViewerMessage,
                    data.Guid,
                    buffer,
                    buffer.Length);

                // 送信.
                msg.Send();
            }
        }

        /// <summary>
        /// キャプチャを行うパケットを送信します。
        /// </summary>
        /// <param name="filePath">保存先のファイルパス</param>
        public void SendCapture(string filePath)
        {
            if (this.IsConnected == false)
            {
                return;
            }

            string dirPath = System.IO.Path.GetDirectoryName(filePath);

            if (System.IO.Directory.Exists(dirPath) == false)
            {
                System.IO.Directory.CreateDirectory(dirPath);
            }

            char[] chFilePath = (filePath + "\0").ToCharArray();

            List<byte> byteList = new List<byte>();

            // コンバート.
            byteList.AddRange(BinaryConversionUtility.ForProtocol.Convert((int)ViewerMessageType.Capture));
            byteList.AddRange(BinaryConversionUtility.ForProtocol.Convert(chFilePath.Length));
            byteList.AddRange(BinaryConversionUtility.ForProtocol.Convert(chFilePath));

            byte[] buffer = byteList.ToArray();

            var msg = new SendBinaryMessage(
                AssetTypes.ViewerMessage,
                MessageGuids[(int)ViewerMessageType.Capture],
                buffer,
                buffer.Length);

            msg.Send();
        }

        /// <summary>
        /// ビューアに描画順を送ります。
        /// </summary>
        public void InnerSendDrawOrders()
        {
            if (this.IsConnected == false)
            {
                return;
            }

            // 描画順データを作成
            byte[] orders = new byte[2 * EffectMakerProjectConfig.DrawPathMaxNum];

            for (int i = 0; i < EffectMakerProjectConfig.DrawPathMaxNum; ++i)
            {
                orders[(i * 2) + 0] = 0xff;
                orders[(i * 2) + 1] = 0xff;
            }

            List<EffectMakerProjectConfig.DrawPath> drawPaths = OptionStore.ProjectConfig.DrawPaths;

            if (drawPaths.Count > 0)
            {
                // 描画パスが指定されているとき描画順を取得
                foreach (var drawPath in OptionStore.ProjectConfig.DrawPaths)
                {
                    orders[(drawPath.Id * 2) + 0] = (byte)drawPath.GroupIndex;
                    orders[(drawPath.Id * 2) + 1] = (byte)drawPath.Order;
                }
            }
            else
            {
                // 描画パスが指定されてないときはデフォルト設定
                for (int i = 0; i < EffectMakerProjectConfig.DrawPathMaxNum; ++i)
                {
                    orders[(i * 2) + 0] = (byte)i;
                    orders[(i * 2) + 1] = 0;
                }
            }

            List<byte> byteList = new List<byte>();

            // コンバート.
            byteList.AddRange(BinaryConversionUtility.ForProtocol.Convert((int)ViewerMessageType.DrawOrder));
            byteList.AddRange(BinaryConversionUtility.ForProtocol.Convert(orders));

            byte[] buffer = byteList.ToArray();

            var msg = new SendBinaryMessage(
                AssetTypes.ViewerMessage,
                MessageGuids[(int)ViewerMessageType.DrawOrder],
                buffer,
                buffer.Length);

            // 送信.
            msg.Send();
        }

        /// <summary>
        /// ビューアオプションの値を設定します。
        /// </summary>
        /// <param name="frameRate">フレームレートを表すインデックス</param>
        /// <param name="resolution">解像度を表すインデックス</param>
        /// <param name="isLinearEditMode">リニア編集モードのON/OFF</param>
        public void SetOptionSettings(
            int frameRate,
            int resolution,
            bool isLinearEditMode)
        {
            // 内部変数に格納.
            this.State.FrameRate = frameRate;
            this.State.Resolution = resolution;
            this.State.IsLinearEditMode = isLinearEditMode;
        }

        /// <summary>
        /// パケット再転送します.
        /// </summary>
        public void Reload()
        {
            // コンフィグ情報を転送.
            this.InnerSendConfing();

            // 描画順情報を転送.
            this.InnerSendDrawOrders();

            // フレーム再生範囲情報を転送.
            this.InnerSendSetFrameRange();
        }

        /// <summary>
        /// 内部状態をクリアします.
        /// </summary>
        public void ClearState()
        {
            // リセット.
            this.State.Reset();
        }

        /// <summary>
        /// 接続状態が変更されたときの処理.
        /// </summary>
        /// <param name="prevState">前の状態</param>
        /// <param name="currState">現在の状態.</param>
        public void OnConnectionStateChanged(ConnectionStates prevState, ConnectionStates currState)
        {
        #if DISABLE_WAIT_APP
            this.IsConnected = MessageManager.Instance.IsConnected;

            this.NotifyState(prevState, currState);
        #else
            // ビューアからメッセージがくるまでは非接続にする
            if (currState == ConnectionStates.Connected && this.IsConnected == false)
            {
                currState = ConnectionStates.Disconnected;
            }

            // 接続状態に変化なし
            if (prevState == currState)
            {
                return;
            }

            // 非接続になった
            if (currState == ConnectionStates.Disconnected)
            {
                this.IsConnected = false;
            }

            this.NotifyState(prevState, currState);
        #endif
        }

        /// <summary>
        /// ビューアからのメッセージ受信時の処理.
        /// </summary>
        /// <param name="msg">メッセージ</param>
        public void OnMessageReceived(Message msg)
        {
            var reader = new PacketReader();
            MessageBase[] messages = reader.Read(msg.Buffer);

            foreach (var message in messages)
            {
                // ViewerMessage以外は処理しない
                if (message.MessageType != MessageTypes.SendBinary)
                {
                    return;
                }

                var viewerMsg = message as ViewerMessage;
                Debug.Assert(viewerMsg != null, "SendBinaryMessageの型が不正");

                if (viewerMsg.ViewerMessageType == ViewerMessageType.AppInitialized &&
                    this.IsConnected == false)
                {
                    this.IsConnected = true;
                    this.NotifyState(ConnectionStates.Disconnected, ConnectionStates.Connected);

                    Logger.Log(LogLevels.Debug, "ViewerController.OnMessageReceived : Received the initialized message.");
                }

            #if DISABLE_WAIT_APP
                // ログだけ出す
                if (viewerMsg.ViewerMessageType == ViewerMessageType.AppInitialized)
                {
                    Logger.Log(LogLevels.Debug, "ViewerController.OnMessageReceived : Received the initialized message.");
                }
            #endif
            }
        }

        /// <summary>
        /// 固定長文字列を取得します.
        /// </summary>
        /// <param name="str">可変長文字列.</param>
        /// <param name="size">固定長サイズ.</param>
        /// <param name="encoding">エンコード</param>
        /// <returns>固定長文字列を返却します.</returns>
        private static byte[] ConvertString(string str, int size, Encoding encoding)
        {
            if (size <= 0)
            {
                size = str.Length;
            }

            byte[] output = Enumerable.Repeat((byte)0, size).ToArray();

            encoding.GetBytes(str, 0, Math.Min(str.Length, size - 1), output, 0);

            return output;
        }

        /// <summary>
        /// 接続状態を通知します.
        /// </summary>
        /// <param name="prevState">以前の接続状態</param>
        /// <param name="currState">現在の接続上</param>
        private void NotifyState(ConnectionStates prevState, ConnectionStates currState)
        {
            foreach (KeyValuePair<uint, IConnectionStateListener> key in this.connStateListeners)
            {
                // 状態を通知.
                key.Value.OnConnectionStateChanged(prevState, currState);
            }
        }

        /// <summary>
        /// コンフィグ情報を転送します.
        /// </summary>
        private void InnerSendConfing()
        {
            if (this.IsConnected)
            {
                List<byte> byteList = new List<byte>();

                // コンバート.
                byteList.AddRange(BinaryConversionUtility.ForProtocol.Convert((int)ViewerMessageType.Config));
                byteList.AddRange(BinaryConversionUtility.ForProtocol.Convert(this.State.FrameRate));
                byteList.AddRange(BinaryConversionUtility.ForProtocol.Convert(this.State.Resolution));
                byteList.AddRange(BinaryConversionUtility.ForProtocol.Convert(this.State.ClearColor.X));
                byteList.AddRange(BinaryConversionUtility.ForProtocol.Convert(this.State.ClearColor.Y));
                byteList.AddRange(BinaryConversionUtility.ForProtocol.Convert(this.State.ClearColor.Z));
                byteList.AddRange(BinaryConversionUtility.ForProtocol.Convert(this.State.ClearColor.W));
                byteList.AddRange(BinaryConversionUtility.ForProtocol.Convert(this.State.WorldScaleRange));
                byteList.AddRange(BinaryConversionUtility.ForProtocol.Convert(this.State.WorldScaleTime));
                byteList.AddRange(BinaryConversionUtility.ForProtocol.Convert(this.State.GridScale));
                byteList.AddRange(BinaryConversionUtility.ForProtocol.Convert((int)(this.State.IsLinearEditMode ? 1 : 0)));

                byte[] buffer = byteList.ToArray();

                var msg = new SendBinaryMessage(
                    AssetTypes.ViewerMessage,
                    MessageGuids[(int)ViewerMessageType.Config],
                    buffer,
                    buffer.Length);

                // 送信.
                msg.Send();
            }
        }

        /// <summary>
        /// フレーム再生範囲情報を転送します.
        /// </summary>
        private void InnerSendSetFrameRange()
        {
            if (this.IsConnected)
            {
                List<byte> byteList = new List<byte>();

                // コンバート.
                byteList.AddRange(BinaryConversionUtility.ForProtocol.Convert((int)ViewerMessageType.SetRange));
                byteList.AddRange(BinaryConversionUtility.ForProtocol.Convert(this.State.StartFrame));
                byteList.AddRange(BinaryConversionUtility.ForProtocol.Convert(this.State.EndFrame));
                byteList.AddRange(BinaryConversionUtility.ForProtocol.Convert((int)(this.State.IsLoop ? 1 : 0)));

                byte[] buffer = byteList.ToArray();

                var msg = new SendBinaryMessage(
                    AssetTypes.ViewerMessage,
                    MessageGuids[(int)ViewerMessageType.SetRange],
                    buffer,
                    buffer.Length);

                // 送信.
                msg.Send();
            }
        }

        /// <summary>
        /// 内部状態を保存するクラスです.
        /// </summary>
        public class InnerState
        {
            /// <summary>
            /// コンストラクタです.
            /// </summary>
            public InnerState()
            {
                // デフォルト値を設定.
                this.Reset();
            }

            /// <summary>
            /// フレームレートです.
            /// </summary>
            public int FrameRate { get; set; }

            /// <summary>
            /// 解像度です.
            /// </summary>
            public int Resolution { get; set; }

            /// <summary>
            /// クリアカラーです.
            /// </summary>
            public Vector4f ClearColor { get; set; }

            /// <summary>
            /// ワールドスケールレンジです.
            /// </summary>
            public float WorldScaleRange { get; set; }

            /// <summary>
            /// ワールドスケール時間です.
            /// </summary>
            public int WorldScaleTime { get; set; }

            /// <summary>
            /// グリッドスケールです.
            /// </summary>
            public float GridScale { get; set; }

            /// <summary>
            /// リニア編集モードです.
            /// </summary>
            public bool IsLinearEditMode { get; set; }

            /// <summary>
            /// 開始フレームです.
            /// </summary>
            public int StartFrame { get; set; }

            /// <summary>
            /// 終了フレームです.
            /// </summary>
            public int EndFrame { get; set; }

            /// <summary>
            /// ループ有効フラグです.
            /// </summary>
            public bool IsLoop { get; set; }

            /// <summary>
            /// デフォルト値にリセットします.
            /// </summary>
            public void Reset()
            {
                this.FrameRate          = 0;
                this.Resolution         = 0;
                this.ClearColor         = new Vector4f(0.03f, 0.03f, 0.03f, 1.0f);
                this.WorldScaleRange    = 1.0f;
                this.WorldScaleTime     = 1;
                this.GridScale          = 1.0f;
                this.IsLinearEditMode   = true;
                this.StartFrame         = 0;
                this.EndFrame           = 100;
                this.IsLoop             = false;
            }
        }
    }
}
