﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using EffectMaker.BusinessLogic.ViewerMessages;
using EffectMaker.Communicator;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Primitives;

namespace EffectMaker.BusinessLogic.Protocol
{
    public class CameraController : IMessageListener
    {
        /// <summary>
        /// シングルトンインスタンスです.
        /// </summary>
        private static CameraController instance = new CameraController();

        /// <summary>
        /// UIスレッドへの同期オブジェクトです.
        /// </summary>
        private static SynchronizationContext syncContext;

        /// <summary>
        /// カメラ情報が変わったときのデリゲートです.
        /// </summary>
        public delegate void CameraInformationDelegate(object sender, Vector3f cameraPosition, Vector3f cameraTarget);

        /// <summary>
        /// カメラ情報が変わったときに実行します.
        /// </summary>
        public CameraInformationDelegate CameraInformationChanged { get; set; }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        private CameraController()
        {
            MessageManager.Instance.AddMessageNotifier(this);
        }

        /// <summary>
        /// 唯一のインスタンスを取得します.
        /// </summary>
        public static CameraController Instance
        {
            get { return instance; }
        }


        /// <summary>
        /// 初期化処理です.
        /// </summary>
        /// <param name="viewerData">ビューアデータ</param>
        /// <param name="syncContext">同期オブジェクト</param>
        public static void Initialize(SynchronizationContext syncContext)
        {
            CameraController.syncContext = syncContext;
        }

        /// <summary>
        /// ビューアからのメッセージ受信時の処理.
        /// </summary>
        /// <param name="msg">メッセージ</param>
        public void OnMessageReceived(Message msg)
        {
            var reader = new PacketReader();
            MessageBase[] messages = reader.Read(msg.Buffer);

            foreach (var message in messages)
            {
                // ViewerMessage以外は処理しない
                if (message.MessageType != MessageTypes.BinaryData)
                {
                    return;
                }

                var cameraMsg = message as CameraMessage;
                Debug.Assert(cameraMsg != null, "CameraMessageの型が不正");

                // カメラ情報の取得
                Logger.Log(LogLevels.Information, "CameraController.OnMessageReceived : Received the camera message.");
                syncContext.Send(
                    (s) =>
                    {
                        if (this.CameraInformationChanged != null)
                        {
                            CameraInformationChanged(this, cameraMsg.CameraPosition, cameraMsg.CameraTarget);
                        }
                    },
                    null);

#if DISABLE_WAIT_APP
                // ログだけ出す
                if (viewerMsg.ViewerMessageType == ViewerMessageType.AppInitialized)
                {
                    Logger.Log(LogLevels.Debug, "ViewerController.OnMessageReceived : Received the initialized message.");
                }
#endif
            }
        }
    }
}
