﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text;
using EffectMaker.BusinessLogic.IO;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Serialization;
using EffectMaker.Foundation.Utility;
using EffectMaker.Foundation.Log;
using EffectMaker.BusinessLogic.Options;

namespace EffectMaker.BusinessLogic.ProjectConfig
{
    /// <summary>
    /// テクスチャバッファの原点の種類です。
    /// </summary>
    public enum TextureOriginMode
    {
        /// <summary>
        /// 左下
        /// </summary>
        LowerLeft,

        /// <summary>
        /// 左上
        /// </summary>
        UpperLeft
    }

    /// <summary>
    /// デプスの範囲の種類です。
    /// </summary>
    public enum DepthMode
    {
        /// <summary>
        /// -1 ～ 1
        /// </summary>
        NearIsMinusW,

        /// <summary>
        /// 0 ～ 1
        /// </summary>
        NearIsZero
    }

    /// <summary>
    /// プロジェクト設定を管理します。
    /// </summary>
    [Serializable]
    public class EffectMakerProjectConfig : IXmlDocSerializable
    {
        /// <summary>描画パスの最大数</summary>
        public const int DrawPathMaxNum = 64;

        /// <summary>
        /// プロジェクト設定ファイルパスです。
        /// </summary>
        private string configFilePath = string.Empty;

        /// <summary>
        /// カスタムシェーダ定義ファイルパスです。
        /// </summary>
        private ExpandablePath customShaderPath;

        /// <summary>
        /// カスタムアクション定義ファイルパスです。
        /// </summary>
        private ExpandablePath customActionPath;

        /// <summary>
        /// エミッタ拡張パラメータ定義ファイルパスです。
        /// </summary>
        private ExpandablePath emitterExtParamsPath;

        /// <summary>
        /// ユーザーデータUIに設定するアセットのベースフォルダです。
        /// </summary>
        private ExpandablePath userAssetsBaseFolderPath;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public EffectMakerProjectConfig()
        {
            this.customShaderPath = new ExpandablePath();
            this.customActionPath = new ExpandablePath();
            this.emitterExtParamsPath = new ExpandablePath();
            this.userAssetsBaseFolderPath = new ExpandablePath();

            this.DrawPaths = new List<DrawPath>();
            this.DefaultDrawPathName = string.Empty;
            this.DrawOrdersData = new DrawOrders();
            this.EmitterSetUserDataSetting = new EmitterSetUserData();
            this.CustomFieldDataSetting = new CustomFieldData();
            this.LinearMode = true;
            this.TextureOriginMode = TextureOriginMode.LowerLeft;
            this.DepthMode = DepthMode.NearIsMinusW;
            this.AdditionalViewerArguments = string.Empty;
            this.EnableRegularGravity = false;
            this.UnitLength = 100.0f;
            this.FpsForRegularGravity = 60;
            this.DrawPathSettingErrors = new List<string>();
            this.EffectCombinerBlockDefinitionPaths = new List<string>();
            this.DocumentPath = string.Empty;
            this.AdditionalShaderIncludeDirectories = new List<string>();
            this.AdditionalShaderPreprocessorDefinitions = new List<string>();
            this.EnableShaderPreprocessOption = false;
            this.FileEventProject = new FileEventProjectConfig();
            this.IncludeEmitterSetFilePathArrayInEditorMode = false;
            this.IsEnabledPreciseGpuCounterMode = false;
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="src">コピー元オブジェクト</param>
        public EffectMakerProjectConfig(object src)
            : this()
        {
            this.Set(src);
        }

        /// <summary>
        /// プロジェクト名を取得または設定します。
        /// </summary>
        public string ProjectName { get; set; }

        /// <summary>
        /// カスタムシェーダ定義ファイルパスを取得または設定します。
        /// </summary>
        public string CustomShaderPath
        {
            get { return this.customShaderPath.ExpandedPath; }
            set { this.customShaderPath.Path = value; }
        }

        /// <summary>
        /// カスタムアクション定義パスを取得または設定します。
        /// </summary>
        public string CustomActionPath
        {
            get
            {
                if (string.IsNullOrEmpty(this.customActionPath.ExpandedPath))
                {
                    return Path.Combine(IOConstants.ExecutableFolderPath, "Addins", "CustomActionSettings.xml");
                }

                return this.customActionPath.ExpandedPath;
            }

            set
            {
                this.customActionPath.Path = value;
            }
        }

        /// <summary>
        /// エミッタ拡張パラメータ定義パスを取得または設定します。
        /// </summary>
        public string EmitterExtParamsPath
        {
            get
            {
                if (string.IsNullOrEmpty(this.emitterExtParamsPath.ExpandedPath))
                {
                    return Path.Combine(IOConstants.ExecutableFolderPath, "Addins", "EmitterExtParams.xml");
                }

                return this.emitterExtParamsPath.ExpandedPath;
            }

            set
            {
                this.emitterExtParamsPath.Path = value;
            }
        }

        /// <summary>
        /// ユーザーデータUIに設定するアセットの基準フォルダを取得または設定します。
        /// </summary>
        public string UserAssetsBaseFolderPath
        {
            get { return this.userAssetsBaseFolderPath.ExpandedPath; }
            set { this.userAssetsBaseFolderPath.Path = value; }
        }

        /// <summary>
        /// リニア編集モードを取得または設定します。
        /// </summary>
        public bool LinearMode { get; set; }

        /// <summary>
        /// テクスチャの原点を取得または設定します。
        /// </summary>
        public TextureOriginMode TextureOriginMode { get; set; }

        /// <summary>
        /// デプスの範囲を取得または設定します。
        /// </summary>
        public DepthMode DepthMode { get; set; }

        /// <summary>
        /// [Experimental]厳密なGPUカウンタモードを使用するかどうかを取得または設定します。
        /// </summary>
        public bool IsEnabledPreciseGpuCounterMode { get; set; }

        /// <summary>
        /// 追加のビューア起動引数を取得または設定します。
        /// </summary>
        public string AdditionalViewerArguments { get; set; }

        /// <summary>
        /// 標準重力有効
        /// </summary>
        public bool EnableRegularGravity { get; set; }

        /// <summary>
        /// 標準重力計算用の単位長
        /// </summary>
        public float UnitLength { get; set; }

        /// <summary>
        /// 標準重力計算用の想定FPS
        /// </summary>
        public int FpsForRegularGravity { get; set; }

        /// <summary>
        /// 描画パス設定を取得または設定します。
        /// </summary>
        public List<DrawPath> DrawPaths { get; set; }

        /// <summary>
        /// デフォルトにする描画パスの名前を取得または設定します。
        /// </summary>
        public string DefaultDrawPathName { get; set; }

        /// <summary>
        /// デフォルトの描画パスを取得します.
        /// </summary>
        public DrawPath DefaultDrawPath { get; set; }

        /// <summary>
        /// 描画順を取得または設定します。
        /// </summary>
        public DrawOrders DrawOrdersData { get; set; }

        /// <summary>
        /// エミッタセットユーザーデータ設定を取得または設定します。
        /// </summary>
        public EmitterSetUserData EmitterSetUserDataSetting { get; set; }

        /// <summary>
        /// カスタムフィールドの設定を取得または設定します。
        /// </summary>
        public CustomFieldData CustomFieldDataSetting { get; set; }

        /// <summary>
        /// 描画パスの設定エラー。
        /// </summary>
        public List<string> DrawPathSettingErrors { get; set; }

        /// <summary>
        /// Get or set the effect combiner shader block definition paths.
        /// </summary>
        public List<string> EffectCombinerBlockDefinitionPaths { get; set; }

        /// <summary>
        /// ドキュメントの起動パスを取得または設定します。
        /// </summary>
        public string DocumentPath { get; set; }

        /// <summary>
        /// 追加のシェーダインクルードディレクトリリストを取得または設定します。
        /// </summary>
        public List<string> AdditionalShaderIncludeDirectories { get; set; }

        /// <summary>
        /// 追加のシェーダプリプロセス定義リストを取得または設定します。
        /// </summary>
        public List<string> AdditionalShaderPreprocessorDefinitions { get; set; }

        /// <summary>
        /// シェーダーコンバータにpreprocessオプションを付加するか否かを取得または設定します。
        /// </summary>
        public bool EnableShaderPreprocessOption { get; set; }

        /// <summary>
        /// Get or set the flag indicating whether to enable EffectCombiner features.
        /// </summary>
        public bool IsEftCombinerEditorEnabled
        {
            get
            {
                return this.EffectCombinerBlockDefinitionPaths != null && this.EffectCombinerBlockDefinitionPaths.Any();
            }
        }

        /// <summary>
        /// プロジェクト設定ファイルパスを取得または設定します。
        /// </summary>
        public string ConfigFilePath
        {
            get
            {
                return this.configFilePath;
            }

            set
            {
                this.configFilePath = value;

                string baseFolderPath = string.IsNullOrEmpty(value) == false ? Path.GetDirectoryName(value) : null;

                this.customShaderPath.BasePath         = baseFolderPath;
                this.customActionPath.BasePath         = baseFolderPath;
                this.emitterExtParamsPath.BasePath     = baseFolderPath;
                this.userAssetsBaseFolderPath.BasePath = baseFolderPath;

                this.FileEventProject.BasePath         = baseFolderPath;
            }
        }

        /// <summary>
        /// Gets or sets the file paths options.
        /// </summary>
        public FileEventProjectConfig FileEventProject { get; set; }

        /// <summary>
        /// エディタモード時に、エミッタセットのファイルパスをバイナリに含めるか
        /// </summary>
        public bool IncludeEmitterSetFilePathArrayInEditorMode { get; set; }

        /// <summary>
        /// プロジェクト設定をファイルから読み込みます。
        /// </summary>
        /// <param name="fullPath">設定ファイルのフルパス</param>
        /// <returns>プロジェクト設定を返します。</returns>
        public static EffectMakerProjectConfig Load(string fullPath)
        {
            Debug.Assert(Path.IsPathRooted(fullPath), "パスの指定が不正。");

            // ファイルの存在をチェック
            if (File.Exists(fullPath) == false)
            {
                Logger.Log(LogLevels.Error, "Can't find file : " + fullPath);
                return null;
            }

            EffectMakerProjectConfig prjConfig = null;

            // プロジェクト設定を読み込む
            try
            {
                using (var stream = new FileStream(fullPath, FileMode.Open, FileAccess.Read))
                {
                    byte[] byteData = CharsetUtility.ConvertToByte(stream, Encoding.UTF8);
                    using (var utfStream = new MemoryStream(byteData))
                    {
                        prjConfig = SerializationHelper.LoadXmlDocSerializable<EffectMakerProjectConfig>(utfStream);
                    }
                }
            }
            catch (Exception e)
            {
                Logger.Log(LogLevels.Error, "Open file error : " + fullPath + " , " + e.Message);
            }

            // プロジェクト設定を読み込めたとき、初期設定を行う
            if (prjConfig != null)
            {
                prjConfig.ConfigFilePath = fullPath;
                prjConfig.ConstructDrawPathSettings();
            }

            return prjConfig;
        }

        /// <summary>
        /// 設定初期化
        /// </summary>
        public void Initialize()
        {
            this.ProjectName = string.Empty;
            this.CustomShaderPath = string.Empty;
            this.CustomActionPath = string.Empty;
            this.LinearMode = true;
            this.TextureOriginMode = TextureOriginMode.LowerLeft;
            this.DepthMode = DepthMode.NearIsMinusW;
            this.AdditionalViewerArguments = string.Empty;
            this.EnableRegularGravity = false;
            this.UnitLength = 100.0f;
            this.FpsForRegularGravity = 60;
            this.DrawPaths = new List<DrawPath>();
            this.DefaultDrawPathName = string.Empty;
            this.EmitterSetUserDataSetting = new EmitterSetUserData();
            this.CustomFieldDataSetting = new CustomFieldData();
            this.DocumentPath = string.Empty;
            this.AdditionalShaderIncludeDirectories = new List<string>();
            this.AdditionalShaderPreprocessorDefinitions = new List<string>();
            this.EnableShaderPreprocessOption = false;
            this.IncludeEmitterSetFilePathArrayInEditorMode = false;
            this.IsEnabledPreciseGpuCounterMode = false;
        }

        /// <summary>
        /// 描画パス設定を構築します。
        /// </summary>
        public void ConstructDrawPathSettings()
        {
            // 構築した描画パス設定をクリア
            this.DefaultDrawPath = null;
            this.DrawPathSettingErrors.Clear();

            // 描画パスに重複がないかチェック
            for (int i = 0; i < this.DrawPaths.Count; ++i)
            {
                if (this.DrawPaths[i].Id < 0 || this.DrawPaths[i].Id >= DrawPathMaxNum)
                {
                    // 描画パスのIDが不正
                    this.DrawPathSettingErrors.Add(Properties.Resources.WarningDrawPathInvalidId);
                    continue;
                }

                if (string.IsNullOrEmpty(this.DrawPaths[i].Text))
                {
                    // 描画パスの名前が不正
                    this.DrawPathSettingErrors.Add(Properties.Resources.WarningDrawPathInvalidPathName);
                    continue;
                }

                for (int j = i + 1; j < this.DrawPaths.Count; ++j)
                {
                    if (this.DrawPaths[j].Id == this.DrawPaths[i].Id)
                    {
                        // 描画パスのIDが重複
                        this.DrawPathSettingErrors.Add(Properties.Resources.WarningDrawPathDuplicateId);
                    }

                    if (this.DrawPaths[j].Text == this.DrawPaths[i].Text)
                    {
                        // 描画パスの名前が重複
                        this.DrawPathSettingErrors.Add(Properties.Resources.WarningDrawPathDuplicateName);
                    }
                }
            }

            // 描画パスが未指定のときは以降の処理を省略
            if (this.DrawPaths.Count == 0)
            {
                return;
            }

            // デフォルト描画パスが設定されているかチェック
            if (string.IsNullOrEmpty(this.DefaultDrawPathName))
            {
                // デフォルト描画パスが未指定
                this.DrawPaths.Clear();
                this.DrawOrdersData.Groups.Clear();
                this.DrawPathSettingErrors.Add(Properties.Resources.WarningDrawPathNoDefaultName);
                return;
            }

            // デフォルト描画パスと一致する名前の描画パスを取得
            DrawPath defaultDrawPath = this.DrawPaths.FirstOrDefault(
                item => item.Text == this.DefaultDrawPathName);
            if (defaultDrawPath != null)
            {
                this.DefaultDrawPath = defaultDrawPath;
            }
            else
            {
                // デフォルト描画パスの指定が不正
                this.DrawPaths.Clear();
                this.DrawOrdersData.Groups.Clear();
                this.DrawPathSettingErrors.Add(Properties.Resources.WarningDrawPathInvalidDefaultName);
                return;
            }

            string[] groupNames = new string[]
            {
                "Default",
                "ReductionBuffer",
                "PreColorCopy",
                "Opaque"
            };

            // 描画順を設定
            if (this.DrawOrdersData.Groups.Count != 0)
            {
                foreach (DrawOrderGroup group in this.DrawOrdersData.Groups)
                {
                    int groupIndex = Array.IndexOf(groupNames, group.Type);
                    if (groupIndex == -1)
                    {
                        // グループタイプの指定が不正
                        this.DrawPathSettingErrors.Add(Properties.Resources.WarningDrawOrderInvalidGroupType);
                        continue;
                    }

                    for (int order = 0; order < group.Items.Count; ++order)
                    {
                        DrawPath drawPath = this.DrawPaths.FirstOrDefault(
                            path => path.Text == group.Items[order].Text);
                        if (drawPath == null)
                        {
                            // パス名の指定が不正
                            this.DrawPathSettingErrors.Add(Properties.Resources.WarningDrawOrderInvalidPathName);
                            continue;
                        }

                        if (drawPath.GroupIndex != -1)
                        {
                            // 重複した描画順の指定
                            this.DrawPathSettingErrors.Add(Properties.Resources.WarningDrawOrderDuplicatePathName);
                            continue;
                        }

                        // 描画順を設定
                        drawPath.GroupIndex = groupIndex;
                        drawPath.Order = order;
                    }
                }
            }
            else
            {
                // 描画順が未設定のときはデフォルト値を使用
                foreach (DrawPath drawPath in this.DrawPaths)
                {
                    drawPath.GroupIndex = drawPath.Id;
                    drawPath.Order = 0;
                }
            }
        }

        /// <summary>
        /// Deserializes from the given XML node.
        /// </summary>
        /// <param name="context">The data context needed for the deserialization.</param>
        /// <returns>True on success.</returns>
        public bool ReadXml(XmlDocSerializationContext context)
        {
            this.ProjectName = this.ReadElement(context, "ProjectName", this.ProjectName);

            // 過去にタイプミスしていたバージョンの値を読み込む
            this.FpsForRegularGravity      = this.ReadElement(context, "FpsForRegualrGravity", this.FpsForRegularGravity);

            // プライベートフィールドの方に読み込む
            this.customShaderPath.Path         = this.ReadElement(context, "CustomShaderPath", this.CustomShaderPath);
            this.customActionPath.Path         = this.ReadElement(context, "CustomActionPath", this.CustomActionPath);
            this.emitterExtParamsPath.Path     = this.ReadElement(context, "EmitterExtParamsPath", this.EmitterExtParamsPath);
            this.userAssetsBaseFolderPath.Path = this.ReadElement(context, "UserAssetsBaseFolderPath", this.UserAssetsBaseFolderPath);

            this.LinearMode                = this.ReadElement(context, nameof(this.LinearMode), this.LinearMode);
            this.TextureOriginMode         = this.ReadElement(context, nameof(this.TextureOriginMode), this.TextureOriginMode);
            this.DepthMode                 = this.ReadElement(context, nameof(this.DepthMode), this.DepthMode);
            this.AdditionalViewerArguments = this.ReadElement(context, nameof(this.AdditionalViewerArguments), this.AdditionalViewerArguments);
            this.EnableRegularGravity      = this.ReadElement(context, nameof(this.EnableRegularGravity), this.EnableRegularGravity);
            this.UnitLength                = this.ReadElement(context, nameof(this.UnitLength), this.UnitLength);
            this.FpsForRegularGravity      = this.ReadElement(context, nameof(this.FpsForRegularGravity), this.FpsForRegularGravity);
            this.DrawPaths                 = this.ReadListElement(context, nameof(this.DrawPaths), this.DrawPaths);
            this.DefaultDrawPathName       = this.ReadElement(context, nameof(this.DefaultDrawPath), this.DefaultDrawPathName);
            this.DrawOrdersData            = this.ReadElement(context, "DrawOrders", this.DrawOrdersData);
            this.EmitterSetUserDataSetting = this.ReadElement(context, "EmitterSetUserData", this.EmitterSetUserDataSetting);
            this.CustomFieldDataSetting    = this.ReadElement(context, "CustomFieldData", this.CustomFieldDataSetting);
            this.EffectCombinerBlockDefinitionPaths = this.ReadListElement(context, nameof(this.EffectCombinerBlockDefinitionPaths), this.EffectCombinerBlockDefinitionPaths);
            this.DocumentPath                       = this.ReadElement(context, nameof(this.DocumentPath), this.DocumentPath);
            this.AdditionalShaderIncludeDirectories = this.ReadListElement(context, nameof(this.AdditionalShaderIncludeDirectories), this.AdditionalShaderIncludeDirectories);
            this.AdditionalShaderPreprocessorDefinitions = this.ReadListElement(context, nameof(this.AdditionalShaderPreprocessorDefinitions), this.AdditionalShaderPreprocessorDefinitions);
            this.EnableShaderPreprocessOption            = this.ReadElement(context, nameof(this.EnableShaderPreprocessOption), this.EnableShaderPreprocessOption);

            this.FileEventProject = this.ReadElement(context, "FilePaths", this.FileEventProject);

            this.IncludeEmitterSetFilePathArrayInEditorMode = this.ReadElement(context, nameof(this.IncludeEmitterSetFilePathArrayInEditorMode), this.IncludeEmitterSetFilePathArrayInEditorMode);
            this.IsEnabledPreciseGpuCounterMode = this.ReadElement(context, nameof(this.IsEnabledPreciseGpuCounterMode), this.IsEnabledPreciseGpuCounterMode);

            return true;
        }

        /// <summary>
        /// Serializes this object to a XML node.
        /// </summary>
        /// <param name="context">The data context needed for the serialization.</param>
        /// <returns>True on success.</returns>
        public bool WriteXml(XmlDocSerializationContext context)
        {
            this.WriteElement(context, "ProjectName", this.ProjectName);

            // プライベートフィールドの値を書き込む
            this.WriteElement(context, nameof(this.CustomShaderPath), this.customShaderPath.Path);
            this.WriteElement(context, nameof(this.CustomActionPath), this.customActionPath.Path);
            this.WriteElement(context, nameof(this.EmitterExtParamsPath), this.emitterExtParamsPath.Path);
            this.WriteElement(context, nameof(this.UserAssetsBaseFolderPath), this.userAssetsBaseFolderPath.Path);

            this.WriteElement(context, nameof(this.LinearMode), this.LinearMode);
            this.WriteElement(context, nameof(this.TextureOriginMode), this.TextureOriginMode);
            this.WriteElement(context, nameof(this.DepthMode), this.DepthMode);
            this.WriteElement(context, nameof(this.AdditionalViewerArguments), this.AdditionalViewerArguments);
            this.WriteElement(context, nameof(this.EnableRegularGravity), this.EnableRegularGravity);
            this.WriteElement(context, nameof(this.UnitLength), this.UnitLength);
            this.WriteElement(context, nameof(this.FpsForRegularGravity), this.FpsForRegularGravity);
            this.WriteEnumerableElement(context, nameof(this.DrawPaths), this.DrawPaths, "Path");
            this.WriteElement(context, "DefaultDrawPath", this.DefaultDrawPathName);
            this.WriteElement(context, "DrawOrders", this.DrawOrdersData);
            this.WriteElement(context, "EmitterSetUserData", this.EmitterSetUserDataSetting);
            this.WriteElement(context, "CustomFieldData", this.CustomFieldDataSetting);

            // コンバイナシェーダ定義フォルダは値が存在する場合のみ書き出し対象とする
            if (this.IsEftCombinerEditorEnabled)
            {
                this.WriteEnumerableElement(context, nameof(this.EffectCombinerBlockDefinitionPaths), this.EffectCombinerBlockDefinitionPaths, "Path");
            }

            this.WriteElement(context, nameof(this.DocumentPath), this.DocumentPath);
            this.WriteEnumerableElement(context, nameof(this.AdditionalShaderIncludeDirectories), this.AdditionalShaderIncludeDirectories, "Path");
            this.WriteEnumerableElement(context, nameof(this.AdditionalShaderPreprocessorDefinitions), this.AdditionalShaderPreprocessorDefinitions, "Definition");
            this.WriteElement(context, nameof(this.EnableShaderPreprocessOption), this.EnableShaderPreprocessOption);

            this.WriteElement(context, "FilePaths", this.FileEventProject);

            this.WriteElement(context, nameof(this.IncludeEmitterSetFilePathArrayInEditorMode), this.IncludeEmitterSetFilePathArrayInEditorMode);

            // Experimental: true の場合のみ書き出し対象
            if (this.IsEnabledPreciseGpuCounterMode)
            {
                this.WriteElement(context, nameof(this.IsEnabledPreciseGpuCounterMode), this.IsEnabledPreciseGpuCounterMode);
            }

            return true;
        }

        /// <summary>
        /// Set data from the source options.
        /// </summary>
        /// <param name="src">The source options.</param>
        /// <returns>True on success.</returns>
        public bool Set(object src)
        {
            var srcConfig = (EffectMakerProjectConfig)src;

            this.ProjectName = srcConfig.ProjectName;

            this.ConfigFilePath = srcConfig.ConfigFilePath;

            // ファイルパスを先に更新してベースパスが設定されてからファイルパスをコピーする
            this.customShaderPath.Set(srcConfig.customShaderPath);
            this.customActionPath.Set(srcConfig.customActionPath);
            this.emitterExtParamsPath.Set(srcConfig.emitterExtParamsPath);
            this.userAssetsBaseFolderPath.Set(srcConfig.userAssetsBaseFolderPath);

            this.LinearMode = srcConfig.LinearMode;
            this.TextureOriginMode = srcConfig.TextureOriginMode;
            this.DepthMode = srcConfig.DepthMode;
            this.AdditionalViewerArguments = srcConfig.AdditionalViewerArguments;
            this.EmitterSetUserDataSetting.Set(srcConfig.EmitterSetUserDataSetting);
            this.CustomFieldDataSetting.Set(srcConfig.CustomFieldDataSetting);

            this.EnableRegularGravity = srcConfig.EnableRegularGravity;
            this.UnitLength = srcConfig.UnitLength;
            this.FpsForRegularGravity = srcConfig.FpsForRegularGravity;

            this.DrawPaths.Clear();
            foreach (DrawPath path in srcConfig.DrawPaths)
            {
                this.DrawPaths.Add((DrawPath)path.Clone());
            }

            this.DefaultDrawPathName = srcConfig.DefaultDrawPathName;

            this.ConstructDrawPathSettings();

            this.DrawPathSettingErrors = new List<string>(srcConfig.DrawPathSettingErrors);

            // エフェクト定義フォルダをコピー
            this.EffectCombinerBlockDefinitionPaths.Clear();
            this.EffectCombinerBlockDefinitionPaths.AddRange(srcConfig.EffectCombinerBlockDefinitionPaths);

            this.DocumentPath = srcConfig.DocumentPath;

            this.AdditionalShaderIncludeDirectories.Clear();
            this.AdditionalShaderIncludeDirectories.AddRange(srcConfig.AdditionalShaderIncludeDirectories);
            this.AdditionalShaderPreprocessorDefinitions.Clear();
            this.AdditionalShaderPreprocessorDefinitions.AddRange(srcConfig.AdditionalShaderPreprocessorDefinitions);

            this.EnableShaderPreprocessOption = srcConfig.EnableShaderPreprocessOption;

            this.FileEventProject.Set(srcConfig.FileEventProject);

            this.IncludeEmitterSetFilePathArrayInEditorMode = srcConfig.IncludeEmitterSetFilePathArrayInEditorMode;

            // Experimental
            this.IsEnabledPreciseGpuCounterMode = srcConfig.IsEnabledPreciseGpuCounterMode;

            return true;
        }

        /// <summary>
        /// Clone this options and return a new instance.
        /// </summary>
        /// <returns>The created instance.</returns>
        public virtual object Clone()
        {
            var newInstance = new EffectMakerProjectConfig(this);

            return newInstance;
        }

        #region DrawPath

        /// <summary>
        /// 描画パス設定です。
        /// </summary>
        public class DrawPath : IXmlDocSerializable, ICloneable, ISettable
        {
            /// <summary>
            /// コンストラクタです。
            /// </summary>
            public DrawPath()
            {
                this.Id = 0;
                this.Text = string.Empty;
                this.ShaderCompileDef1 = string.Empty;
                this.ShaderCompileDef2 = string.Empty;

                this.GroupIndex = -1;
                this.Order = -1;
            }

            /// <summary>
            /// IDを取得または設定します。
            /// </summary>
            public int Id { get; set; }

            /// <summary>
            /// 名前を取得または設定します。
            /// </summary>
            public string Text { get; set; }

            /// <summary>
            /// シェーダー定義文字列1を取得または設定します。
            /// </summary>
            public string ShaderCompileDef1 { get; set; }

            /// <summary>
            /// シェーダー定義文字列2を取得または設定します。
            /// </summary>
            public string ShaderCompileDef2 { get; set; }

            /// <summary>
            /// 描画順のグループインデックスを取得または設定します。
            /// このプロパティは内部処理用のためシリアライズ/デシリアライズは行いません。
            /// </summary>
            public int GroupIndex { get; set; }

            /// <summary>
            /// 描画順のグループ内番号を取得または設定します。
            /// このプロパティは内部処理用のためシリアライズ/デシリアライズは行いません。
            /// </summary>
            public int Order { get; set; }

            /// <summary>
            /// データをセットします。
            /// </summary>
            /// <param name="src">コピー元データ</param>
            /// <returns>成功したときtrueを返します。</returns>
            public bool Set(object src)
            {
                DrawPath srcData = (DrawPath)src;

                this.Id = srcData.Id;
                this.Text = srcData.Text;
                this.ShaderCompileDef1 = srcData.ShaderCompileDef1;
                this.ShaderCompileDef2 = srcData.ShaderCompileDef2;
                this.GroupIndex = srcData.GroupIndex;
                this.Order = srcData.Order;

                return true;
            }

            /// <summary>
            /// オブジェクトのクローンを取得します。
            /// </summary>
            /// <returns>オブジェクトのクローンを返します。</returns>
            public virtual object Clone()
            {
                DrawPath clone = new DrawPath();
                clone.Set(this);

                return clone;
            }

            /// <summary>
            /// Deserializes from the given XML node.
            /// </summary>
            /// <param name="context">The data context needed for the deserialization.</param>
            /// <returns>True on success.</returns>
            public bool ReadXml(XmlDocSerializationContext context)
            {
                this.Id = this.ReadAttribute(context, "Id", this.Id);
                this.Text = this.ReadAttribute(context, "Text", this.Text);
                this.ShaderCompileDef1 = this.ReadAttribute(context, "ShaderCompileDef1", this.ShaderCompileDef1);
                this.ShaderCompileDef2 = this.ReadAttribute(context, "ShaderCompileDef2", this.ShaderCompileDef2);

                return true;
            }

            /// <summary>
            /// Serializes this object to a XML node.
            /// </summary>
            /// <param name="context">The data context needed for the serialization.</param>
            /// <returns>True on success.</returns>
            public bool WriteXml(XmlDocSerializationContext context)
            {
                this.WriteAttribute(context, "Id", this.Id);
                this.WriteAttribute(context, "Text", this.Text);

                if (string.IsNullOrEmpty(this.ShaderCompileDef1) == false)
                {
                    this.WriteAttribute(context, "ShaderCompileDef1", this.ShaderCompileDef1);
                }

                if (string.IsNullOrEmpty(this.ShaderCompileDef2) == false)
                {
                    this.WriteAttribute(context, "ShaderCompileDef2", this.ShaderCompileDef2);
                }

                return true;
            }
        }

        #endregion

        #region DrawOrders

        /// <summary>
        /// 描画順定義です。
        /// </summary>
        public class DrawOrders : IXmlDocSerializable, ICloneable, ISettable
        {
            /// <summary>
            /// コンストラクタです。
            /// </summary>
            public DrawOrders()
            {
                this.Groups = new List<DrawOrderGroup>();
            }

            /// <summary>
            /// グループリストを取得します。
            /// </summary>
            public List<DrawOrderGroup> Groups { get; private set; }

            /// <summary>
            /// データをセットします。
            /// </summary>
            /// <param name="src">コピー元データ</param>
            /// <returns>成功したときtrueを返します。</returns>
            public bool Set(object src)
            {
                DrawOrders srcData = (DrawOrders)src;

                this.Groups.Clear();
                foreach (DrawOrderGroup group in srcData.Groups)
                {
                    this.Groups.Add((DrawOrderGroup)group.Clone());
                }

                return true;
            }

            /// <summary>
            /// オブジェクトのクローンを取得します。
            /// </summary>
            /// <returns>オブジェクトのクローンを返します。</returns>
            public virtual object Clone()
            {
                DrawOrders clone = new DrawOrders();
                clone.Set(this);

                return clone;
            }

            /// <summary>
            /// オブジェクトをXMLからデシリアライズします。
            /// </summary>
            /// <param name="context">シリアライズコンテキスト</param>
            /// <returns>処理が成功したときtrueを返します。</returns>
            public bool ReadXml(XmlDocSerializationContext context)
            {
                this.Groups = this.ReadElementsByTagName<DrawOrderGroup>(context, "DrawGroup").ToList();

                return true;
            }

            /// <summary>
            /// オブジェクトをXMLにシリアライズします。
            /// </summary>
            /// <param name="context">シリアライズコンテキスト</param>
            /// <returns>処理が成功したときtrueを返します。</returns>
            public bool WriteXml(XmlDocSerializationContext context)
            {
                foreach (DrawOrderGroup group in this.Groups)
                {
                    this.WriteElement(context, "DrawGroup", group);
                }

                return true;
            }
        }

        /// <summary>
        /// 描画順定義グループです。
        /// </summary>
        public class DrawOrderGroup : IXmlDocSerializable, ICloneable, ISettable
        {
            /// <summary>
            /// コンストラクタです。
            /// </summary>
            public DrawOrderGroup()
            {
                this.Type = string.Empty;
                this.Items = new List<DrawOrderItem>();
            }

            /// <summary>
            /// グループタイプを取得します。
            /// </summary>
            public string Type { get; private set; }

            /// <summary>
            /// アイテムリストを取得します。
            /// </summary>
            public List<DrawOrderItem> Items { get; private set; }

            /// <summary>
            /// データをセットします。
            /// </summary>
            /// <param name="src">コピー元データ</param>
            /// <returns>成功したときtrueを返します。</returns>
            public bool Set(object src)
            {
                DrawOrderGroup srcData = (DrawOrderGroup)src;

                this.Type = srcData.Type;

                this.Items.Clear();
                foreach (DrawOrderItem item in srcData.Items)
                {
                    this.Items.Add((DrawOrderItem)item.Clone());
                }

                return true;
            }

            /// <summary>
            /// オブジェクトのクローンを取得します。
            /// </summary>
            /// <returns>オブジェクトのクローンを返します。</returns>
            public virtual object Clone()
            {
                DrawOrderGroup clone = new DrawOrderGroup();
                clone.Set(this);

                return clone;
            }

            /// <summary>
            /// オブジェクトをXMLからデシリアライズします。
            /// </summary>
            /// <param name="context">シリアライズコンテキスト</param>
            /// <returns>処理が成功したときtrueを返します。</returns>
            public bool ReadXml(XmlDocSerializationContext context)
            {
                this.Type = this.ReadAttribute(context, "Type", this.Type);
                this.Items = this.ReadElementsByTagName<DrawOrderItem>(context, "DrawItem").ToList();

                return true;
            }

            /// <summary>
            /// オブジェクトをXMLにシリアライズします。
            /// </summary>
            /// <param name="context">シリアライズコンテキスト</param>
            /// <returns>処理が成功したときtrueを返します。</returns>
            public bool WriteXml(XmlDocSerializationContext context)
            {
                this.WriteAttribute(context, "Type", this.Type);

                foreach (DrawOrderItem item in this.Items)
                {
                    this.WriteElement(context, "DrawItem", item);
                }

                return true;
            }
        }

        /// <summary>
        /// 描画順定義アイテムです。
        /// </summary>
        public class DrawOrderItem : IXmlDocSerializable, ICloneable, ISettable
        {
            /// <summary>
            /// コンストラクタです。
            /// </summary>
            public DrawOrderItem()
            {
                this.Text = string.Empty;
            }

            /// <summary>
            /// 名前を取得します。
            /// </summary>
            public string Text { get; private set; }

            /// <summary>
            /// データをセットします。
            /// </summary>
            /// <param name="src">コピー元データ</param>
            /// <returns>成功したときtrueを返します。</returns>
            public bool Set(object src)
            {
                DrawOrderItem srcData = (DrawOrderItem)src;

                this.Text = srcData.Text;

                return true;
            }

            /// <summary>
            /// オブジェクトのクローンを取得します。
            /// </summary>
            /// <returns>オブジェクトのクローンを返します。</returns>
            public virtual object Clone()
            {
                DrawOrderItem clone = new DrawOrderItem();
                clone.Set(this);

                return clone;
            }

            /// <summary>
            /// オブジェクトをXMLからデシリアライズします。
            /// /// </summary>
            /// <param name="context">シリアライズコンテキスト</param>
            /// <returns>処理が成功したときtrueを返します。</returns>
            public bool ReadXml(XmlDocSerializationContext context)
            {
                this.Text = this.ReadAttribute(context, "Text", this.Text);

                return true;
            }

            /// <summary>
            /// オブジェクトをXMLにシリアライズします。
            /// </summary>
            /// <param name="context">シリアライズコンテキスト</param>
            /// <returns>処理が成功したときtrueを返します。</returns>
            public bool WriteXml(XmlDocSerializationContext context)
            {
                this.WriteAttribute(context, "Text", this.Text);

                return true;
            }
        }

        #endregion

        #region EmitterSetUserData

        /// <summary>
        /// エミッタセットユーザーデータ設定です。
        /// </summary>
        public class EmitterSetUserData : IXmlDocSerializable, ICloneable, ISettable
        {
            /// <summary>
            /// コンストラクタです.
            /// </summary>
            public EmitterSetUserData()
            {
                this.BitLabel = new List<string>();
                this.IntNumberLabel = new List<string>();
            }

            /// <summary>
            /// ビットのラベルを取得または設定します.
            /// </summary>
            public List<string> BitLabel { get; set; }

            /// <summary>
            /// 数値のラベルを取得または設定します.
            /// </summary>
            public List<string> IntNumberLabel { get; set; }

            /// <summary>
            /// データをセットします。
            /// </summary>
            /// <param name="src">コピー元データ</param>
            /// <returns>成功したときtrueを返します。</returns>
            public bool Set(object src)
            {
                var srcConfig = (EmitterSetUserData)src;

                this.BitLabel = new List<string>(srcConfig.BitLabel);
                this.IntNumberLabel = new List<string>(srcConfig.IntNumberLabel);

                return true;
            }

            /// <summary>
            /// オブジェクトのクローンを取得します。
            /// </summary>
            /// <returns>オブジェクトのクローンを返します。</returns>
            public virtual object Clone()
            {
                var clone = new EmitterSetUserData();
                clone.Set(this);

                return clone;
            }

            /// <summary>
            /// オブジェクトをXMLからデシリアライズします.
            /// </summary>
            /// <param name="context">シリアライズコンテキスト</param>
            /// <returns>処理が成功したときtrueを返します.</returns>
            public bool ReadXml(XmlDocSerializationContext context)
            {
                this.BitLabel = this.ReadListElement(context, "BitLabel", this.BitLabel);
                this.IntNumberLabel = this.ReadListElement(context, "IntNumberLabel", this.IntNumberLabel);

                return true;
            }

            /// <summary>
            /// オブジェクトをXMLにシリアライズします.
            /// </summary>
            /// <param name="context">シリアライズコンテキスト</param>
            /// <returns>処理が成功したときtrueを返します.</returns>
            public bool WriteXml(XmlDocSerializationContext context)
            {
                this.WriteEnumerableElement(context, "BitLabel", this.BitLabel);
                this.WriteEnumerableElement(context, "IntNumberLabel", this.IntNumberLabel);

                return true;
            }
        }

        #endregion

        #region CustomFieldData

        /// <summary>
        /// カスタムフィールドの設定です。
        /// </summary>
        public class CustomFieldData : IXmlDocSerializable, ICloneable, ISettable
        {
            /// <summary>
            /// コンストラクタです.
            /// </summary>
            public CustomFieldData()
            {
                this.BitLabel = new List<string>();
                this.FloatLabel = new List<string>();
            }

            /// <summary>
            /// ビットのラベルを取得または設定します.
            /// </summary>
            public List<string> BitLabel { get; set; }

            /// <summary>
            /// 数値のラベルを取得または設定します.
            /// </summary>
            public List<string> FloatLabel { get; set; }

            /// <summary>
            /// データをセットします。
            /// </summary>
            /// <param name="src">コピー元データ</param>
            /// <returns>成功したときtrueを返します。</returns>
            public bool Set(object src)
            {
                var srcConfig = (CustomFieldData)src;

                this.BitLabel = new List<string>(srcConfig.BitLabel);
                this.FloatLabel = new List<string>(srcConfig.FloatLabel);

                return true;
            }

            /// <summary>
            /// オブジェクトのクローンを取得します。
            /// </summary>
            /// <returns>オブジェクトのクローンを返します。</returns>
            public virtual object Clone()
            {
                var clone = new CustomFieldData();
                clone.Set(this);

                return clone;
            }

            /// <summary>
            /// オブジェクトをXMLからデシリアライズします.
            /// </summary>
            /// <param name="context">シリアライズコンテキスト</param>
            /// <returns>処理が成功したときtrueを返します.</returns>
            public bool ReadXml(XmlDocSerializationContext context)
            {
                this.BitLabel = this.ReadListElement(context, "BitLabel", this.BitLabel);
                this.FloatLabel = this.ReadListElement(context, "FloatLabel", this.FloatLabel);

                return true;
            }

            /// <summary>
            /// オブジェクトをXMLにシリアライズします.
            /// </summary>
            /// <param name="context">シリアライズコンテキスト</param>
            /// <returns>処理が成功したときtrueを返します.</returns>
            public bool WriteXml(XmlDocSerializationContext context)
            {
                this.WriteEnumerableElement(context, "BitLabel", this.BitLabel);
                this.WriteEnumerableElement(context, "FloatLabel", this.FloatLabel);

                return true;
            }
        }

        #endregion
    }
}
