﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Drawing;
using System.Windows.Forms;
using System.Xml.Serialization;
using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Serialization;

namespace EffectMaker.BusinessLogic.Options
{
    /// <summary>
    /// Interface options.
    /// </summary>
    public class InterfaceOptions : OptionsBase
    {
        /// <summary>
        /// Initializes the InterfaceOptions instance.
        /// </summary>
        public InterfaceOptions()
        {
            this.MainFormBounds = Rectangle.Empty;
            this.DebugConsoleBounds = Rectangle.Empty;
            this.ControlTreeViewBounds = Rectangle.Empty;
            this.UseTipsDialog = true;
            this.ShowControlTreeView = true;
            this.OptionalMsgBoxSettings = new SerializableDictionary<string, OptionalMsgBoxSetting>();
            this.IsResizedOptionWindow = false;
            this.CurveEditorSnapSettings = new SerializableDictionary<string, CurveEditorSnapSetting>();
        }

        /// <summary>
        /// Copy the InterfaceOptions instance.
        /// </summary>
        /// <param name="src">The source optins to copy from.</param>
        public InterfaceOptions(InterfaceOptions src)
            : this()
        {
            this.Set(src);
        }

        /// <summary>
        /// Get or set the bounds of the main form.
        /// </summary>
        public Rectangle MainFormBounds { get; set; }

        /// <summary>
        /// Get or set the bounds of the debug console.
        /// </summary>
        public Rectangle DebugConsoleBounds { get; set; }

        /// <summary>
        /// Get or set the bounds of the control tree view.
        /// </summary>
        public Rectangle ControlTreeViewBounds { get; set; }

        /// <summary>
        /// Get or set the flag indicating whether to show the control tree view.
        /// </summary>
        public bool ShowControlTreeView { get; set; }

        /// <summary>
        /// Gets or sets whether to use Windows XP style.
        /// </summary>
        [XmlAttribute]
        public bool UseWindowsXPStyle { get; set; }

        /// <summary>
        /// 今日のヒントを使うかどうか.
        /// </summary>
        [XmlAttribute]
        public bool UseTipsDialog { get; set; }

        /// <summary>
        /// Gets or sets flag telling whether clicking into a text box select all its text or not.
        /// </summary>
        [XmlAttribute]
        public bool ClickToSelectAllInTextBox { get; set; }

        /// <summary>
        /// Get or set the flag indicating whether to allow users to change the option window size.
        /// </summary>
        [XmlAttribute]
        public bool IsResizedOptionWindow { get; set; }

        /// <summary>
        /// Get or set the bounds of the option form.
        /// </summary>
        public Rectangle OptionFormBounds { get; set; }

        /// <summary>
        /// Gets or sets whether top and bottom layout or not.
        /// </summary>
        public bool UseSingleColumnLayout { get; set; }

        /// <summary>
        /// 使用するテキストエディタのフルパスを取得または設定します。
        /// </summary>
        public string TextEditorPath { get; set; }

        /// <summary>
        /// カーブエディタのスナップ設定を取得します。
        /// </summary>
        public SerializableDictionary<string, CurveEditorSnapSetting> CurveEditorSnapSettings { get; private set; }

        /// <summary>
        /// 表示したヒントのインデックスを取得または設定します。
        /// </summary>
        public int HintIndex { get; set; }

        /// <summary>
        /// メッセージボックスの表示オプションを設定または取得します。
        /// </summary>
        public SerializableDictionary<string, OptionalMsgBoxSetting> OptionalMsgBoxSettings { get; set; }

        /// <summary>
        /// Set data from the source options.
        /// </summary>
        /// <param name="src">The source options.</param>
        /// <returns>True on success.</returns>
        public override bool Set(object src)
        {
            var options = src as InterfaceOptions;
            if (options == null)
            {
                return false;
            }

            this.MainFormBounds = options.MainFormBounds;
            this.DebugConsoleBounds = options.DebugConsoleBounds;
            this.ControlTreeViewBounds = options.ControlTreeViewBounds;
            this.ShowControlTreeView = options.ShowControlTreeView;
            this.UseWindowsXPStyle = options.UseWindowsXPStyle;
            this.UseTipsDialog = options.UseTipsDialog;
            this.ClickToSelectAllInTextBox = options.ClickToSelectAllInTextBox;
            this.UseSingleColumnLayout = options.UseSingleColumnLayout;
            this.TextEditorPath = options.TextEditorPath;
            this.HintIndex = options.HintIndex;
            this.IsResizedOptionWindow = options.IsResizedOptionWindow;
            this.OptionFormBounds = options.OptionFormBounds;

            this.OptionalMsgBoxSettings.Clear();
            options.OptionalMsgBoxSettings.ForEach(
                p => this.OptionalMsgBoxSettings.Add(p.Key, (OptionalMsgBoxSetting)p.Value.Clone()));

            this.CurveEditorSnapSettings.Clear();
            options.CurveEditorSnapSettings.ForEach(
                p => this.CurveEditorSnapSettings.Add(p.Key, (CurveEditorSnapSetting)p.Value.Clone()));

            return true;
        }

        /// <summary>
        /// Deserializes from the given XML node.
        /// </summary>
        /// <param name="context">The data context needed for the deserialization.</param>
        /// <returns>True on success.</returns>
        public override bool ReadXml(XmlDocSerializationContext context)
        {
            this.MainFormBounds            = this.ReadElement(context, nameof(this.MainFormBounds), this.MainFormBounds);
            this.DebugConsoleBounds        = this.ReadElement(context, nameof(this.DebugConsoleBounds), this.DebugConsoleBounds);
            this.ControlTreeViewBounds     = this.ReadElement(context, nameof(this.ControlTreeViewBounds), this.ControlTreeViewBounds);
            this.ShowControlTreeView       = this.ReadElement(context, nameof(this.ShowControlTreeView), this.ShowControlTreeView);
            this.UseWindowsXPStyle         = this.ReadAttribute(context, nameof(this.UseWindowsXPStyle), this.UseWindowsXPStyle);
            this.UseTipsDialog             = this.ReadAttribute(context, nameof(this.UseTipsDialog), this.UseTipsDialog);
            this.ClickToSelectAllInTextBox = this.ReadAttribute(context, nameof(this.ClickToSelectAllInTextBox), this.ClickToSelectAllInTextBox);
            this.UseSingleColumnLayout     = this.ReadElement(context, nameof(this.UseSingleColumnLayout), this.UseSingleColumnLayout);
            this.OptionalMsgBoxSettings    = this.ReadElement(context, nameof(this.OptionalMsgBoxSettings), this.OptionalMsgBoxSettings);
            this.TextEditorPath            = this.ReadElement(context, nameof(this.TextEditorPath), this.TextEditorPath);
            this.CurveEditorSnapSettings   = this.ReadElement(context, nameof(this.CurveEditorSnapSettings), this.CurveEditorSnapSettings);
            this.HintIndex                 = this.ReadElement(context, nameof(this.HintIndex), this.HintIndex);

            this.IsResizedOptionWindow     = this.ReadElement(context, nameof(this.IsResizedOptionWindow), this.IsResizedOptionWindow);

            this.OptionFormBounds          = this.ReadElement(context, nameof(this.OptionFormBounds), this.OptionFormBounds);

            return true;
        }

        /// <summary>
        /// Serializes this object to a XML node.
        /// </summary>
        /// <param name="context">The data context needed for the serialization.</param>
        /// <returns>True on success.</returns>
        public override bool WriteXml(XmlDocSerializationContext context)
        {
            this.WriteElement(context, nameof(this.MainFormBounds), this.MainFormBounds);
            this.WriteElement(context, nameof(this.DebugConsoleBounds), this.DebugConsoleBounds);
            this.WriteElement(context, nameof(this.ControlTreeViewBounds), this.ControlTreeViewBounds);
            this.WriteElement(context, nameof(this.ShowControlTreeView), this.ShowControlTreeView);
            this.WriteAttribute(context, nameof(this.UseWindowsXPStyle), this.UseWindowsXPStyle);
            this.WriteAttribute(context, nameof(this.UseTipsDialog), this.UseTipsDialog);
            this.WriteAttribute(context, nameof(this.ClickToSelectAllInTextBox), this.ClickToSelectAllInTextBox);
            this.WriteElement(context, nameof(this.UseSingleColumnLayout), this.UseSingleColumnLayout);
            this.WriteElement(context, nameof(this.OptionalMsgBoxSettings), this.OptionalMsgBoxSettings);
            this.WriteElement(context, nameof(this.TextEditorPath), this.TextEditorPath);
            this.WriteElement(context, nameof(this.CurveEditorSnapSettings), this.CurveEditorSnapSettings);
            this.WriteElement(context, nameof(this.HintIndex), this.HintIndex);
            this.WriteElement(context, nameof(this.IsResizedOptionWindow), this.IsResizedOptionWindow);
            this.WriteElement(context, nameof(this.OptionFormBounds), this.OptionFormBounds);

            return true;
        }
    }

    /// <summary>
    /// カーブエディタのスナップのセッティングです。
    /// </summary>
    [Serializable]
    public class CurveEditorSnapSetting : ICloneable, ISettable, IXmlDocSerializable
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public CurveEditorSnapSetting()
        {
        }

        #region CopyConstructor, ISettable, ICloneable methods

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="src">初期値</param>
        public CurveEditorSnapSetting(CurveEditorSnapSetting src)
            : this()
        {
            this.Set(src);
        }

        /// <summary>
        /// 値をセットします。
        /// </summary>
        /// <param name="src">値</param>
        /// <returns>処理が成功したときは true、失敗したときは false を返します。</returns>
        public bool Set(object src)
        {
            var options = src as CurveEditorSnapSetting;
            if (options == null)
            {
                return false;
            }

            this.EnableSnap = options.EnableSnap;
            this.SnapValueX = options.SnapValueX;
            this.SnapValueY = options.SnapValueY;

            return true;
        }

        /// <summary>
        /// インスタンスのコピーを生成します。
        /// </summary>
        /// <returns>インスタンスのコピーを返します。</returns>
        public object Clone()
        {
            return new CurveEditorSnapSetting(this);
        }

        #endregion

        /// <summary>
        /// スナップを有効にするかどうか取得または設定します。
        /// </summary>
        public bool EnableSnap { get; set; }

        /// <summary>
        /// 横軸のスナップ値を取得または設定します。
        /// </summary>
        public int SnapValueX { get; set; }

        /// <summary>
        /// 縦軸のスナップ値を取得または設定します。
        /// </summary>
        public int SnapValueY { get; set; }

        /// <summary>
        /// Deserializes from the given XML node.
        /// </summary>
        /// <param name="context">The data context needed for the deserialization.</param>
        /// <returns>True on success.</returns>
        public bool ReadXml(XmlDocSerializationContext context)
        {
            this.EnableSnap = this.ReadElement(context, nameof(this.EnableSnap), this.EnableSnap);
            this.SnapValueX = this.ReadElement(context, nameof(this.SnapValueX), this.SnapValueX);
            this.SnapValueY = this.ReadElement(context, nameof(this.SnapValueY), this.SnapValueY);

            return true;
        }

        /// <summary>
        /// Serializes this object to a XML node.
        /// </summary>
        /// <param name="context">The data context needed for the serialization.</param>
        /// <returns>True on success.</returns>
        public bool WriteXml(XmlDocSerializationContext context)
        {
            this.WriteElement(context, nameof(this.EnableSnap), this.EnableSnap);
            this.WriteElement(context, nameof(this.SnapValueX), this.SnapValueX);
            this.WriteElement(context, nameof(this.SnapValueY), this.SnapValueY);

            return true;
        }
    }

    /// <summary>
    /// オプションメッセージボックスのセッティングです。
    /// </summary>
    [Serializable]
    public class OptionalMsgBoxSetting : ICloneable, ISettable, IXmlDocSerializable
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public OptionalMsgBoxSetting()
        {
            this.IsEnabled = false;
            this.DialogResult = DialogResult.None;
        }

        #region CopyConstructor, ISettable, ICloneable methods

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="src">初期値</param>
        public OptionalMsgBoxSetting(OptionalMsgBoxSetting src)
            : this()
        {
            this.Set(src);
        }

        /// <summary>
        /// 値をセットします。
        /// </summary>
        /// <param name="src">値</param>
        /// <returns>処理が成功したときは true、失敗したときは false を返します。</returns>
        public bool Set(object src)
        {
            var options = src as OptionalMsgBoxSetting;
            if (options == null)
            {
                return false;
            }

            this.IsEnabled = options.IsEnabled;
            this.DialogResult = options.DialogResult;

            return true;
        }

        /// <summary>
        /// インスタンスのコピーを生成します。
        /// </summary>
        /// <returns>インスタンスのコピーを返します。</returns>
        public object Clone()
        {
            return new OptionalMsgBoxSetting(this);
        }

        #endregion

        /// <summary>
        /// メッセージボックスを表示するかどうかのフラグを設定または取得します。
        /// </summary>
        public bool IsEnabled { get; set; }

        /// <summary>
        /// ダイアログの表示を省略するときユーザーの入力として返す結果を設定または取得します。
        /// </summary>
        public DialogResult DialogResult { get; set; }

        /// <summary>
        /// Deserializes from the given XML node.
        /// </summary>
        /// <param name="context">The data context needed for the deserialization.</param>
        /// <returns>True on success.</returns>
        public bool ReadXml(XmlDocSerializationContext context)
        {
            this.IsEnabled = this.ReadElement(context, "IsEnabled", this.IsEnabled);
            this.DialogResult = this.ReadElement(context, "DialogResult", this.DialogResult);

            return true;
        }

        /// <summary>
        /// Serializes this object to a XML node.
        /// </summary>
        /// <param name="context">The data context needed for the serialization.</param>
        /// <returns>True on success.</returns>
        public bool WriteXml(XmlDocSerializationContext context)
        {
            this.WriteElement(context, "IsEnabled", this.IsEnabled);
            this.WriteElement(context, "DialogResult", this.DialogResult);

            return true;
        }
    }
}
