﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using EffectMaker.BusinessLogic.IO;
using EffectMaker.Foundation.Core;
using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Serialization;

namespace EffectMaker.BusinessLogic.Options
{
    /// <summary>
    /// ファイルイベントオプションデータです。
    /// </summary>
    public class FileEventOptions : OptionsBase
    {
        /// <summary>
        /// 最近使用したファイルに保存する数です。
        /// </summary>
        private const int MaxLastLoaderFilesCount = 20;

        /// <summary>
        /// プロジェクト設定ファイルパスです。
        /// </summary>
        private ExpandablePath userSettingsFilePath;

        /// <summary>
        /// プリセットデータフォルダパスです。
        /// </summary>
        private ExpandablePath presetFolderPath;

        /// <summary>
        /// 保存前コマンドパスです。
        /// </summary>
        private ExpandablePath preSaveCommandFilePath;

        /// <summary>
        /// 保存後コマンドパスです。
        /// </summary>
        private ExpandablePath postSaveCommandFilePath;

        /// <summary>
        /// 削除前コマンドパスです。
        /// </summary>
        private ExpandablePath preDeleteCommandFilePath;

        /// <summary>
        /// 削除後コマンドパスです。
        /// </summary>
        private ExpandablePath postDeleteCommandFilePath;

        /// <summary>
        /// オープン前コマンドパスを取得または設定します。
        /// </summary>
        private ExpandablePath preOpenCommandFilePath;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public FileEventOptions()
        {
            this.userSettingsFilePath      = new ExpandablePath();

            this.presetFolderPath          = new ExpandablePath(Path.Combine(IOConstants.ExecutableFolderPath, "Presets"));
            this.preSaveCommandFilePath    = new ExpandablePath();
            this.postSaveCommandFilePath   = new ExpandablePath();
            this.preDeleteCommandFilePath  = new ExpandablePath();
            this.postDeleteCommandFilePath = new ExpandablePath();
            this.preOpenCommandFilePath    = new ExpandablePath();

            this.LastLoadedFileList        = new List<string>();
            this.ExportFolderPathList      = new List<string>();
            this.LastAccessedDirectories   = new SerializableDictionary<LastAccessDirectoryTypes, string>();
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="src">初期値</param>
        public FileEventOptions(FileEventOptions src)
            : this()
        {
            this.Set(src);
        }

        /// <summary>
        /// プロジェクト設定ファイルパスを取得または設定します。
        /// </summary>
        public string UserSettingsFilePath
        {
            get { return this.userSettingsFilePath.ExpandedPath; }
            set { this.userSettingsFilePath.Path = value; }
        }

        /// <summary>
        /// プリセットデータフォルダパスを取得または設定します。
        /// </summary>
        public string PresetFolderPath
        {
            get { return this.presetFolderPath.ExpandedPath; }
            set { this.presetFolderPath.Path = value; }
        }

        /// <summary>
        /// 保存前コマンドパスを取得または設定します。
        /// </summary>
        public string PreSaveCommandFilePath
        {
            get { return this.preSaveCommandFilePath.ExpandedPath; }
            set { this.preSaveCommandFilePath.Path = value; }
        }

        /// <summary>
        /// 保存後コマンドパスを取得または設定します。
        /// </summary>
        public string PostSaveCommandFilePath
        {
            get { return this.postSaveCommandFilePath.ExpandedPath; }
            set { this.postSaveCommandFilePath.Path = value; }
        }

        /// <summary>
        /// 削除前コマンドパスを取得または設定します。
        /// </summary>
        public string PreDeleteCommandFilePath
        {
            get { return this.preDeleteCommandFilePath.ExpandedPath; }
            set { this.preDeleteCommandFilePath.Path = value; }
        }

        /// <summary>
        /// 削除後コマンドパスを取得または設定します。
        /// </summary>
        public string PostDeleteCommandFilePath
        {
            get { return this.postDeleteCommandFilePath.ExpandedPath; }
            set { this.postDeleteCommandFilePath.Path = value; }
        }

        /// <summary>
        /// オープン前コマンドパスを取得または設定します。
        /// </summary>
        public string PreOpenCommandFilePath
        {
            get { return this.preOpenCommandFilePath.ExpandedPath; }
            set { this.preOpenCommandFilePath.Path = value; }
        }

        /// <summary>
        /// 最近使用したファイルのパスを取得または設定します。
        /// </summary>
        public List<string> LastLoadedFileList { get; set; }

        /// <summary>
        /// エクスポート先のフォルダパスを取得または設定します。
        /// </summary>
        public List<string> ExportFolderPathList { get; set; }

        /// <summary>
        /// ファイル選択ダイアログで最後に開いていたフォルダのパスを取得または設定します。
        /// </summary>
        public SerializableDictionary<LastAccessDirectoryTypes, string> LastAccessedDirectories { get; set; }

        /// <summary>
        /// Deserializes from the given XML node.
        /// </summary>
        /// <param name="context">The data context needed for the deserialization.</param>
        /// <returns>True on success.</returns>
        public override bool ReadXml(XmlDocSerializationContext context)
        {
            this.userSettingsFilePath.Path      = this.ReadElement(context, "UserSettingsFilePath", this.UserSettingsFilePath).Trim();

            this.presetFolderPath.Path          = this.ReadElement(context, "PresetFolderPath", this.PresetFolderPath).Trim();
            this.preSaveCommandFilePath.Path    = this.ReadElement(context, "PreSaveCommandFilePath", this.PreSaveCommandFilePath).Trim();
            this.postSaveCommandFilePath.Path   = this.ReadElement(context, "PostSaveCommandFilePath", this.PostSaveCommandFilePath).Trim();
            this.preDeleteCommandFilePath.Path  = this.ReadElement(context, "PreDeleteCommandFilePath", this.PreDeleteCommandFilePath).Trim();
            this.postDeleteCommandFilePath.Path = this.ReadElement(context, "PostDeleteCommandFilePath", this.PostDeleteCommandFilePath).Trim();
            this.preOpenCommandFilePath.Path    = this.ReadElement(context, "PreOpenCommandFilePath", this.PreOpenCommandFilePath).Trim();

            this.LastLoadedFileList             = this.ReadListElement(context, "LastLoadedFileList", this.LastLoadedFileList);
            this.ExportFolderPathList           = this.ReadListElement(context, "ExportFolderPathList", this.ExportFolderPathList);
            this.LastAccessedDirectories        = this.ReadElement(context, "LastAccessedDirectories", this.LastAccessedDirectories);

            return true;
        }

        /// <summary>
        /// Serializes this object to a XML node.
        /// </summary>
        /// <param name="context">The data context needed for the serialization.</param>
        /// <returns>True on success.</returns>
        public override bool WriteXml(XmlDocSerializationContext context)
        {
            this.WriteElement(context, "UserSettingsFilePath", this.userSettingsFilePath.Path);

            this.WriteElement(context, "PresetFolderPath", this.presetFolderPath.Path);
            this.WriteElement(context, "PreSaveCommandFilePath", this.preSaveCommandFilePath.Path);
            this.WriteElement(context, "PostSaveCommandFilePath", this.postSaveCommandFilePath.Path);
            this.WriteElement(context, "PreDeleteCommandFilePath", this.preDeleteCommandFilePath.Path);
            this.WriteElement(context, "PostDeleteCommandFilePath", this.postDeleteCommandFilePath.Path);
            this.WriteElement(context, "PreOpenCommandFilePath", this.preOpenCommandFilePath.Path);

            this.WriteEnumerableElement(context, "LastLoadedFileList", this.LastLoadedFileList);
            this.WriteEnumerableElement(context, "ExportFolderPathList", this.ExportFolderPathList);
            this.WriteElement(context, "LastAccessedDirectories", this.LastAccessedDirectories);

            return true;
        }

        /// <summary>
        /// Set data from the source options.
        /// </summary>
        /// <param name="src">The source options.</param>
        /// <returns>True on success.</returns>
        public override bool Set(object src)
        {
            FileEventOptions options = src as FileEventOptions;

            if (options == null)
            {
                return false;
            }

            this.userSettingsFilePath.Set(options.userSettingsFilePath);

            this.presetFolderPath.Set(options.presetFolderPath);
            this.preSaveCommandFilePath.Set(options.preSaveCommandFilePath);
            this.postSaveCommandFilePath.Set(options.postSaveCommandFilePath);
            this.preDeleteCommandFilePath.Set(options.preDeleteCommandFilePath);
            this.postDeleteCommandFilePath.Set(options.postDeleteCommandFilePath);
            this.preOpenCommandFilePath.Set(options.preOpenCommandFilePath);

            // Listをコピー
            this.LastLoadedFileList.Clear();
            this.LastLoadedFileList.AddRange(options.LastLoadedFileList);

            this.ExportFolderPathList.Clear();
            this.ExportFolderPathList.AddRange(options.ExportFolderPathList);

            // Dictionaryをコピー
            this.LastAccessedDirectories.Clear();
            options.LastAccessedDirectories.ForEach(p => this.LastAccessedDirectories.Add(p.Key, p.Value));

            return true;
        }

        /// <summary>
        /// 最近使用したファイルを追加します.
        /// </summary>
        /// <param name="filePath">ファイルパス</param>
        public void AddLastLoadedFile(string filePath)
        {
            Debug.Assert(string.IsNullOrEmpty(filePath) == false, "ファイルパスが不正");

            // 登録済みのファイルパスは先頭に移動させるため一旦リストから削除
            this.LastLoadedFileList.Remove(filePath);

            // リストの上限を超えるときは一番古いファイルを削除
            if (this.LastLoadedFileList.Count >= MaxLastLoaderFilesCount)
            {
                this.LastLoadedFileList.RemoveAt(0);
            }

            // ファイルパスを追加
            this.LastLoadedFileList.Add(filePath);
        }

        /// <summary>
        /// 最近使用したファイルリストをクリアします.
        /// </summary>
        public void ClearLastLoadedFileList()
        {
            this.LastLoadedFileList.Clear();
        }

        /// <summary>
        /// 最近使用したファイルを追加します.
        /// </summary>
        /// <param name="folderPath">ファイルパス</param>
        public void AddExportFolderPath(string folderPath)
        {
            Debug.Assert(string.IsNullOrEmpty(folderPath) == false, "ファイルパスが不正");

            // 登録済みのファイルパスは先頭に移動させるため一旦リストから削除
            this.ExportFolderPathList.Remove(folderPath);

            // リストの上限を超えるときは一番古いファイルを削除
            if (this.ExportFolderPathList.Count >= MaxLastLoaderFilesCount)
            {
                this.ExportFolderPathList.RemoveAt(0);
            }

            // ファイルパスを追加
            if (this.ExportFolderPathList.Count > 0)
            {
                this.ExportFolderPathList.Insert(0, folderPath);
            }
            else
            {
                this.ExportFolderPathList.Add(folderPath);
            }
        }

        /// <summary>
        /// エクスポート先フォルダパスリストをクリアします.
        /// </summary>
        public void ClearExportFolderPathList()
        {
            this.ExportFolderPathList.Clear();
        }
    }
}
