﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

////#define DEBUG_MODEL_INFO_MANAGER

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Threading;
using EffectMaker.BusinessLogic.Protocol;
using EffectMaker.BusinessLogic.SpecDefinitions;
using EffectMaker.BusinessLogic.ViewerMessages;
using EffectMaker.Communicator;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Model.Types;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.BusinessLogic.Manager
{
    /// <summary>
    /// モデル情報を管理します.
    /// </summary>
    public static class ModelInfoManager
    {
        /// <summary>
        /// 通信メッセージを受信するProxyクラスです.
        /// </summary>
        public static readonly MessageNotifier Notifier = new MessageNotifier();

        /// <summary>
        /// モデルバイナリのキャッシュ情報です.
        /// </summary>
        private static Dictionary<string, BinaryCacheInfo> cacheInfo = new Dictionary<string, BinaryCacheInfo>();

        /// <summary>
        /// ビューアデータです.
        /// </summary>
        private static ViewerData viewerData;

        /// <summary>
        /// メインスレッドへの同期オブジェクトです.
        /// 通信スレッドのイベントハンドラでフォームを編集するために使います.
        /// </summary>
        private static SynchronizationContext syncContext;

        /// <summary>
        /// モデル情報です.
        /// </summary>
        private static Dictionary<Guid, ModelData> modelInfo;

        /// <summary>
        /// staticコンストラクタです。
        /// </summary>
        static ModelInfoManager()
        {
            modelInfo = new Dictionary<Guid, ModelData>();
            SpecManager.CurrentSpecChanged += (sender, args) =>
            {
                var caches = cacheInfo.ToArray();
                foreach (var pair in caches)
                {
                    UpdateCache(pair.Key, true);
                }
            };
        }

        /// <summary>
        /// モデル名が変わったときのイベントです.
        /// </summary>
        public static event EventHandler ModelNameChanged;

        /// <summary>
        /// ボーン名が変わったときのイベントです.
        /// </summary>
        public static event EventHandler BoneNamesChanged;

        /// <summary>
        /// モデル名でソートされたモデル名とGUIDのリストを取得します.
        /// </summary>
        /// <returns>モデル名とGUIDのリストを返します.</returns>
        public static IEnumerable<KeyValuePair<string, Guid>> GetSortedModelNameAndGuidList()
        {
            var sortedList = viewerData.ModelList
                .Select(x => new KeyValuePair<string, Guid>(x.Name, x.Guid))
                .OrderBy(x => x.Key);

            return sortedList;
        }

        /// <summary>
        /// GUIDに対応するモデルを取得します.
        /// </summary>
        /// <param name="guid">GUID</param>
        /// <returns>GUIDに対応するモデルを返します.</returns>
        public static ModelData GetModel(Guid guid)
        {
            ModelData model;
            modelInfo.TryGetValue(guid, out model);
            return model;
        }

        /// <summary>
        /// IDに対応するモデルバイナリを取得します.
        /// </summary>
        /// <param name="id">ID</param>
        /// <returns>IDに対応するモデルバイナリを返します.</returns>
        public static byte[] GetModelBinary(int id)
        {
            KeyValuePair<string, BinaryCacheInfo> info = cacheInfo.FirstOrDefault(item => item.Value.Id == id);
            bool isDefault = info.Equals(default(KeyValuePair<string, BinaryCacheInfo>));

            if (isDefault == false)
            {
                return info.Value.BinaryData;
            }
            else
            {
                return null;
            }
        }

        /// <summary>
        /// 初期化処理です.
        /// </summary>
        /// <param name="viewerData">ビューアデータ</param>
        /// <param name="syncContext">同期オブジェクト</param>
        public static void Initialize(ViewerData viewerData, SynchronizationContext syncContext)
        {
            ModelInfoManager.viewerData = viewerData;
            ModelInfoManager.syncContext = syncContext;

            // ワークスペースを作りなおして2回目以降初期化する際には以下3点が必要
            ModelNameChanged = null;
            BoneNamesChanged = null;
            modelInfo.Clear();

            foreach (ModelData model in viewerData.ModelList)
            {
                modelInfo.Add(model.Guid, model);
            }
        }

        /// <summary>
        /// モデルデータを更新します。
        /// </summary>
        /// <param name="modelData">モデルデータモデル</param>
        public static void UpdateModel(ModelData modelData)
        {
            if (modelData == null)
                throw new ArgumentNullException(nameof(modelData));

            // 処理に必要なデータモデルを取得
            ModelBasicSettingData modelBasicSettingData = modelData.ModelBasicSettingData;
            ModelBasicModelData modelBasicModelData = modelBasicSettingData.ModelBasicModelData;
            ModelBasicAnimationData modelBasicAnimationData = modelBasicSettingData.ModelBasicAnimationData;
            ModelBasicVisibilityAnimationData modelBasicVisibilityAnimationData = modelBasicSettingData.ModelBasicVisibilityAnimationData;

            // 処理に必要な情報を収集
            string modelFilePath = modelBasicModelData.ModelFilePath;
            string skelAnimFilePath = modelBasicAnimationData.AnimationFilePath;
            string visiAnimFilePath = modelBasicVisibilityAnimationData.AnimationFilePath;
            int oldBinaryId = modelData.ModelBinaryId;

            // モデルファイルが指定されていないとき
            if (string.IsNullOrEmpty(modelFilePath))
            {
                modelData.ModelBinaryId = -1;
                CacheReferenceCheck(oldBinaryId);
                return;
            }

            // キャッシュチェック
            string key = GetModelCacheKey(modelFilePath, skelAnimFilePath, visiAnimFilePath);
            BinaryCacheInfo info;
            bool cacheHit = cacheInfo.TryGetValue(key, out info);

            if (cacheHit)
            {
                // タイムスタンプが変わっていたときはキャッシュを更新
                if (info.ModelTimestamp != GetTimeStamp(modelFilePath) ||
                    info.SkeltalAnimationTimestamp != GetTimeStamp(skelAnimFilePath) ||
                    info.VisibilityAnimationTimestamp != GetTimeStamp(visiAnimFilePath))
                {
                    UpdateCache(key);
                }

                modelData.ModelBinaryId = info.Id;
                CacheReferenceCheck(oldBinaryId);
            }
            else
            {
                // コンバートを実行
                var stream = new MemoryStream();
                var result = PrimitiveManager.Instance.ConvertBinary(
                        modelFilePath, new string[] { skelAnimFilePath, visiAnimFilePath }, stream);

                if (result.Result != LoadModelResults.Success)
                {
                    Logger.Log(
                        LogLevels.Warning,
                        "ModelInfoManager.UpdateModel : Failed to convert model {0}/{1}/{2}",
                        modelFilePath,
                        skelAnimFilePath,
                        visiAnimFilePath);

                    Logger.Log(
                        "LogView",
                        LogLevels.Warning,
                        Properties.Resources.ModelInfoManagerFailedToLoadModel,
                        modelFilePath);

                    if (string.IsNullOrEmpty(skelAnimFilePath) == false)
                    {
                        Logger.Log(
                            "LogView",
                            LogLevels.Warning,
                            Properties.Resources.ModelInfoManagerFailedToLoadSkeltalAnimaion,
                            modelFilePath,
                            skelAnimFilePath);
                    }

                    if (string.IsNullOrEmpty(visiAnimFilePath) == false)
                    {
                        Logger.Log(
                            "LogView",
                            LogLevels.Warning,
                            Properties.Resources.ModelInfoManagerFailedToLoadVisibilityAnimaion,
                            modelFilePath,
                            visiAnimFilePath);
                    }

                    foreach (var modelMessage in result.DetailMessages)
                    {
                        foreach (string messageArgs in modelMessage.MessageArgs)
                        {
                            Logger.Log(
                                "LogView",
                                LogLevels.Warning,
                                Properties.Resources.PrimitiveManagerFailedToLoadTexture,
                                modelFilePath,
                                messageArgs);
                        }
                    }

                    modelData.ModelBinaryId = -1;
                    CacheReferenceCheck(oldBinaryId);
                    return;
                }

                // 新しいIDを作成
                int newId = 0;
                if (cacheInfo.Count > 0)
                {
                    newId = cacheInfo.Max(item => item.Value.Id) + 1;
                }

                // キャッシュを更新
                info.ModelFilePath = modelFilePath;
                info.ModelTimestamp = GetTimeStamp(modelFilePath);
                info.SkeltalAnimationFilePath = skelAnimFilePath;
                info.SkeltalAnimationTimestamp = GetTimeStamp(skelAnimFilePath);
                info.VisibilityAnimationFilePath = visiAnimFilePath;
                info.VisibilityAnimationTimestamp = GetTimeStamp(visiAnimFilePath);
                info.Id = newId;
                info.BinaryData = stream.ToArray();

                cacheInfo.Add(key, info);

                modelData.ModelBinaryId = info.Id;
                CacheReferenceCheck(oldBinaryId);
            }
        }

        /// <summary>
        /// モデルバイナリIDを取得します.
        /// </summary>
        /// <param name="modelData">モデルデータモデル</param>
        /// <returns>モデルバイナリIDを返します.</returns>
        public static int GetModelId(ModelData modelData)
        {
            if (modelData == null)
                throw new ArgumentNullException(nameof(modelData));

            // 処理に必要なデータモデルを取得
            ModelBasicSettingData modelBasicSettingData = modelData.ModelBasicSettingData;
            ModelBasicModelData modelBasicModelData = modelBasicSettingData.ModelBasicModelData;
            ModelBasicAnimationData modelBasicAnimationData = modelBasicSettingData.ModelBasicAnimationData;
            ModelBasicVisibilityAnimationData modelBasicVisibilityAnmimationData = modelBasicSettingData.ModelBasicVisibilityAnimationData;

            // 処理に必要な情報を収集
            string modelFilePath = modelBasicModelData.ModelFilePath;
            string skelAnimFilePath = modelBasicAnimationData.AnimationFilePath;
            string visiAnimFilePath = modelBasicVisibilityAnmimationData.AnimationFilePath;
            int oldBinaryId = modelData.ModelBinaryId;

            // キャッシュチェック
            string key = GetModelCacheKey(modelFilePath, skelAnimFilePath, visiAnimFilePath);
            BinaryCacheInfo info;
            bool cacheHit = cacheInfo.TryGetValue(key, out info);

            if (cacheHit)
            {
                return info.Id;
            }
            else
            {
                return -1;
            }
        }

        /// <summary>
        /// モデル情報の更新イベントを発生させます。
        /// </summary>
        public static void RaiseModelEvents()
        {
            // モデル名の変更を通知
            TriggerModelNameChangedEvent(null, EventArgs.Empty);

            // ボーン名の変更を通知
            TriggerBoneNamesChangedEvent(null, EventArgs.Empty);
        }

        /// <summary>
        /// メッセージ処理のテストを行います。
        /// </summary>
        /// <param name="msg">Viewerから送信されたメッセージの再現データ</param>
        public static void ProcessMessageTest(Message msg)
        {
            OnMessageReceived(msg);
        }

        /// <summary>
        /// ファイルのタイムスタンプを取得します.
        /// 例外を捕捉したときはデフォルト値を返します。
        /// </summary>
        /// <param name="path">ファイルパス</param>
        /// <returns>タイムスタンプを返します.</returns>
        private static DateTime GetTimeStamp(string path)
        {
            try
            {
                return File.GetLastWriteTime(path);
            }
            catch (Exception)
            {
                return new DateTime(0);
            }
        }

        /// <summary>
        /// キャッシュへの参照をチェックして参照切れのキャッシュがあれば削除します.
        /// </summary>
        /// <param name="checkId">キャッシュのID</param>
        private static void CacheReferenceCheck(int checkId)
        {
            if (checkId < 0)
            {
                return;
            }

            // キャッシュへの参照をチェック
            bool find = viewerData.ModelList.Any(item => item.ModelBinaryId == checkId);

            // 参照があるときは何もしない
            if (find)
            {
                return;
            }

            // 参照切れのときは該当するキャッシュを削除
            string key = cacheInfo.FirstOrDefault(
                item => item.Value.Id == checkId).Key;

            if (string.IsNullOrEmpty(key) == false)
            {
                cacheInfo.Remove(key);
            }
        }

        /// <summary>
        /// バイナリキャッシュを更新します.
        /// </summary>
        /// <param name="key">キー</param>
        /// <param name="keepId">IDを維持したまま更新する場合はtrue,発行しなおす場合はfalse(省略).</param>
        private static void UpdateCache(string key, bool keepId = false)
        {
            BinaryCacheInfo info = cacheInfo[key];
            int oldId = -1;
            if (keepId)
            {
                oldId = info.Id;
            }

            // コンバートを実行
            var stream = new MemoryStream();
            var result = PrimitiveManager.Instance.ConvertBinary(
                info.ModelFilePath,
                new string[] { info.SkeltalAnimationFilePath, info.VisibilityAnimationFilePath },
                stream);

            if (result.Result != Foundation.Model.Types.LoadModelResults.Success)
            {
                Logger.Log(
                    LogLevels.Warning,
                    "ModelInfoManager.UpdateModel : Failed to convert model {0}/{1}/{2}",
                    info.ModelFilePath,
                    info.SkeltalAnimationFilePath,
                    info.VisibilityAnimationFilePath);

                Logger.Log(
                    "LogView",
                    LogLevels.Warning,
                    Properties.Resources.ModelInfoManagerFailedToLoadModel,
                    info.ModelFilePath);

                if (string.IsNullOrEmpty(info.SkeltalAnimationFilePath) == false)
                {
                    Logger.Log(
                        "LogView",
                        LogLevels.Warning,
                        Properties.Resources.ModelInfoManagerFailedToLoadSkeltalAnimaion,
                        info.ModelFilePath,
                        info.SkeltalAnimationFilePath);
                }

                if (string.IsNullOrEmpty(info.VisibilityAnimationFilePath) == false)
                {
                    Logger.Log(
                        "LogView",
                        LogLevels.Warning,
                        Properties.Resources.ModelInfoManagerFailedToLoadVisibilityAnimaion,
                        info.ModelFilePath,
                        info.VisibilityAnimationFilePath);
                }

                foreach (var modelMessage in result.DetailMessages)
                {
                    foreach (string messageArgs in modelMessage.MessageArgs)
                    {
                        Logger.Log(
                            "LogView",
                            LogLevels.Warning,
                            Properties.Resources.PrimitiveManagerFailedToLoadTexture,
                            info.ModelFilePath,
                            messageArgs);
                    }
                }

                return;
            }

            // キャッシュを更新
            info.ModelTimestamp = GetTimeStamp(info.ModelFilePath);
            info.SkeltalAnimationTimestamp = GetTimeStamp(info.SkeltalAnimationFilePath);
            info.VisibilityAnimationTimestamp = GetTimeStamp(info.VisibilityAnimationFilePath);
            info.BinaryData = stream.ToArray();

            if (keepId)
            {
                info.Id = oldId;
            }

            cacheInfo[key] = info;
        }

        /// <summary>
        /// メッセージ通知の処理です.
        /// </summary>
        /// <param name="msg">Viewerから送信されたパケットです</param>
        private static void OnMessageReceived(Message msg)
        {
            var reader = new PacketReader();
            MessageBase[] messages = reader.Read(msg.Buffer);

            foreach (var message in messages)
            {
                // ModelInfoMessage以外は処理しない
                if (message.MessageType != MessageTypes.ModelInfo)
                {
                    return;
                }

                // ModelInfoMessageを取得
                var modelInfoMsg = message as ModelInfoMessage;
                Debug.Assert(modelInfoMsg != null, "ModelInfoMessageの型が不正");

                switch (modelInfoMsg.ModelInfoMessageType)
                {
                    case ModelInfoMessageType.Clear:
                        #if DEBUG_MODEL_INFO_MANAGER
                            Logger.Log(LogLevels.Information, "ModelInfoManager.OnMessageReceived : Received clear message");
                        #endif

                        ProcessClearMessage(modelInfoMsg);
                        break;

                    case ModelInfoMessageType.Begin:
                        #if DEBUG_MODEL_INFO_MANAGER
                            Logger.Log(LogLevels.Information, "ModelInfoManager.OnMessageReceived : Received begin message");
                        #endif

                        break;

                    case ModelInfoMessageType.Send:
                        #if DEBUG_MODEL_INFO_MANAGER
                            Logger.Log(LogLevels.Information, "ModelInfoManager.OnMessageReceived : Received send message");
                        #endif

                        ProcessSendMessage(modelInfoMsg);
                        break;

                    case ModelInfoMessageType.Remove:
                        #if DEBUG_MODEL_INFO_MANAGER
                            Logger.Log(LogLevels.Information, "ModelInfoManager.OnMessageReceived : Received remove message");
                        #endif

                        ProcessRemoveMessage(modelInfoMsg);
                        break;

                    case ModelInfoMessageType.End:
                        #if DEBUG_MODEL_INFO_MANAGER
                            Logger.Log(LogLevels.Information, "ModelInfoManager.OnMessageReceived : Received end message");
                        #endif

                        ProcessEndMessage(modelInfoMsg);
                        break;
                }
            }
        }

        /// <summary>
        /// ゲームでロードしたモデル情報をクリアします.
        /// </summary>
        /// <param name="msg">メッセージ</param>
        private static void ProcessClearMessage(ModelInfoMessage msg)
        {
            // ゲームでロードしたモデルの情報を削除する
            var removeList = new List<ModelData>();

            foreach (ModelData modelData in viewerData.ModelList)
            {
                if (modelData.IsRuntime == true)
                {
                    modelInfo.Remove(modelData.Guid);
                    removeList.Add(modelData);
                }
            }

            foreach (var modelData in removeList)
            {
                viewerData.ModelList.Remove(modelData);
            }
        }

        /// <summary>
        /// モデル名, ボーン名を設定します.
        /// </summary>
        /// <param name="msg">メッセージ</param>
        private static void ProcessSendMessage(ModelInfoMessage msg)
        {
            // GUIDの一致するモデルを検索
            ModelData model = GetModel(msg.Guid);

            // ランタイムでロードしたモデル用のデータモデルを作成
            if (model == null)
            {
                if (msg.IsRuntimeModel)
                {
                    model = new ModelData();
                    model.Guid = msg.Guid;
                    model.IsRuntime = true;
                    modelInfo.Add(msg.Guid, model);
                    viewerData.ModelList.Add(model);
                }
                else
                {
                    // モデルが見つからなかったとき処理をキャンセル
                    Logger.Log(LogLevels.Warning, "ModelInfoManager.ProcessSendMessage : Model not found.");
                    return;
                }
            }

            // モデル名を設定
            model.Name = msg.Name;

            // ボーン名をコンマ','区切りに分割
            string[] boneNameArray = msg.BoneNames.Split(',');

            // 受信したボーン名の数とBoneCountの数が違ったとき
            if (boneNameArray.Length != msg.BoneCount)
            {
                Logger.Log(LogLevels.Warning, "ModelInfoManager.ProcessSendMessage : The received bone count is invalid.");

                // ボーン名の数を強制的にBoneCountに合わせる
                string[] replaceArray = new string[msg.BoneCount];
                for (int i = 0; i < msg.BoneCount; ++i)
                {
                    if (i < boneNameArray.Length)
                    {
                        replaceArray[i] = boneNameArray[i];
                    }
                    else
                    {
                        replaceArray[i] = "bone " + i;
                    }
                }

                boneNameArray = replaceArray;
            }

            // ボーン名を設定
            model.BoneNames = new List<string>(boneNameArray);
        }

        /// <summary>
        /// ゲームでロードした特定のモデル情報を削除します。
        /// </summary>
        /// <param name="msg">メッセージ</param>
        private static void ProcessRemoveMessage(ModelInfoMessage msg)
        {
            // GUIDの一致するモデルを検索
            ModelData model = GetModel(msg.Guid);

            // モデルが見つからなければ処理をキャンセルする
            if (model == null || model.IsRuntime == false)
            {
                return;
            }

            // 指定されたモデルの情報を削除する
            modelInfo.Remove(model.Guid);
            viewerData.ModelList.Remove(model);
        }

        /// <summary>
        /// モデル名, ボーン名を確定します.
        /// </summary>
        /// <param name="msg">メッセージ</param>
        private static void ProcessEndMessage(ModelInfoMessage msg)
        {
            RaiseModelEvents();
        }

        /// <summary>
        /// キャッシュの取得に使うキーを取得します.
        /// </summary>
        /// <param name="modelFilePath">モデルファイルパス</param>
        /// <param name="akelAnimFilePath">スケルタルアニメーションファイルパス</param>
        /// <param name="visiAnimFilePath">ビジビリティアニメーションファイルパス</param>
        /// <returns>キーを返します.</returns>
        private static string GetModelCacheKey(string modelFilePath, string skelAnimFilePath, string visiAnimFilePath)
        {
            return $"{modelFilePath}:{skelAnimFilePath}:{visiAnimFilePath}";
        }

        /// <summary>
        /// モデル名が変わったときのイベントを発行します.
        /// </summary>
        /// <param name="sender">The sender object</param>
        /// <param name="e">event arguments</param>
        private static void TriggerModelNameChangedEvent(object sender, EventArgs e)
        {
            // 通信スレッドからメインスレッドに同期してイベントを実行
            // (イベントハンドラでフォームの変更などを行うため)
            syncContext.Send(s =>
            {
                if (ModelNameChanged != null)
                {
                    ModelNameChanged(sender, e);
                }
            },
            null);
        }

        /// <summary>
        /// ボーン名が変わったときのイベントを発行します.
        /// </summary>
        /// <param name="sender">The sender object</param>
        /// <param name="e">event arguments</param>
        private static void TriggerBoneNamesChangedEvent(object sender, EventArgs e)
        {
            // 通信スレッドからメインスレッドに同期してイベントを実行
            // (イベントハンドラでフォームの変更などを行うため)
            syncContext.Send(s =>
            {
                if (BoneNamesChanged != null)
                {
                    BoneNamesChanged(sender, e);
                }
            },
            null);
        }

        /// <summary>
        /// モデルバイナリのキャッシュ情報です.
        /// </summary>
        private struct BinaryCacheInfo
        {
            /// <summary>
            /// モデルファイルパスです。
            /// </summary>
            public string ModelFilePath;

            /// <summary>
            /// モデルファイルのタイムスタンプです。
            /// </summary>
            public DateTime ModelTimestamp;

            /// <summary>
            /// スケルタルアニメーションファイルパスです。
            /// </summary>
            public string SkeltalAnimationFilePath;

            /// <summary>
            /// スケルタルアニメーションファイルのタイムスタンプです。
            /// </summary>
            public DateTime SkeltalAnimationTimestamp;

            /// <summary>
            /// ビジビリティアニメーションファイルパスです。
            /// </summary>
            public string VisibilityAnimationFilePath;

            /// <summary>
            /// ビジビリティアニメーションファイルのタイムスタンプです。
            /// </summary>
            public DateTime VisibilityAnimationTimestamp;

            /// <summary>
            /// モデルバイナリです。
            /// </summary>
            public byte[] BinaryData;

            /// <summary>
            /// バイナリIDです。
            /// </summary>
            public int Id;
        }

        /// <summary>
        /// メッセージ通知を受信するProxyクラスです.
        /// ModelInfoManagerが直接IMessageListenerを実装する方法だと、ModelInfoManagerを使用している
        /// ViewModelのプロジェクト(EffectMaker.UILogic)にEffectMaker.Communicatorの参照を追加しないと
        /// いけなくなるので、間にMessageNotifierを挟んで回避.
        /// </summary>
        public class MessageNotifier : IMessageListener
        {
            /// <summary>
            /// メッセージ通知の処理です.
            /// </summary>
            /// <param name="msg">Viewerから送信されたパケットです</param>
            public void OnMessageReceived(Message msg)
            {
                ModelInfoManager.OnMessageReceived(msg);
            }
        }
    }

    /// <summary>
    /// ModelInfoMessage.
    /// </summary>
    public class ModelInfoMessage : MessageBase
    {
        /// <summary>
        /// モデル名の長さです.
        /// </summary>
        private const int ModelNameLength = 260 * 2;

        /// <summary>
        /// モデル情報メッセージタイプを取得します.
        /// </summary>
        public ModelInfoMessageType ModelInfoMessageType { get; private set; }

        /// <summary>
        /// モデルのGUIDを取得します.
        /// </summary>
        public Guid Guid { get; private set; }

        /// <summary>
        /// ランタイムでロードしたモデルかどうか取得します.
        /// </summary>
        public bool IsRuntimeModel { get; private set; }

        /// <summary>
        /// モデルの名前を取得します.
        /// </summary>
        public string Name { get; private set; }

        /// <summary>
        /// ボーン数を取得します.
        /// </summary>
        public uint BoneCount { get; private set; }

        /// <summary>
        /// ボーン名リストを取得します.
        /// 内容はカンマ区切りのリストです.
        /// </summary>
        public string BoneNames { get; private set; }

        /// <summary>
        /// Get the type of the message.
        /// </summary>
        public override MessageTypes MessageType
        {
            get
            {
                return MessageTypes.ModelInfo;
            }
        }

        /// <summary>
        /// Get the size of the message.
        /// </summary>
        public override int MessageSize
        {
            get
            {
                return this.ReceivedMessageSize;
            }
        }

        /// <summary>
        /// Read message data from binary data.
        /// </summary>
        /// <param name="stream">The stream contains the message's binary data.</param>
        /// <returns>True on success.</returns>
        public override bool Read(Stream stream)
        {
            if (base.Read(stream) == false)
            {
                return false;
            }

            // メッセージタイプを取得
            this.ModelInfoMessageType = (ModelInfoMessageType)BinaryConversionUtility.ForProtocol.ReadStream<uint>(stream);

            // サイズを取得
            uint size = BinaryConversionUtility.ForProtocol.ReadStream<uint>(stream);

            if (this.ModelInfoMessageType == ModelInfoMessageType.Send ||
                this.ModelInfoMessageType == ModelInfoMessageType.Remove)
            {
                // GUIDを取得
                this.Guid = BinaryConversionUtility.ForProtocol.ReadStream<Guid>(stream);

                // ランタイムでロードされたモデルかどうかのフラグを取得
                this.IsRuntimeModel = BinaryConversionUtility.ForProtocol.ReadStream<uint>(stream) == 1;

                // モデル名を取得
                char[] charModelName = new char[ModelNameLength];
                for (int i = 0; i < ModelNameLength; ++i)
                {
                    charModelName[i] = BinaryConversionUtility.ForProtocol.ReadStream<char>(stream);
                }

                this.Name = new string(charModelName).TrimEnd(new char[] { '\0' });

                // ボーン数を読み飛ばす
                this.BoneCount = BinaryConversionUtility.ForProtocol.ReadStream<uint>(stream);

                // ボーンデータサイズを取得
                uint boneDataSize = BinaryConversionUtility.ForProtocol.ReadStream<uint>(stream);

                // ボーン名リストを取得
                char[] charBoneNames = new char[boneDataSize];
                for (int i = 0; i < boneDataSize; ++i)
                {
                    charBoneNames[i] = BinaryConversionUtility.ForProtocol.ReadStream<char>(stream);
                }

                this.BoneNames = new string(charBoneNames).TrimEnd(new char[] { '\0' });
            }
            else
            {
                if (size != 0)
                {
                    Logger.Log(LogLevels.Warning, "ModelInfoMessage.Read(): Invalid message size.");
                }

                return false;
            }

            return true;
        }
    }
}
