﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Diagnostics;
using System.IO;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.BusinessLogic.ProjectConfig;
using EffectMaker.BusinessLogic.SpecDefinitions;
using EffectMaker.Foundation.Log;

namespace EffectMaker.BusinessLogic.IO
{
    /// <summary>
    /// ビューアを起動する処理を集約したクラスです。
    /// </summary>
    public static class ViewerRunner
    {
        /// <summary>
        /// PCビューアを起動します。
        /// </summary>
        /// <returns>ビューアのプロセスを返します。</returns>
        public static Process LaunchPCViewer()
        {
            // パッケージ内のビューアフォルダから実行ファイルのパスを得る
            var validatedPath = Path.Combine(
                IOConstants.ToolsDirectoryPath,
                @"Graphics\EffectViewer",
                SpecManager.CurrentSpec.PCViewerPath);

            // パッケージ用のパスで見つからなかったらリポジトリ内を想定して探す
            if (File.Exists(validatedPath) == false)
            {
                validatedPath = Path.Combine(
                    IOConstants.ToolsDirectoryPath,
                    @"Graphics\EffectViewerGfx",
                    SpecManager.CurrentSpec.PCViewerPath);
            }

            // ビューアが見つからなければエラーにする
            if (File.Exists(validatedPath) == false)
            {
                Logger.Log(
                    "LogView",
                    LogLevels.Error,
                    Properties.Resources.WarningViewerExecutableNotFound,
                    validatedPath);

                return null;
            }

            Logger.Log(LogLevels.Debug, "Start {0}", validatedPath);

            var info = new ProcessStartInfo
            {
                FileName = validatedPath,
                WorkingDirectory = Path.GetDirectoryName(validatedPath),
                Arguments = GenerateStartArguments(),
                CreateNoWindow = false,
                UseShellExecute = false,
            };

            Process process = Process.Start(info);

            return process;
        }

        /// <summary>
        /// 実機ビューアを起動します。
        /// </summary>
        /// <returns>ビューア起動処理のプロセスを返します。</returns>
        public static Process LaunchTargetViewer()
        {
            // 実機のスターターパスを得る
            var runOnTargetPath = Path.Combine(
                IOConstants.ToolsDirectoryPath,
                @"CommandLineTools\RunOnTarget.exe");

            // スターターが存在しなければエラーにする
            if (File.Exists(runOnTargetPath) == false)
            {
                Logger.Log(
                    "LogView",
                    LogLevels.Error,
                    Properties.Resources.WarningViewerExecutableNotFound,
                    runOnTargetPath);

                return null;
            }

            // パッケージ内のビューアフォルダから実行ファイルのパスを得る
            var validatedPath = Path.Combine(
                IOConstants.ToolsDirectoryPath,
                @"Graphics\EffectViewer",
                SpecManager.CurrentSpec.TargetViewerPath);

            // パッケージ用のパスで見つからなかったらリポジトリ内を想定して探す
            if (File.Exists(validatedPath) == false)
            {
                validatedPath = Path.Combine(
                    IOConstants.ToolsDirectoryPath,
                    @"Graphics\EffectViewerGfx",
                    SpecManager.CurrentSpec.TargetViewerPath);
            }

            // ビューアが見つからなければエラーにする
            if (File.Exists(validatedPath) == false)
            {
                Logger.Log(
                    "LogView",
                    LogLevels.Error,
                    Properties.Resources.WarningViewerExecutableNotFound,
                    validatedPath);

                return null;
            }

            Logger.Log(LogLevels.Debug, "Start {0} {1}", runOnTargetPath, validatedPath);

            var info = new ProcessStartInfo
            {
                FileName = runOnTargetPath,
                WorkingDirectory = Path.GetDirectoryName(runOnTargetPath),
                Arguments = validatedPath + " --no-wait" + " --" + GenerateStartArguments(),
                CreateNoWindow = true,
                UseShellExecute = false,
            };

            Process process = Process.Start(info);

            return process;
        }

        /// <summary>
        /// ビューアに渡す起動引数を生成します。
        /// </summary>
        /// <returns>ビューアに渡す起動引数を返します。</returns>
        private static string GenerateStartArguments()
        {
            string args = string.Empty;

            // リニアモードの引数を設定
            string strSwapChain = OptionStore.ProjectConfig.LinearMode
                ? "linear"
                : "non_linear";
            args += " --swap-chain " + strSwapChain;

            // テクスチャバッファの原点の引数を設定
            string strTextureOrigin = OptionStore.ProjectConfig.TextureOriginMode == TextureOriginMode.LowerLeft
                ? "lower_left"
                : "upper_left";
            args += " --texture-origin " + strTextureOrigin;

            // デプスの範囲の引数を設定
            string strDepthMode = OptionStore.ProjectConfig.DepthMode == DepthMode.NearIsMinusW
                ? "near_is_minus_w"
                : "near_is_zero";
            args += " --depth-mode " + strDepthMode;

            // [Experimental]厳密なGPUカウンタモードを使用するかどうか
            // MEMO: ON/OFF の指定だけなので、直接オプション名を書く。（ONのときにオプションを書くだけ）
            string strEnablePreciseGpuCounterMode = OptionStore.ProjectConfig.IsEnabledPreciseGpuCounterMode ? " --enable_precise_gpu_counter_mode" : "";
            args += strEnablePreciseGpuCounterMode;

            string additionalArguments = OptionStore.ProjectConfig.AdditionalViewerArguments;

            // 追加引数を設定
            if (string.IsNullOrEmpty(additionalArguments) == false)
            {
                args += " " + additionalArguments;
            }

            return args;
        }
    }
}
