﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Texture;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.BusinessLogic.IO
{
    /// <summary>
    /// AssetPathValidator.
    /// </summary>
    public static class AssetPathValidator
    {
        /// <summary>
        /// 与えられたアセットのファイルパスをロケートします。
        /// </summary>
        /// <param name="type">アセットタイプ</param>
        /// <param name="baseDir">ESETファイルのフォルダ(フルパス)。</param>
        /// <param name="filePath">アセットファイルのパス(フルパス)。</param>
        /// <param name="fullPath">ロケートしたファイルパス(フルパス)。</param>
        /// <returns>ロケートしたときTrueを返します。</returns>
        public static bool LocateAssetPath(
            AssetResourceTypes type,
            string baseDir,
            string filePath,
            out string fullPath)
        {
            // out変数を初期化
            fullPath = string.Empty;

            // ESETファイルのフォルダパスをチェック
            if (!PathUtility.IsPathSetAndRooted(baseDir))
            {
                Logger.Log("IsPathSetAndRooted(" + baseDir + ") failed.");
                return false;
            }

            // アセットファイルの名前を取得
            string fileName = Path.GetFileName(filePath);

            string path;

            // "basedir/fileName"のファイルがあるかチェック
            path = Path.Combine(baseDir, fileName);
            if (File.Exists(path))
            {
                fullPath = path;
                return true;
            }

            // アセットの相対パスを取得
            string assetDir = IOConstants.GetAssetFolderName(type);
            string assetPath = Path.Combine(assetDir, fileName);

            // ファイルが見つかるまでbaseDirの上の階層を辿っていく
            do
            {
                // "baseDir/assetDir/fileName"にファイルがあるかチェック
                path = Path.Combine(baseDir, assetPath);
                if (File.Exists(path))
                {
                    fullPath = path;
                    return true;
                }

                // baseDir = "baseDir/../"
                baseDir = Path.GetDirectoryName(baseDir);
                if (string.IsNullOrEmpty(baseDir))
                {
                    break;
                }
            }
            while (Path.IsPathRooted(baseDir));

            return false;
        }

        /// <summary>
        /// テクスチャの読み込みを行います。
        /// </summary>
        /// <param name="path">テクスチャパス</param>
        /// <returns>読み込み結果を返します。</returns>
        public static LoadTextureResult LoadTexture(
            string path)
        {
            if (!Path.IsPathRooted(path) ||
                !BusinessLogic.Manager.TextureManager.Instance.CanLoad(path))
            {
                return new LoadTextureResult(LoadTextureResultCode.FileNotFound);
            }

            return BusinessLogic.Manager.TextureManager.Instance.LoadTexture(path, true);
        }

        /// <summary>
        /// 与えられたアセットのファイルパスをロケート可能かを取得します。
        /// </summary>
        /// <param name="type">アセットタイプ</param>
        /// <param name="baseDir">ESETファイルのフォルダ(フルパス)。</param>
        /// <param name="filePath">アセットファイルのパス(フルパス)。</param>
        /// <returns>ロケート可能なときTrueを返します。</returns>
        public static bool CanLocateAssetPath(
            AssetResourceTypes type,
            string baseDir,
            string filePath)
        {
            // ファイルパスを小文字に統一
            baseDir = baseDir.ToLower();
            filePath = filePath.ToLower();

            // アセットの相対パスを取得
            string assetFolderName = IOConstants.GetAssetFolderName(type).ToLower();

            // 一つ上のフォルダ名を取得
            string assetDir = PathUtility.GetDirectoryName(filePath);
            string assetDirName = Path.GetFileName(assetDir);
            bool searchRecursive = false;

            // フォルダ名がアセット用のフォルダ名と同じとき、上にたどっていく
            if (assetDirName == assetFolderName)
            {
                searchRecursive = true;
                assetDir = Path.GetDirectoryName(assetDir);
            }

            do
            {
                if (baseDir.Equals(assetDir))
                {
                    return true;
                }

                // baseDir = "baseDir/../"
                baseDir = Path.GetDirectoryName(baseDir);
                if (string.IsNullOrEmpty(baseDir))
                {
                    break;
                }
            }
            while (searchRecursive && Path.IsPathRooted(baseDir));

            return false;
        }
    }
}
