﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Model;
using EffectMaker.Foundation.Model.Types;

using PrimitiveManagerClass = EffectMaker.BusinessLogic.Manager.PrimitiveManager;

namespace EffectMaker.BusinessLogic.GfxToolsUtility
{
    /// <summary>
    /// G3Dプリミティブをバイナリにコンバートします。
    /// データに変更がないときはキャッシュを利用します。
    /// </summary>
    public class GfxPrimitiveBinaryConverter
    {
        #region Singleton Members

        /// <summary>
        /// シングルトンインスタンスです。
        /// </summary>
        private static GfxPrimitiveBinaryConverter singletonInstance;

        /// <summary>
        /// シングルトンインスタンスを取得します。
        /// </summary>
        public static GfxPrimitiveBinaryConverter Instance
        {
            get
            {
                if (singletonInstance == null)
                {
                    singletonInstance = new GfxPrimitiveBinaryConverter();
                }

                return singletonInstance;
            }
        }

        #endregion

        /// <summary>
        /// バイナリキャッシュです。
        /// キャッシュ情報はエミッタセットごとに作成します。
        /// </summary>
        private Dictionary<Guid, BinaryCacheInfo> binaryCache = new Dictionary<Guid, BinaryCacheInfo>();

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        private GfxPrimitiveBinaryConverter()
        {
#if false
            // 現時点では不要
            // スペック切り替え時にキャッシュをクリアする
            SpecManager.CurrentSpecChanged += (s, e) =>
            {
                this.binaryCache.Clear();
            };
#endif
        }

        /// <summary>
        /// G3Dプリミティブをコンバートしてbfresバイナリを生成します。
        /// コンバート元データに変更がないときはキャッシュを利用してコンバート処理を省略します。
        /// </summary>
        /// <param name="emitterSetGuid">bfresバイナリを利用するエミッタセットのGUID</param>
        /// <param name="primitivePaths">コンバートするプリミティブファイル</param>
        /// <param name="forceToBinarize">キャッシュを利用せずに、強制的にバイナリコンバートを実行するときtrue</param>
        /// <returns>生成したバイナリデータを返します。コンバートに失敗したときはnullを返します。</returns>
        public byte[] Convert(Guid                emitterSetGuid,
                              IEnumerable<string> primitivePaths,
                              bool                forceToBinarize)
        {
            // バイナリキャッシュが利用可能ならキャッシュを返す
            if (forceToBinarize == false)
            {
                string[] sourceFileArray = primitivePaths.ToArray();

                BinaryCacheInfo cacheInfo;
                bool findCache = binaryCache.TryGetValue(emitterSetGuid, out cacheInfo);

                if (findCache)
                {
                    if (IsBinaryCacheValid(cacheInfo, sourceFileArray))
                    {
                        return cacheInfo.BinaryData;
                    }
                    else
                    {
                        binaryCache.Remove(emitterSetGuid);
                    }
                }
            }

            byte[] binaryData = null;

            // コンバートを実行
            {
                MemoryStream memoryStream = new MemoryStream();
                LoadModelResultWithFilePath result = PrimitiveManagerClass.Instance.LoadFmdbModels(primitivePaths, memoryStream);

                // 成功時はバイナリデータを取得する
                if (result.Result == LoadModelResults.Success)
                {
                    binaryData = memoryStream.ToArray();
                }
                // エラー処理を行う
                else
                {
                    switch (result.Result)
                    {
                        case LoadModelResults.UnknowModelType:
                            Logger.Log(LogLevels.Error, "Unknown g3d primtive  type {0} at G3dPrimitiveArrayWriter", result.FilePath);
                            break;
                        case LoadModelResults.FileNotFound:
                            Logger.Log(LogLevels.Error, "Not found {0} at G3dPrimitiveArrayWriter", result.FilePath);
                            break;
                        case LoadModelResults.InvalidFileVersion:
                            Logger.Log(LogLevels.Error, "Invalid file version {0} at G3dPrimitiveArrayWriter", result.FilePath);
                            break;
                        case LoadModelResults.FailedLoadingModel:
                            Logger.Log(LogLevels.Error, "Failed to load {0} at G3dPrimitiveArrayWriter", result.FilePath);
                            break;
                        default:
                            Logger.Log(LogLevels.Error, "G3d primitive load failed! {0} at G3dPrimitiveArrayWriter", result.FilePath);
                            break;
                    }
                }
            }

            if (binaryData == null)
            {
                return null;
            }

            // バイナリデータをキャッシュに追加する
            {
                PrimitiveManagerClass primitiveManager = PrimitiveManagerClass.Instance;

                BinarySourceFileInfo[] sourceFileInfo = primitivePaths.Select(x => new BinarySourceFileInfo()
                {
                    FilePath = x,
                    ModifyTime = primitiveManager.GetLoadedFileModifyTime(x)
                }).ToArray();

                BinaryCacheInfo cacheInfo = new BinaryCacheInfo()
                {
                    EmitterSetGuid = emitterSetGuid,
                    SourceFiles = sourceFileInfo,
                    BinaryData = binaryData
                };

                binaryCache[emitterSetGuid] = cacheInfo;
            }

            return binaryData;
        }

        /// <summary>
        /// バイナリキャッシュが利用可能かどうかチェックします。
        /// </summary>
        /// <param name="cacheInfo">バイナリキャッシュ情報</param>
        /// <param name="sourceFiles">生成元となるファイル</param>
        /// <returns>バイナリキャッシュが利用可能であればtrue、それ以外はfalseを返します。</returns>
        private bool IsBinaryCacheValid(BinaryCacheInfo cacheInfo, string[] sourceFiles)
        {
            // 生成元となるファイルの増減をチェック
            if (cacheInfo.SourceFiles.Length != sourceFiles.Length)
            {
                return false;
            }

            // 生成元となるファイルに差異がないかチェック
            // ファイルの順番が入れ替わった場合もキャッシュは無効となる
            for (int i = 0; i < cacheInfo.SourceFiles.Length; ++i)
            {
                // ファイルパスを比較
                if (cacheInfo.SourceFiles[i].FilePath != sourceFiles[i])
                {
                    return false;
                }

                DateTime modifyTime = File.GetLastWriteTime(cacheInfo.SourceFiles[i].FilePath);

                // ファイルの変更日時を比較
                if (modifyTime != cacheInfo.SourceFiles[i].ModifyTime)
                {
                    return false;
                }
            }

            return true;
        }

        /// <summary>
        /// エミッタセットに関連付けられたキャッシュを削除します。
        /// </summary>
        /// <param name="emitterSetGuid">エミッタセットのGUID</param>
        public void RemoveCache(Guid emitterSetGuid)
        {
            this.binaryCache.Remove(emitterSetGuid);
        }

        /// <summary>
        /// バイナリキャッシュ情報です。
        /// </summary>
        private class BinaryCacheInfo
        {
            /// <summary>
            /// エミッタセットのGUIDを取得または設定します。
            /// </summary>
            public Guid EmitterSetGuid { get; set; }

            /// <summary>
            /// バイナリの生成元となるファイルの情報を取得または設定します。
            /// </summary>
            public BinarySourceFileInfo[] SourceFiles { get; set; }

            /// <summary>
            /// バイナリデータを取得または設定します。
            /// </summary>
            public byte[] BinaryData { get; set; }
        }

        /// <summary>
        /// バイナリの生成元となるファイルの情報です。
        /// </summary>
        private class BinarySourceFileInfo
        {
            /// <summary>
            /// ファイルパスを取得または設定します。
            /// </summary>
            public string FilePath { get; set; }

            /// <summary>
            /// ファイルの変更日時を取得または設定します。
            /// </summary>
            public DateTime ModifyTime { get; set; }
        }
    }
}
