﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using EffectMaker.BusinessLogic.BinaryHeaders;
using EffectMaker.BusinessLogic.DataModelOperation;
using EffectMaker.BusinessLogic.GfxToolsUtility;
using EffectMaker.BusinessLogic.Properties;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Texture;

namespace EffectMaker.BusinessLogic.BinaryResourceWriters.Texture
{
    /// <summary>
    /// GFX版のテクスチャアレイライタです。
    /// </summary>
    public class GfxTextureArrayWriter : IBinaryResourceWriter
    {
        /// <summary>
        /// 処理対象のエミッタセットデータです。
        /// </summary>
        private EmitterSetData[] emitterSets;

        /// <summary>
        /// テクスチャデータの初期化に失敗したかどうか。
        /// </summary>
        private bool initializeFailed;

        /// <summary>
        /// テクスチャバイナリとして出力するbntxファイルのパス
        /// </summary>
        private string outputPath;

        /// <summary>
        /// テクスチャバイナリをエフェクトバイナリとは別ファイルに出力する場合はtrue
        /// エフェクトバイナリに含める場合は場合はfalse
        /// </summary>
        private bool outputBinaryFile;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="emitterSets">エミッタセットリスト</param>
        /// <param name="outputFilePath">出力バイナリファイルパス</param>
        public GfxTextureArrayWriter(IEnumerable<EmitterSetData> emitterSets,
                                     string outputFilePath,
                                     bool forceToWriteAllTexturesToNameTable)
        {
            this.emitterSets = emitterSets.ToArray();

            // Prepare the texture array data model.
            // We need the data model to select binary header later.
            var textureArray = new GfxTextureArrayData();

            // First collect all the texture paths in the emitter sets.
            var textures = this.emitterSets.SelectMany(eset =>
                eset.EnumerateEmitterNameAndTexturePaths()
                    .Select(x => new Tuple<EmitterSetData, string, string>(eset, x.Key, x.Value)));

            // テクスチャファイルが存在するかチェック
            foreach (Tuple<EmitterSetData, string, string> data in textures)
            {
                if (File.Exists(data.Item3) == false)
                {
                    Logger.Log(
                        LogLevels.Warning,
                        "Texture \"{0} / {1} / {2}\" is missing. (eset file at {3})",
                        data.Item1.Name,
                        data.Item2,
                        data.Item3,
                        data.Item1.FilePath);
                }
            }

            // Convert all the texture paths to lower case and discard repeated ones.
            var texturePaths = textures.Select(data => data.Item3).Distinct();

            // Add all the distinct texture paths to the texture array data.
            this.TextureCount = 0;
            foreach (string path in texturePaths)
            {
                // ファイルの有無はチェック済みなので簡単なエラー処理だけ行う
                if (File.Exists(path) == false)
                {
                    this.initializeFailed = true;
                    continue;
                }

                string lowerCasePath = path.ToLower();
                LoadTextureResult result = Manager.TextureManager.Instance.LoadTexture(path, true);

                if (result.ResultCode != LoadTextureResultCode.Success)
                {
                    switch (result.ResultCode)
                    {
                        case LoadTextureResultCode.UnknowTextureType:
                            Logger.Log(LogLevels.Error, "Unknown texture type {0} at GfxTextureArrayWriter", lowerCasePath);
                            break;
                        case LoadTextureResultCode.FileNotFound:
                            Logger.Log(LogLevels.Error, "Not found {0} at GfxTextureArrayWriter", lowerCasePath);
                            break;
                        case LoadTextureResultCode.FailedLoadingTexture:
                            Logger.Log(LogLevels.Error, "Failed to load {0} at GfxTextureArrayWriter", lowerCasePath);
                            break;
                        case LoadTextureResultCode.InvalidFileVersion:
                            Logger.Log(LogLevels.Error, "Invalid file version {0} at GfxTextureArrayWriter", lowerCasePath);
                            break;
                    }

                    this.initializeFailed = true;
                    continue;
                }

                ulong guid = Manager.TextureManager.Instance.GetGuid(path);
                if (Manager.TextureManager.Instance.IsLocatedOnAnotherBinary(guid))
                {
                    // --resident-texture-listオプション対応
                    if (forceToWriteAllTexturesToNameTable == true)
                    {
                        // TextureConverter.exeは、各テクスチャのkeyをファイル名からそのまま作ります。
                        // EffectMakerでは、テクスチャのkeyを小文字化したファイル名から作っていますが、
                        // TextureConverter.exeと改行形式のテクスチャリストを共有するために、ファイル名からそのままkeyを作ります。
                        textureArray.TableData.AddTexturePath(path);
                    }

                    // 別バイナリにあるテクスチャはスキップ
                    continue;
                }

                textureArray.AddTexturePath(lowerCasePath);

                ++this.TextureCount;
            }

            this.DataModel = textureArray;
            this.Position = -1;
            this.Size = 0;
            this.Offset = 0;

            // ファイルパスが渡された場合は、バイナリを別ファイルで出力するとみなす
            if (string.IsNullOrEmpty(outputFilePath) == false)
            {
                this.outputPath = outputFilePath;
                this.outputBinaryFile = true;
            }
            else
            {
                this.outputPath = Path.Combine(TextureConverterExecuter.Instance.IntermediateDirectoryPath, "output.bntx");
                this.outputBinaryFile = false;
            }
        }

        /// <summary>
        /// Get the data model the writer is writing.
        /// </summary>
        public DataModelBase DataModel { get; private set; }

        /// <summary>
        /// Get the start position of the texture resource in the stream.
        /// </summary>
        public long Position { get; private set; }

        /// <summary>
        /// Get the size of the written data.
        /// </summary>
        public long Size { get; private set; }

        /// <summary>
        /// Get the offset between the binary header and
        /// the beginning of the binary resource data.
        /// </summary>
        public long Offset { get; private set; }

        /// <summary>
        /// Get the number of textures that is to be written.
        /// </summary>
        public int TextureCount { get; private set; }

        /// <summary>
        /// Write data to the stream in the given context.
        /// </summary>
        /// <param name="context">The binary resource writer context.</param>
        /// <returns>True on success.</returns>
        public bool Write(BinaryResourceWriterContext context)
        {
            Stream stream = context.Stream;
            if (stream == null)
            {
                return false;
            }

            var textureArray = this.DataModel as GfxTextureArrayData;
            if (textureArray == null)
            {
                return false;
            }

            // Save the start position in the stream.
            this.Position = stream.Position;

            // Write an empty binary header for place holder,
            // we will come back and fill in the correct values later.
            BinaryStructHeader.Empty.Write(stream);

            // 初期化の時点で失敗していたときはエラーにする
            if (this.initializeFailed)
            {
                return false;
            }

            if (textureArray.TableData.TexturePaths.Any())
            {
                // テクスチャテーブルを書き込む
                var tableWriter = new GfxTextureTableWriter(textureArray);
                if (tableWriter.Write(context) == false)
                {
                    return false;
                }
            }

            if (textureArray.TexturePaths.Any())
            {
                byte[] bntxData = null;

                // コンバートを実行
                bntxData = GfxTextureBinaryConverter.Instance.Convert(this.emitterSets[0].Guid,
                                                                      textureArray.TexturePaths,
                                                                      this.outputPath,
                                                                      Options.OptionStore.RuntimeOptions.IsCommandLineMode);

                if (bntxData == null)
                {
                    return false;
                }

                // テクスチャバイナリからアラインサイズを取得
                // nn::util::BinaryFileHeaderのアラインサイズを取り出す
                const int alignPos = 14;
                long alignment = 1;

                // 巨大なメモリ領域を取得してしまわないように64MB程度を上限とする
                if (bntxData.Length > alignPos && bntxData[alignPos] <= 26)
                {
                    alignment = 1 << bntxData[alignPos];
                    context.UpdateAlignmentSize((uint)alignment);
                }
                else
                {
                    Logger.Log("Console", LogLevels.Error, "Texture alignment size is too large.");
                    return false;
                }

                // テクスチャバイナリをファイルに出力する場合はエフェクトバイナリへの書き込みをスキップする
                if (this.outputBinaryFile == false)
                {
                    // Compute alignment(テーブルを書き込んだ上でのバイナリ先頭からのアラインメント).
                    long paddingSize = stream.Position % alignment;
                    if (paddingSize > 0)
                    {
                        paddingSize = alignment - paddingSize;
                    }

                    // Write alignment with zeroed byte array.
                    stream.Write(
                        Enumerable.Repeat((byte)0, (int)paddingSize).ToArray(),
                        0,
                        (int)paddingSize);

                    // BNTX本体のオフセットを再計算
                    this.Offset = stream.Position - this.Position - BinaryStructHeader.Size;

                    // テクスチャバイナリを書き込む
                    stream.Write(bntxData, 0, bntxData.Length);
                }
                else
                {
                    // エフェクトバイナリに何も追加しないので、オフセットを無効値である0xFFFFFFFFにする
                    this.Offset = 0xFFFFFFFF - BinaryStructHeader.Size;
                }
            }
            else
            {
                // 何もバイナリがない場合は、オフセットを無効値である0xFFFFFFFFにする
                this.Offset = 0xFFFFFFFF - BinaryStructHeader.Size;
            }

            // Save the size of the binary data.
            this.Size = stream.Position - this.Position;

            // Add this writer to the context, the context will write the binary header
            // for the added binary writers.
            context.AddBinaryWriter(this);

            return true;
        }
    }
}
