﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Runtime.CompilerServices;
using EffectMaker.BusinessLogic.BinaryHeaders;
using EffectMaker.BusinessLogic.BinaryHeaders.Helpers;
using EffectMaker.BusinessLogic.SpecDefinitions;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.Foundation.Model;
using EffectMaker.Foundation.Model.Types;
using EffectMaker.Foundation.Primitives;
using EffectMaker.Foundation.Utility;

using MyPrimitiveManager = EffectMaker.BusinessLogic.Manager.PrimitiveManager;

namespace EffectMaker.BusinessLogic.BinaryResourceWriters.Primitive
{
    /// <summary>
    /// Write primitive resource to binary stream.
    /// </summary>
    public class PrimitiveResourceWriter : IBinaryResourceWriter
    {
        /// <summary>
        /// 頂点バッファのアライメントサイズ(いずれスペックに移す)
        /// </summary>
        private const uint VertexBufferAlignment = 64;

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="data">The primitive resource data.</param>
        public PrimitiveResourceWriter(PrimitiveResourceData data)
        {
            if (data == null)
            {
                throw new ArgumentException("The primitive resource data must be null.");
            }

            this.DataModel = data;
            this.Position = -1;
            this.Size = 0;
            this.Offset = 0;
        }

        /// <summary>
        /// Get the data model the writer is writing.
        /// </summary>
        public DataModelBase DataModel { get; private set; }

        /// <summary>
        /// Get the start position of the texture resource in the stream.
        /// </summary>
        public long Position { get; private set; }

        /// <summary>
        /// Get the size of the written data.
        /// </summary>
        public long Size { get; private set; }

        /// <summary>
        /// Get the offset between the binary header and
        /// the beginning of the binary resource data.
        /// </summary>
        public long Offset { get; private set; }

        /// <summary>
        /// Write data to the stream in the given context.
        /// </summary>
        /// <param name="context">The binary resource writer context.</param>
        /// <returns>True on success.</returns>
        public bool Write(BinaryResourceWriterContext context)
        {
            Stream stream = context.Stream;
            if (stream == null)
            {
                return false;
            }

            var primitiveRes = this.DataModel as PrimitiveResourceData;
            if (primitiveRes == null)
            {
                return false;
            }

            this.Position = stream.Position;

            EffectMaker.Foundation.Model.ModelData data;
            LoadModelResults result =
                MyPrimitiveManager.Instance.LoadModelWithData(
                    primitiveRes.PrimitivePath,
                    true,
                    out data);

            // Failed to find model, bail out.
            if (result != LoadModelResults.Success)
            {
                return false;
            }

            WriteCore(stream, data, MyPrimitiveManager.Instance.GetGuid(primitiveRes.PrimitivePath));

            this.Size = stream.Position - this.Position;
            context.AddBinaryWriter(this);

            // 正常終了.
            return true;
        }

        /// <summary>
        /// プリミティブデータ本体を書き込む.
        /// </summary>
        /// <param name="stream">ストリーム</param>
        /// <param name="data">データ</param>
        /// <param name="uniqueId">ユニークID</param>
        internal static void WriteCore(Stream stream, Foundation.Model.ModelData data, ulong uniqueId)
        {
            BinaryStructHeader.Empty.Write(stream);

            //// 送信データは下記の通り.
            //// u64  uniqueID;         // 識別用ID.
            //// u32  VertexCount;      // 頂点数.
            //// u32  VertexElement;    // 頂点構成数.
            //// u32  NormalCount;      // 法線ベクトル数.
            //// u32  NormalElement;    // 法線ベクトル構成数.
            //// u32  TangentCount;     // 接線ベクトル数.
            //// u32  TangentElement;   // 接線ベクトル構成数.
            //// u32  ColorCount;       // 頂点カラー数.
            //// u32  ColorElement;     // 頂点カラー構成数.
            //// u32  TexCoord0Count;   // テクスチャ座標0数.
            //// u32  TexCoord0Element; // テクスチャ座標0構成数.
            //// u32  TexCoord1Count;   // テクスチャ座標1数.
            //// u32  TexCoord1Element; // テクスチャ座標1構成数.
            //// u32  IndexCount;       // 頂点インデックス数.
            //// u32  pVertexData;      // 頂点配列へのオフセット(必須).
            //// u32  pNormalData;      // 法線配列へのオフセット.
            //// u32  pTangentData;     // 接線配列へのオフセット.
            //// u32  pColorData;       // 頂点カラー配列へのオフセット.
            //// u32  pTexCoordData;    // テクスチャ座標結合配列へのオフセット.
            //// u32  pIndexData;       // インデックス配列へのオフセット.

            // 構造体の先頭を基点としてオフセットを算出するため位置を記憶
            var structTop = (int)stream.Position;

            // 構造体メンバを書き込み.
            BinaryConversionUtility.ForResource.WriteStream(stream, uniqueId);

            // 要素数の情報はVec4強制化前の情報を送る(配列サイズはCountに4掛ければ求まる)
            WriteVertexElementInfo(stream, data.Position);
            WriteVertexElementInfo(stream, data.Normal);
            WriteVertexElementInfo(stream, data.Tangent);
            WriteVertexElementInfo(stream, data.Color);
            WriteVertexElementInfo(stream, data.TexCoord0);
            WriteVertexElementInfo(stream, data.TexCoord1);
            BinaryConversionUtility.ForResource.WriteStream(stream, (uint)data.Indexes.Indexes.Length);

            // オフセット書き込み用の位置を保存して0フィルしておく
            var offsetTop = stream.Position;
            BinaryConversionUtility.ForResource.WriteStream(stream, (uint)0);
            BinaryConversionUtility.ForResource.WriteStream(stream, (uint)0);
            BinaryConversionUtility.ForResource.WriteStream(stream, (uint)0);
            BinaryConversionUtility.ForResource.WriteStream(stream, (uint)0);
            BinaryConversionUtility.ForResource.WriteStream(stream, (uint)0);
            BinaryConversionUtility.ForResource.WriteStream(stream, (uint)0);

            // 8バイトアライン向けに4バイト詰め
            BinaryConversionUtility.ForResource.WriteStream(stream, (uint)0);

            // 構造体の先頭を基点としてオフセットを算出する
            Func<int, int> calcOffset = (p) => p > 0 ? p - structTop : 0;

            // 実データを書き込み.
            int posOffset = calcOffset(WriteVertexElementData(stream, data.Position, new Vector4f(0.0f, 0.0f, 0.0f, 1.0f)));
            int norOffset = calcOffset(WriteVertexElementData(stream, data.Normal, new Vector4f(0.0f, 0.0f, 0.0f, 0.0f)));
            int tanOffset = calcOffset(WriteVertexElementData(stream, data.Tangent, new Vector4f(0.0f, 0.0f, 0.0f, 0.0f)));
            int colOffset = calcOffset(WriteVertexElementData(stream, data.Color, new Vector4f(0.0f, 0.0f, 0.0f, 1.0f)));
            int uvOffset = calcOffset(WriteVertexElementDataCombinedVec2ToVec4(stream, data.TexCoord0, data.TexCoord1));
            int idxOffset = calcOffset(WriteIndexData(stream, data.Indexes));

            // いったんストリームを巻き戻してオフセットを書き込み、末尾に戻す.
            var endOffset = stream.Position;
            stream.Seek(offsetTop, SeekOrigin.Begin);
            BinaryConversionUtility.ForResource.WriteStream(stream, (uint)posOffset);
            BinaryConversionUtility.ForResource.WriteStream(stream, (uint)norOffset);
            BinaryConversionUtility.ForResource.WriteStream(stream, (uint)tanOffset);
            BinaryConversionUtility.ForResource.WriteStream(stream, (uint)colOffset);
            BinaryConversionUtility.ForResource.WriteStream(stream, (uint)uvOffset);
            BinaryConversionUtility.ForResource.WriteStream(stream, (uint)idxOffset);
            stream.Seek(endOffset, SeekOrigin.Begin);
        }

        /// <summary>
        /// 頂点要素情報を書き込む.
        /// </summary>
        /// <param name="stream">ストリーム.</param>
        /// <param name="element">頂点要素</param>
        private static void WriteVertexElementInfo(Stream stream, VertexElement element)
        {
            BinaryConversionUtility.ForResource.WriteStream(stream, (uint)element.Count);
            BinaryConversionUtility.ForResource.WriteStream(stream, (uint)element.Column);
        }

        /// <summary>
        /// 頂点要素データを書き込む.
        /// </summary>
        /// <param name="stream">ストリーム</param>
        /// <param name="element">頂点要素</param>
        /// <param name="defaultValue">Vec4に展開する際の空欄を埋める値</param>
        /// <returns>データの先頭位置(書き込まなかった場合は0)</returns>
        private static int WriteVertexElementData(Stream stream, VertexElement element, Vector4f defaultValue)
        {
            if (element.ElementData.Length <= 0)
            {
                return 0;
            }

            int offset = (int)stream.Position;
            offset += WritePadding(stream);
            int column = element.Column;
            if (column < 4)
            {
                for (int i = 0; i < element.Count; ++i)
                {
                    for (int j = 0; j < 4; ++j)
                    {
                        BinaryConversionUtility.ForResource.WriteStream(
                            stream,
                            j < column ? element.ElementData[i*column + j] : defaultValue[j]);
                    }
                }
            }
            else
            {
                for (int i = 0; i < element.ElementData.Length; ++i)
                {
                    BinaryConversionUtility.ForResource.WriteStream(stream, element.ElementData[i]);
                }
            }

            return offset;
        }

        /// <summary>
        /// UV0,1を結合して書き込む.
        /// </summary>
        /// <param name="stream">ストリーム</param>
        /// <param name="element1">頂点要素1</param>
        /// <param name="element2">頂点要素2</param>
        /// <returns>データの先頭位置(書き込まなかった場合は0)</returns>
        private static int WriteVertexElementDataCombinedVec2ToVec4(Stream stream, VertexElement element1, VertexElement element2)
        {
            int length = Math.Max(element1.ElementData.Length, element2.ElementData.Length);
            if (length <= 0)
            {
                return 0;
            }

            int offset = (int)stream.Position;
            offset += WritePadding(stream);

            for (int i = 0; i < length; i += 2)
            {
                if (i < element1.ElementData.Length)
                {
                    BinaryConversionUtility.ForResource.WriteStream(stream, element1.ElementData[i]);
                    BinaryConversionUtility.ForResource.WriteStream(stream, element1.ElementData[i + 1]);
                }
                else
                {
                    BinaryConversionUtility.ForResource.WriteStream(stream, 0.0f);
                    BinaryConversionUtility.ForResource.WriteStream(stream, 0.0f);
                }

                if (i < element2.ElementData.Length)
                {
                    BinaryConversionUtility.ForResource.WriteStream(stream, element2.ElementData[i]);
                    BinaryConversionUtility.ForResource.WriteStream(stream, element2.ElementData[i + 1]);
                }
                else
                {
                    BinaryConversionUtility.ForResource.WriteStream(stream, 0.0f);
                    BinaryConversionUtility.ForResource.WriteStream(stream, 0.0f);
                }
            }

            return offset;
        }

        /// <summary>
        /// インデックスバッファを書き込む.
        /// </summary>
        /// <param name="stream">ストリーム.</param>
        /// <param name="indexBuffer">インデックスバッファ.</param>
        /// <returns>データの先頭位置(書き込まなかった場合は0)</returns>
        private static int WriteIndexData(Stream stream, IndexBuffer indexBuffer)
        {
            if (indexBuffer.Indexes.Length <= 0)
            {
                return 0;
            }

            int offset = (int)stream.Position;
            offset += WritePadding(stream);
            for (int i = 0; i < indexBuffer.Indexes.Length; ++i)
            {
                BinaryConversionUtility.ForResource.WriteStream(stream, indexBuffer.Indexes[i]);
            }

            return offset;
        }

        /// <summary>
        /// 頂点バッファのアライメントを書き込み、そのサイズを返します。
        /// </summary>
        /// <param name="stream">ストリーム</param>
        /// <returns>パディングサイズ</returns>
        private static int WritePadding(Stream stream)
        {
            Func<int, int, int> calcPadding = (pos, align) =>
                align <= 0 || pos % align == 0 ? 0 : align - (pos % align);
            int size = calcPadding((int)stream.Position, (int)VertexBufferAlignment);
            if (size > 0)
            {
                var alignmentBytes = Enumerable.Repeat((byte)0xCC, size).ToArray();
                stream.Write(alignmentBytes, 0, size);
            }

            return size;
        }
    }
}
