﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using EffectMaker.BusinessLogic.BinaryHeaders;
using EffectMaker.BusinessLogic.Manager;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.Foundation.Log;

namespace EffectMaker.BusinessLogic.BinaryResourceWriters.Primitive
{
    /// <summary>
    /// Write model resource to binary stream.
    /// </summary>
    public class ModelPrimitiveResourceWriter : IBinaryResourceWriter
    {
        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="data">The texture binary data to write.</param>
        public ModelPrimitiveResourceWriter(ModelData data)
        {
            if (data == null)
            {
                throw new ArgumentException("The model data must not be null.");
            }

            // Prepare the model primitive resource data model.
            // We need the data model to select binary header later.
            ModelPrimitiveResourceData primitiveResource = new ModelPrimitiveResourceData();
            primitiveResource.PrimitiveId = data.ModelBinaryId;

            this.DataModel = primitiveResource;
            this.Position = -1;
            this.Size = 0;
            this.Offset = 0;
        }

        /// <summary>
        /// Get the data model the writer is writing.
        /// </summary>
        public DataModelBase DataModel { get; private set; }

        /// <summary>
        /// Get the start position of the texture resource in the stream.
        /// </summary>
        public long Position { get; private set; }

        /// <summary>
        /// Get the size of the written data.
        /// </summary>
        public long Size { get; private set; }

        /// <summary>
        /// Get the offset between the binary header and
        /// the beginning of the binary resource data.
        /// </summary>
        public long Offset { get; private set; }

        /// <summary>
        /// Get the flag indicating whether should set the next pointer
        /// to the end of the binary data.
        /// </summary>
        public bool ShouldSetNextToEnd
        {
            get { return true; }
        }

        /// <summary>
        /// Write data to the stream in the given context.
        /// </summary>
        /// <param name="context">The binary resource writer context.</param>
        /// <returns>True on success.</returns>
        public bool Write(BinaryResourceWriterContext context)
        {
            var data = (ModelPrimitiveResourceData)this.DataModel;

            Stream stream = context.Stream;
            if (stream == null)
            {
                return false;
            }

            // Save the start position in the stream.
            this.Position = stream.Position;

            // モデルバイナリを取得
            byte[] modelBinary = ModelInfoManager.GetModelBinary(data.PrimitiveId);
            if (modelBinary == null)
            {
                Logger.Log(LogLevels.Warning, "ModelBinaryWriter.Write : Failed to get the model binary.");
                return false;
            }

            BinaryStructHeader.Empty.Write(stream);

            // アラインはランタイム側で調整するので不要

            // Write model binary data.
            stream.Write(modelBinary, 0, modelBinary.Length);

            // Set the size.
            this.Size = stream.Position - this.Position;

            // Add this writer to the context, the context will write the binary header
            // for the added binary writers.
            context.AddBinaryWriter(this);

            return true;
        }
    }
}
