﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.CodeDom;
using System.Collections.Generic;
using System.IO;
using System.Linq;

using EffectMaker.BusinessLogic.BinaryHeaders;
using EffectMaker.BusinessLogic.BinaryHeaders.Helpers;

using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Texture;

namespace EffectMaker.BusinessLogic.BinaryResourceWriters
{
    /// <summary>
    /// Data class that holds all the data needed for writing binary resources.
    /// </summary>
    public class BinaryResourceWriterContext : IDisposable
    {
        /// <summary>The dictionary maps the child writers and their source data model.</summary>
        private Dictionary<DataModelBase, IBinaryResourceWriter> writerMap =
            new Dictionary<DataModelBase, IBinaryResourceWriter>();

        /// <summary>The alignment to the next binary block.</summary>
        private uint alignmentToNext;

        /// <summary>
        /// 最後のバイナリリソースなのか指定します。
        /// </summary>
        private bool lastBinaryResource;

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="stream">The stream.</param>
        /// <param name="alignmentToNext">The alignment to the next binary block.</param>
        /// <param name="lastBinaryResource">最後のバイナリリソースならtrue, そうでないならfalse</param>
        public BinaryResourceWriterContext(Stream stream, uint alignmentToNext, bool lastBinaryResource)
        {
            // The stream must be seekable.
            if (stream == null || stream.CanSeek == false)
            {
                throw new ArgumentException("The given stream is not seekable.");
            }

            this.Stream = stream;
            this.alignmentToNext = alignmentToNext;
            this.lastBinaryResource = lastBinaryResource;

            // Save the start position for later. We need it to compute offsets.
            this.StartPosition = stream.Length;
            this.Stream.Seek(0, SeekOrigin.End);
        }

        /// <summary>
        /// Get the stream to write to.
        /// </summary>
        public Stream Stream { get; private set; }

        /// <summary>
        /// Get the start position in the stream.
        /// </summary>
        public long StartPosition { get; private set; }

        /// <summary>
        /// このコンテキスト内で調整したアライメントの最大サイズ（2の指数）を取得します。
        /// </summary>
        public byte AdjustedAlignment { get; private set; }

        /// <summary>
        /// Dispose this instance.
        /// </summary>
        public void Dispose()
        {
            this.WriteBinaryHeaders();
        }

        /// <summary>
        /// 子バイナリ要素の個数を調べて返します。
        /// </summary>
        /// <param name="helper">個数を調べたい親バイナリのヘルパー</param>
        /// <param name="data">個数を調べたい親バイナリに対応するデータモデル</param>
        /// <returns>子バイナリ要素の個数</returns>
        public static ushort GetChildNum(IBinaryHeaderHelper helper, DataModelBase data)
        {
            var childData = helper.GetChild(data);
            if (childData == null)
            {
                return 0;
            }

            var childHelper = BinaryHeaderHelperSelector.GetHelper(childData);
            if (childHelper == null)
            {
                return 0;
            }

            ushort count = 0;
            while (childData != null)
            {
                ++count;
                childData = childHelper.GetNext(childData);
            }

            return count;
        }

        /// <summary>
        /// Add a binary resource writer to the map.
        /// </summary>
        /// <param name="writer">The binary resource writer.</param>
        public void AddBinaryWriter(IBinaryResourceWriter writer)
        {
            this.writerMap.Add(writer.DataModel, writer);
        }

        /// <summary>
        /// Write binary header for the binary writers.
        /// </summary>
        /// <returns>True on success.</returns>
        public bool WriteBinaryHeaders()
        {
            var header = new BinaryStructHeader();
            foreach (IBinaryResourceWriter writer in this.writerMap.Values)
            {
                header.Reset();

                // Do null-check.
                DataModelBase data = writer.DataModel;
                if (data == null)
                {
                    Logger.Log(
                        LogLevels.Debug,
                        "BinaryResourceWriterContext.WriteBinaryHeaders : Unable to write binary header for a null data model.");

                    return false;
                }

                // Get binary header helper for the data model.
                IBinaryHeaderHelper helper = BinaryHeaderHelperSelector.GetHelper(data);
                if (helper == null)
                {
                    Logger.Log(
                        LogLevels.Debug,
                        "BinaryResourceWriterContext.WriteBinaryHeaders : Failed getting binary header helper for the data model {0}.",
                        data.GetType().Name);

                    return false;
                }

                // Set up the binary header.
                header.Tag = helper.GetTag(data);
                header.Child = this.ComputeWriterOffset(helper.GetChild(data), writer.Position);
                header.Sub = this.ComputeWriterOffset(helper.GetSub(data), writer.Position);
                header.Offset = (uint)writer.Offset + BinaryStructHeader.Size;
                header.ChildNum = GetChildNum(helper, data);

                if ( header.Offset == 0xFFFFFFFF )
                {
                    // オフセットが0xFFFFFFFFの場合は、バイナリが存在しないのでバイナリサイズは0に
                    header.BinarySize = 0;
                }
                else
                {
                    // ここで、バイナリ本体サイズを計算している。
                    //
                    // IBinaryResourceWriter.Sizeは、ヘッダの先頭からバイナリ本体の終端までの距離になる。
                    // BinaryStructHeader.BinarySizeは、バイナリ本体のサイズが入るので計算し直す必要がある。
                    // BinaryStructHeader.Offsetは、ヘッダの先頭からバイナリ本体の先頭までの距離である。
                    // よって、writer.Sizeからheader.Offsetを引けば、バイナリ本体のサイズが求まる。
                    header.BinarySize = (uint)writer.Size - header.Offset;
                }

                if (helper.ShouldSetNextToEnd == true)
                {
                    // 後ろに続くバイナリリソースがある場合は、Nextに次のブロックの位置を書き込んでおく。
                    if (lastBinaryResource == false)
                    {
                        uint end = (uint)(writer.Position + writer.Size);

                        // Compute alignment to the next block.
                        uint alignment = 0;
                        if (this.alignmentToNext > 0)
                        {
                            alignment = end % this.alignmentToNext;
                            if (alignment != 0)
                            {
                                alignment = this.alignmentToNext - alignment;
                            }
                        }

                        header.Next = (uint)writer.Size + alignment;
                    }
                    else
                    {
                        // 最後のバイナリリソースなら、後ろに続くブロックはない
                        header.Next = 0xFFFFFFFF;
                    }
                }
                else
                {
                    header.Next = this.ComputeWriterOffset(helper.GetNext(data), writer.Position);
                }

                // Set stream position to the place holder the writer put
                // while they write data to the stream.
                this.Stream.Seek(writer.Position, SeekOrigin.Begin);

                // Write the header.
                header.Write(this.Stream);
            }

            return true;
        }

        /// <summary>
        /// このコンテキスト内における最大のアライメントサイズを更新します。
        /// </summary>
        /// <param name="size">アライメントサイズ（バイト）</param>
        public void UpdateAlignmentSize(uint size)
        {
            this.AdjustedAlignment = Math.Max((byte)Math.Log(size, 2.0), this.AdjustedAlignment);
        }

        /// <summary>
        /// Find the writer of the given data model from the binary writer map.
        /// </summary>
        /// <param name="dataModel">The data model.</param>
        /// <returns>The binary writer for the data model or null if not found.</returns>
        private IBinaryResourceWriter FindWriter(DataModelBase dataModel)
        {
            if (dataModel == null)
            {
                return null;
            }

            IBinaryResourceWriter writer;
            if (this.writerMap.TryGetValue(dataModel, out writer) == false)
            {
                return null;
            }

            return writer;
        }

        /// <summary>
        /// Find the writer of the given data model from the binary writer map and compute the
        /// offset of the writer's data with the specified base position.
        /// </summary>
        /// <param name="dataModel">The data model.</param>
        /// <param name="basePosition">The base position to compute the writer offset from.</param>
        /// <returns>The offset from the base position.</returns>
        private uint ComputeWriterOffset(DataModelBase dataModel, long basePosition)
        {
            IBinaryResourceWriter writer = this.FindWriter(dataModel);
            if (writer == null)
            {
                return 0xFFFFFFFF;
            }

            return (uint)(writer.Position - basePosition);
        }
    }
}
