﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Collections.Generic;
using System.Linq;
using EffectMaker.BusinessLogic.SpecDefinitions;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.Foundation.Log;

namespace EffectMaker.BusinessLogic.BinaryHeaders.Helpers
{
    /// <summary>
    /// Binary header helper for emitter binary structures.
    /// </summary>
    public class EmitterBinaryHeaderHelper : IBinaryHeaderHelper
    {
        /// <summary>
        /// Constructor.
        /// </summary>
        public EmitterBinaryHeaderHelper()
        {
        }

        /// <summary>
        /// Get the flag indicating whether should set the next pointer
        /// to the end of the binary data.
        /// </summary>
        public bool ShouldSetNextToEnd
        {
            get { return false; }
        }

        /// <summary>
        /// Get the alignment inserts before the next element.
        /// </summary>
        public uint AlignmentToNext
        {
            get { return 0; }
        }

        /// <summary>
        /// Check if the given data model is accepted for this binary header helper.
        /// </summary>
        /// <param name="dataModel">The data model.</param>
        /// <returns>True if accepted.</returns>
        public bool IsAccepted(DataModelBase dataModel)
        {
            if (dataModel is EmitterData)
            {
                return true;
            }

            return false;
        }

        /// <summary>
        /// Get tag string for the binary header.
        /// </summary>
        /// <param name="dataModel">The data model.</param>
        /// <returns>The tag string.</returns>
        public string GetTag(DataModelBase dataModel)
        {
            return "EMTR";
        }

        /// <summary>
        /// Getting the next of the given data model.
        /// </summary>
        /// <param name="current">The current data model.</param>
        /// <returns>The next data model or null if the given is the last.</returns>
        public DataModelBase GetNext(DataModelBase current)
        {
            // エミッタを取得
            var emitter = current as EmitterData;
            if (emitter == null)
            {
                Logger.Log(LogLevels.Debug, "currentが不正");
                return null;
            }

            IEnumerable<EmitterData> list = null;

            // 親データが持っているエミッタのリストを取得
            if (emitter.Parent is EmitterData)
            {
                var parent = (EmitterData)emitter.Parent;
                list = parent.ActiveEmitterList;
            }
            else if (emitter.Parent is EmitterSetData)
            {
                var parent = (EmitterSetData)emitter.Parent;
                list = parent.ActiveEmitterList;
            }

            if (list == null)
            {
                Logger.Log(LogLevels.Debug, "ActiveEmitterListが不正");
                return null;
            }

            // エミッタのインデックスを取得
            int index = 0;
            foreach (EmitterData item in list)
            {
                if (item == emitter)
                {
                    break;
                }

                ++index;
            }

            // 次のエミッタがあればそれを返す
            if (index < list.Count() - 1)
            {
                return list.ElementAt(index + 1);
            }

            return null;
        }

        /// <summary>
        /// Getting the first child of the given data model.
        /// </summary>
        /// <param name="current">The current data model.</param>
        /// <returns>The first child data model or null if the child does not exist.</returns>
        public DataModelBase GetChild(DataModelBase current)
        {
            var emitter = current as EmitterData;
            if (emitter != null &&
                emitter.AllActiveChildEmitters.Count() > 0)
            {
                return emitter.AllActiveChildEmitters.ElementAt(0);
            }

            return null;
        }

        /// <summary>
        /// Getting the first data model of the given data model's sub tree.
        /// </summary>
        /// <param name="current">The current data model.</param>
        /// <returns>The first data model of the sub tree or null there is no sub tree.</returns>
        public DataModelBase GetSub(DataModelBase current)
        {
            var emitter = current as EmitterData;
            if (emitter != null)
            {
                if (emitter.EmitterCustomShaderData.SelectedSettingData != null)
                {
                    return emitter.EmitterCustomShaderData.SelectedSettingData;
                }
                else if (emitter.ActiveCustomActionData != null)
                {
                    return emitter.ActiveCustomActionData;
                }
                else if (emitter.SelectedEmitterExtParams != null)
                {
                    return emitter.SelectedEmitterExtParams;
                }
                else if (emitter.ActiveReservedShader != null)
                {
                    return emitter.ActiveReservedShader;
                }
                else if (emitter.ActiveFieldList != null &&
                         emitter.ActiveFieldList.Any())
                {
                    return emitter.ActiveFieldList.ElementAt(0);
                }
                else
                {
                    return EmitterAnimationBinaryHeaderHelper.GetFirstEnableAnime(emitter);
                }
            }

            return null;
        }

        /// <summary>
        /// Getting the offset from the binary header to the binary data.
        /// </summary>
        /// <param name="current">The current data model.</param>
        /// <param name="headerPos">The position of the binary header.</param>
        /// <returns>The offset from the binary header to the binary data.</returns>
        public uint GetOffset(DataModelBase current, uint headerPos)
        {
            // Compute alignment to the emitter binary data.
            uint alignment =
                (headerPos + BinaryStructHeader.Size) % (uint)SpecManager.CurrentSpec.ShaderConversionOption.UboAlignment;
            if (alignment != 0)
            {
                alignment = (uint)SpecManager.CurrentSpec.ShaderConversionOption.UboAlignment - alignment;
            }

            return BinaryStructHeader.Size + alignment;
        }
    }
}
