﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System.Collections.Generic;
using System.Linq;
using EffectMaker.DataModel.DataModels;
using EffectMaker.Foundation.Log;

namespace EffectMaker.BusinessLogic.BinaryHeaders.Helpers
{
    /// <summary>
    /// Selects binary header helper for the specified data model.
    /// </summary>
    public static class BinaryHeaderHelperSelector
    {
        /// <summary>The list of registered binary header helpers.</summary>
        private static List<IBinaryHeaderHelper> registeredBinaryHeaderHelpers =
            new List<IBinaryHeaderHelper>();

        /// <summary>
        /// Static constructor.
        /// </summary>
        static BinaryHeaderHelperSelector()
        {
            RegisterHelper(new EmitterBinaryHeaderHelper());
            RegisterHelper(new CustomShaderBinaryHeaderHelper());
            RegisterHelper(new CustomActionBinaryHeaderHelper());
            RegisterHelper(new EmitterExtParamsBinaryHeaderHelper());
            RegisterHelper(new FieldBinaryHeaderHelper());
            RegisterHelper(new ReservedShaderBinaryHeaderHelper());
            RegisterHelper(new StripeBinaryHeaderHelper());
            RegisterHelper(new EmitterSetBinaryHeaderHelper());
            RegisterHelper(new TextureArrayBinaryHeaderHelper());
            RegisterHelper(new TextureResourceBinaryHeaderHelper());
            RegisterHelper(new TextureBinaryBinaryHeaderHelper());
            RegisterHelper(new GfxTextureArrayBinaryHeaderHelper());
            RegisterHelper(new GfxTextureTableBinaryHeaderHelper());
            RegisterHelper(new PrimitiveArrayBinaryHeaderHelper());
            RegisterHelper(new PrimitiveResourceBinaryHeaderHelper());
            RegisterHelper(new TrimmingPrimitiveResourceBinaryHeaderHelper());
            RegisterHelper(new ShaderArrayBinaryHeaderHelper());
            RegisterHelper(new ShaderResourceBinaryHeaderHelper());
            RegisterHelper(new PreviewBinaryHeaderHelper());
            RegisterHelper(new EmitterAnimationBinaryHeaderHelper());
            RegisterHelper(new ModelBinaryHeaderHelper());
            RegisterHelper(new ModelPrimitiveResourceBinaryHeaderHelper());
            RegisterHelper(new ViewerBinaryHeaderHelper());
            RegisterHelper(new ViewerPtclBinaryHeaderHelper());
            RegisterHelper(new ViewerTextureResourceBinaryHeaderHelper());
            RegisterHelper(new G3dPrimitiveArrayBinaryHeaderHelper());
            RegisterHelper(new G3dPrimitiveTableBinaryHeaderHelper());
            RegisterHelper(new EmitterSetFilePathArrayBinaryHeaderHelper());
        }

        /// <summary>
        /// Choose a binary header helper for the given data model.
        /// </summary>
        /// <param name="dataModel">The data model.</param>
        /// <returns>The binary header helper or null if the data model is not valid.</returns>
        public static IBinaryHeaderHelper GetHelper(DataModelBase dataModel)
        {
            IBinaryHeaderHelper helper =
                registeredBinaryHeaderHelpers.FirstOrDefault(hp => hp.IsAccepted(dataModel));

            if (helper == null)
            {
                Logger.Log(
                    LogLevels.Debug,
                    "BinaryHeaderHelperSelector.GetHelper : A binary header helper is requested with {0} but no available helper are registered.",
                    dataModel.GetType().Name);
                return null;
            }

            return helper;
        }

        /// <summary>
        /// Register binary header helper.
        /// </summary>
        /// <param name="helper">The binary header helper instance.</param>
        private static void RegisterHelper(IBinaryHeaderHelper helper)
        {
            if (helper == null)
            {
                return;
            }

            registeredBinaryHeaderHelpers.Add(helper);
        }
    }
}
