﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Runtime.InteropServices;
using System.Text;

namespace EffectMaker.Foundation.Utility
{
    /// <summary>
    /// 型変換ユーティリティ
    /// </summary>
    public static class TypeConversionUtility
    {
        /// <summary>
        /// Check if the value can be converted to the target type.
        /// </summary>
        /// <param name="targetType">The target type.</param>
        /// <param name="value">The value.</param>
        /// <returns>True if the valu can be converted to the target type.</returns>
        public static bool CanConvert(Type targetType, object value)
        {
            if (value == null)
            {
                return targetType.IsValueType == false || Nullable.GetUnderlyingType(targetType) != null;
            }
            else
            {
                return TypeConversionUtility.TryConvert(value.GetType(), targetType, ref value);
            }
        }

        /// <summary>
        /// Tries to canvert the value from a type to another.
        /// </summary>
        /// <param name="sourceType">The type of the value.</param>
        /// <param name="targetType">The whished new type of the value.</param>
        /// <param name="value">The value to convert.</param>
        /// <returns>Returns true if the value could be converted, false otherwise.</returns>
        public static bool TryConvert(Type sourceType, Type targetType, ref object value)
        {
            if (targetType.IsAssignableFrom(sourceType))
            {
                return true;
            }

            var converter = TypeDescriptor.GetConverter(targetType);
            if (converter != null && converter.CanConvertFrom(sourceType))
            {
                try
                {
                    value = converter.ConvertFrom(value);
                    return true;
                }
                catch
                {
                    // if conversion failed, keep going to try the next one
                }
            }

            converter = TypeDescriptor.GetConverter(sourceType);
            if (converter != null && converter.CanConvertTo(targetType))
            {
                try
                {
                    value = converter.ConvertTo(value, targetType);
                    return true;
                }
                catch
                {
                    // if conversion failed, it's really over and return false
                }
            }

            return false;
        }
    }
}
