﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Drawing.Imaging;
using System.Threading.Tasks;
using EffectMaker.Foundation.Texture;

namespace EffectMaker.Foundation.Utility
{
    /// <summary>
    /// Render utilities.
    /// </summary>
    public class RenderUtility
    {
        /// <summary>
        /// Do view rectangle clipping for the given line segment.
        /// </summary>
        /// <param name="viewRect">The view rectangle.</param>
        /// <param name="p1">An end point of the line segment.</param>
        /// <param name="p2">The other end point of the line segment.</param>
        /// <param name="clippedP1">The clipped end point of the line segment.</param>
        /// <param name="clippedP2">The other clipped end point of the line segment.</param>
        /// <returns>True if the line segment is fully or partially in the view rectangle.</returns>
        public static bool ClipLineSegment(
            Rectangle viewRect,
            PointF p1,
            PointF p2,
            out PointF clippedP1,
            out PointF clippedP2)
        {
            if ((p1.X < viewRect.Left && p2.X < viewRect.Left) ||
                (p1.X > viewRect.Right && p2.X > viewRect.Right) ||
                (p1.Y < viewRect.Top && p2.Y < viewRect.Top) ||
                (p1.Y > viewRect.Bottom && p2.Y > viewRect.Bottom))
            {
                // The line segment is outside of the view and does not intersect with it.
                clippedP1 = PointF.Empty;
                clippedP2 = PointF.Empty;
                return false;
            }
            else if ((p1.X >= viewRect.Left && p1.X <= viewRect.Right) &&
                     (p2.X >= viewRect.Left && p2.X <= viewRect.Right) &&
                     (p1.Y >= viewRect.Top && p1.Y <= viewRect.Bottom) &&
                     (p2.Y >= viewRect.Top && p2.Y <= viewRect.Bottom))
            {
                // The line segment is fully inside the rectangle.
                clippedP1 = p1;
                clippedP2 = p2;
                return true;
            }

            if ((p1.X == p2.X) && (p1.X >= viewRect.Left) && (p1.X <= viewRect.Right))
            {
                // It's a vertical line segment.
                float minY = Math.Max(viewRect.Top, Math.Min(p1.Y, p2.Y));
                float maxY = Math.Min(viewRect.Bottom, Math.Max(p1.Y, p2.Y));

                clippedP1 = new PointF(p1.X, minY);
                clippedP2 = new PointF(p1.X, maxY);
                return true;
            }
            else if ((p1.Y == p2.Y) && (p1.Y >= viewRect.Top) && (p1.Y <= viewRect.Bottom))
            {
                // It's a horizontal line segment.
                float minX = Math.Max(viewRect.Left, Math.Min(p1.X, p2.X));
                float maxX = Math.Min(viewRect.Right, Math.Max(p1.X, p2.X));

                clippedP1 = new PointF(minX, p1.Y);
                clippedP2 = new PointF(maxX, p1.Y);
                return true;
            }

            // The line segment is neither vertical nor horizontal, compute the slope.
            // Slope m = (y1 - y2) / (x1 - x2)
            float m = (float)(p1.Y - p2.Y) / (float)(p1.X - p2.X);

            // Line equation : y = mx + b => b = y - mx
            float b = (float)p1.Y - (m * (float)p1.X);

            // Find the intersection on each side of the clipping rectangle.
            float intersectTop = ((float)viewRect.Top - b) / m;
            float intersectBottom = ((float)viewRect.Bottom - b) / m;
            float intersectLeft = ((float)viewRect.Left * m) + b;
            float intersectRight = ((float)viewRect.Right * m) + b;

            if ((intersectTop < viewRect.Left || intersectTop > viewRect.Right) &&
                (intersectBottom < viewRect.Left || intersectBottom > viewRect.Right) &&
                (intersectRight < viewRect.Top || intersectRight > viewRect.Bottom) &&
                (intersectLeft < viewRect.Top || intersectLeft > viewRect.Bottom))
            {
                // The line segment doesn't intersect with the view rectangle.
                clippedP1 = PointF.Empty;
                clippedP2 = PointF.Empty;
                return false;
            }

            PointF fp1 = p1.X < p2.X ? p1 : p2;
            PointF fp2 = p1.X > p2.X ? p1 : p2;

            if (m > 0)
            {
                fp1 = fp1.X > intersectTop ? fp1 : new PointF(intersectTop, (float)viewRect.Top);
                fp1 = fp1.X > (float)viewRect.Left ? fp1 : new PointF((float)viewRect.Left, intersectLeft);

                fp2 = fp2.X < intersectBottom ? fp2 : new PointF(intersectBottom, (float)viewRect.Bottom);
                fp2 = fp2.X < (float)viewRect.Right ? fp2 : new PointF((float)viewRect.Right, intersectRight);
            }
            else
            {
                fp1 = fp1.X > intersectBottom ? fp1 : new PointF(intersectBottom, (float)viewRect.Bottom);
                fp1 = fp1.X > (float)viewRect.Left ? fp1 : new PointF((float)viewRect.Left, intersectLeft);

                fp2 = fp2.X < intersectTop ? fp2 : new PointF(intersectTop, (float)viewRect.Top);
                fp2 = fp2.X < (float)viewRect.Right ? fp2 : new PointF((float)viewRect.Right, intersectRight);
            }

            clippedP1 = fp1;
            clippedP2 = fp2;
            return true;
        }

        /// <summary>
        /// Apply transformation to the given rectangle.
        /// </summary>
        /// <param name="transformationMatrix">The transformation matrix.</param>
        /// <param name="rect">The rectangle to be transformed.</param>
        /// <param name="translationOnly">True to perform only translation.</param>
        /// <returns>The new rectangle that is transformed with the matrix.</returns>
        public static RectangleF TransformRectangle(
            Matrix transformationMatrix,
            RectangleF rect,
            bool translationOnly = false)
        {
            // Compute the correct bounds for rendering
            // in case the width and/or height are negative.
            float top = Math.Min(rect.Top, rect.Bottom);
            float bottom = Math.Max(rect.Top, rect.Bottom);
            float left = Math.Min(rect.Left, rect.Right);
            float right = Math.Max(rect.Left, rect.Right);

            PointF[] points =
            {
                new PointF(left, top),
                new PointF(right, top),
                new PointF(left, bottom)
            };

            if (translationOnly == true)
            {
                // Create a new matrix that only contains the translation part
                // of the transformation matrix.
                var translationOnlyMatrix = new Matrix();
                translationOnlyMatrix.Translate(
                    transformationMatrix.OffsetX,
                    transformationMatrix.OffsetY);

                // Transform the vertices with the transformation matrix.
                translationOnlyMatrix.TransformPoints(points);

                return new RectangleF(
                    points[0].X,
                    points[0].Y,
                    points[1].X - points[0].X,
                    points[2].Y - points[0].Y);
            }
            else
            {
                // Transform the vertices with the transformation matrix.
                transformationMatrix.TransformPoints(points);

                return new RectangleF(
                    points[0].X,
                    points[0].Y,
                    points[1].X - points[0].X,
                    points[2].Y - points[0].Y);
            }
        }

        /// <summary>
        /// Create a bitmap image and render checker board on it.
        /// </summary>
        /// <param name="width">The width of the image to create.</param>
        /// <param name="height">The height of the image to create.</param>
        /// <param name="checkerSize">The checker size.</param>
        /// <returns>The created checker board bitmap image.</returns>
        public static Bitmap CreateCheckerBoardImage(int width, int height, int checkerSize)
        {
            return CreateCheckerBoardImage(
                width,
                height,
                checkerSize,
                Color.White,
                Color.FromArgb(255, 230, 230, 230));
        }

        /// <summary>
        /// Create a bitmap image and render checker board on it.
        /// </summary>
        /// <param name="width">The width of the image to create.</param>
        /// <param name="height">The height of the image to create.</param>
        /// <param name="checkerSize">The checker size.</param>
        /// <param name="lightColor">The light color of the checker pattern.</param>
        /// <param name="darkColor">The dark color of the checker pattern.</param>
        /// <returns>The created checker board bitmap image.</returns>
        public static Bitmap CreateCheckerBoardImage(
            int width,
            int height,
            int checkerSize,
            Color lightColor,
            Color darkColor)
        {
            var image = new Bitmap(width, height, PixelFormat.Format8bppIndexed);

            var palette = image.Palette;
            {
                palette.Entries[0] = darkColor;
                palette.Entries[1] = lightColor;
            }

            image.Palette = palette;

            var rect = new Rectangle(0, 0, width, height);

            var bmpData = image.LockBits(
                rect,
                ImageLockMode.ReadWrite,
                PixelFormat.Format8bppIndexed);

            unsafe
            {
                var buffer = (byte*)(void*)bmpData.Scan0;
                var stride = bmpData.Stride;

                Parallel.For(
                    0,
                    height,
                    y =>
                    {
                        var index = y * stride;
                        var modY = (y / checkerSize) & 1;

                        for (var x = 0; x != width; ++x)
                        {
                            var modX = (x / checkerSize) & 1;

                            buffer[index + x] = (byte)((modX == modY) ? 0 : 1);
                        }
                    });
            }

            image.UnlockBits(bmpData);

            return image;
        }

        /// <summary>
        /// RGBカラー変換行列をつくる
        /// </summary>
        /// <returns>行列</returns>
        public static ColorMatrix CreateRgbColorMatrix()
        {
            return CreateRgbColorMatrix(
                new[]
                {
                    ColorComponents.Red,
                    ColorComponents.Green,
                    ColorComponents.Blue,
                    ColorComponents.Alpha
                },
                PixelFormats.Uint_8_8_8_8);
        }

        /// <summary>
        /// RGBカラー変換行列をつくる
        /// </summary>
        /// <param name="compSel">コンポーネントセレクタ</param>
        /// <param name="pixelFormat">Pixel format.</param>
        /// <returns>行列</returns>
        public static ColorMatrix CreateRgbColorMatrix(
            ColorComponents[] compSel,
            PixelFormats pixelFormat)
        {
            var compSelR = MakeCompSelMatrixCol_Color(compSel[0]);
            var compSelG = MakeCompSelMatrixCol_Color(compSel[1]);
            var compSelB = MakeCompSelMatrixCol_Color(compSel[2]);

            ColorMatrix matrix = null;
            {
                var isForceNormalMap = false;
                {
                    // snorm_8_8 と snorm_bc5 が "r g 0 1" の場合のみ、
                    // 特殊処理として法線マップのように b 成分を表示したい。
                    if ((pixelFormat == PixelFormats.Snorm_8_8) ||
                        (pixelFormat == PixelFormats.Snorm_bc5))
                    {
                        isForceNormalMap =
                            (compSel[0] == ColorComponents.Red) &&
                            (compSel[1] == ColorComponents.Green) &&
                            (compSel[2] == ColorComponents.Item0) &&
                            (compSel[3] == ColorComponents.Item1);
                    }
                }

                if (isForceNormalMap)
                {
                    // 変換しない
                    matrix = new ColorMatrix(
                        new[]
                        {
                            new[] { 1.0f, 0.0f, 0.0f, 0.0f, 0.0f },
                            new[] { 0.0f, 1.0f, 0.0f, 0.0f, 0.0f },
                            new[] { 0.0f, 0.0f, 1.0f, 0.0f, 0.0f },
                            new[] { 0.0f, 0.0f, 0.0f, 0.0f, 0.0f },
                            new[] { 0.0f, 0.0f, 0.0f, 1.0f, 0.0f }
                        });
                }
                else
                {
                    matrix = new ColorMatrix(
                        new[]
                        {
                            new[] { compSelR[0], compSelG[0], compSelB[0], 0.0f, 0.0f },
                            new[] { compSelR[1], compSelG[1], compSelB[1], 0.0f, 0.0f },
                            new[] { compSelR[2], compSelG[2], compSelB[2], 0.0f, 0.0f },
                            new[] { compSelR[3], compSelG[3], compSelB[3], 0.0f, 0.0f },
                            new[] { compSelR[4], compSelG[4], compSelB[4], 1.0f, 0.0f }
                        });
                }
            }

            return matrix;
        }

        /// <summary>
        /// RGBAカラー変換行列をつくる
        /// </summary>
        /// <param name="compSel">コンポーネントセレクタ</param>
        /// <param name="pixelFormat">Pixel format.</param>
        /// <returns>行列</returns>
        public static ColorMatrix CreateRgbaColorMatrix(
            ColorComponents[] compSel,
            PixelFormats pixelFormat)
        {
            var compSelR = MakeCompSelMatrixCol_Color(compSel[0]);
            var compSelG = MakeCompSelMatrixCol_Color(compSel[1]);
            var compSelB = MakeCompSelMatrixCol_Color(compSel[2]);
            var compSelA = MakeCompSelMatrixCol_Color(compSel[3]);

            ColorMatrix matrix = null;
            {
                var isForceNormalMap = false;
                {
                    // snorm_8_8 と snorm_bc5 が "r g 0 1" の場合のみ、
                    // 特殊処理として法線マップのように b 成分を表示したい。
                    if ((pixelFormat == PixelFormats.Snorm_8_8) ||
                        (pixelFormat == PixelFormats.Snorm_bc5))
                    {
                        isForceNormalMap =
                            (compSel[0] == ColorComponents.Red) &&
                            (compSel[1] == ColorComponents.Green) &&
                            (compSel[2] == ColorComponents.Item0) &&
                            (compSel[3] == ColorComponents.Item1);
                    }
                }

                if (isForceNormalMap)
                {
                    // 変換しない
                    matrix = new ColorMatrix(
                        new[]
                        {
                            new[] { 1.0f, 0.0f, 0.0f, 0.0f, 0.0f },
                            new[] { 0.0f, 1.0f, 0.0f, 0.0f, 0.0f },
                            new[] { 0.0f, 0.0f, 1.0f, 0.0f, 0.0f },
                            new[] { 0.0f, 0.0f, 0.0f, 1.0f, 0.0f },
                            new[] { 0.0f, 0.0f, 0.0f, 0.0f, 0.0f }
                        });
                }
                else
                {
                    matrix = new ColorMatrix(
                        new[]
                        {
                            new[] { compSelR[0], compSelG[0], compSelB[0], compSelA[0], 0.0f },
                            new[] { compSelR[1], compSelG[1], compSelB[1], compSelA[1], 0.0f },
                            new[] { compSelR[2], compSelG[2], compSelB[2], compSelA[2], 0.0f },
                            new[] { compSelR[3], compSelG[3], compSelB[3], compSelA[3], 0.0f },
                            new[] { compSelR[4], compSelG[4], compSelB[4], compSelA[4], 0.0f }
                        });
                }
            }

            return matrix;
        }

        /// <summary>
        /// アルファカラー変換行列をつくる
        /// </summary>
        /// <returns>行列</returns>
        public static ColorMatrix CreateAlphaColorMatrix()
        {
            return CreateAlphaColorMatrix(
                new[]
                {
                    ColorComponents.Red,
                    ColorComponents.Green,
                    ColorComponents.Blue,
                    ColorComponents.Alpha
                },
                PixelFormats.Uint_8_8_8_8);
        }

        /// <summary>
        /// アルファカラー変換行列をつくる
        /// </summary>
        /// <param name="compSel">コンポーネントセレクタ</param>
        /// <param name="pixelFormat">ピクセルフォーマット</param>
        /// <returns>行列</returns>
        public static ColorMatrix CreateAlphaColorMatrix(
            ColorComponents[] compSel,
            PixelFormats pixelFormat)
        {
            var compSelA = MakeCompSelMatrixCol_Color(compSel[3]);

            ColorMatrix matrix = null;
            {
                var isForceNormalMap = false;
                {
                    // snorm_8_8 と snorm_bc5 が "r g 0 1" の場合のみ、
                    // 特殊処理として法線マップのように b 成分を表示したい。
                    if ((pixelFormat == PixelFormats.Snorm_8_8) ||
                        (pixelFormat == PixelFormats.Snorm_bc5))
                    {
                        isForceNormalMap =
                            (compSel[0] == ColorComponents.Red) &&
                            (compSel[1] == ColorComponents.Green) &&
                            (compSel[2] == ColorComponents.Item0) &&
                            (compSel[3] == ColorComponents.Item1);
                    }
                }

                if (isForceNormalMap)
                {
                    matrix = new ColorMatrix(
                        new[]
                        {
                            new[] { 0.0f, 0.0f, 0.0f, 0.0f, 0.0f },
                            new[] { 0.0f, 0.0f, 0.0f, 0.0f, 0.0f },
                            new[] { 0.0f, 0.0f, 0.0f, 0.0f, 0.0f },
                            new[] { 1.0f, 1.0f, 1.0f, 0.0f, 0.0f },
                            new[] { 0.0f, 0.0f, 0.0f, 1.0f, 0.0f }
                        });
                }
                else
                {
                    matrix = new ColorMatrix(
                        new[]
                        {
                            new[] { compSelA[0], compSelA[0], compSelA[0], 0.0f, 0.0f },
                            new[] { compSelA[1], compSelA[1], compSelA[1], 0.0f, 0.0f },
                            new[] { compSelA[2], compSelA[2], compSelA[2], 0.0f, 0.0f },
                            new[] { compSelA[3], compSelA[3], compSelA[3], 0.0f, 0.0f },
                            new[] { compSelA[4], compSelA[4], compSelA[4], 1.0f, 0.0f }
                        });
                }
            }

            return matrix;
        }

        /// <summary>
        /// 角丸矩形描画パスを作る
        /// </summary>
        /// <param name="srcRect">矩形</param>
        /// <param name="cornerRadius">角丸半径</param>
        /// <returns>描画パス</returns>
        public static GraphicsPath MakeRoundRectangleGraphicsPath(
            RectangleF srcRect,
            float cornerRadius)
        {
            var backgroundPath = new GraphicsPath();
            {
                backgroundPath.AddArc(
                    srcRect.Left,
                    srcRect.Top,
                    cornerRadius,
                    cornerRadius,
                    180.0f,
                    90.0f);

                backgroundPath.AddArc(
                    srcRect.Right - cornerRadius,
                    srcRect.Top,
                    cornerRadius,
                    cornerRadius,
                    270.0f,
                    90.0f);

                backgroundPath.AddArc(
                    srcRect.Right - cornerRadius,
                    srcRect.Bottom - cornerRadius,
                    cornerRadius,
                    cornerRadius,
                    0.0f,
                    90.0f);

                backgroundPath.AddArc(
                    srcRect.Left,
                    srcRect.Bottom - cornerRadius,
                    cornerRadius,
                    cornerRadius,
                    90.0f,
                    90.0f);

                backgroundPath.CloseAllFigures();
            }

            return backgroundPath;
        }

        /// <summary>
        /// コンポーネントセレクタの色変換行列の列を求める
        /// </summary>
        /// <param name="compSel">コンポーネントセレクタ</param>
        /// <returns>列</returns>
        private static float[] MakeCompSelMatrixCol_Color(ColorComponents compSel)
        {
            var s = new[] { 0.0f, 0.0f, 0.0f, 0.0f, 0.0f };
            {
                switch (compSel)
                {
                    case ColorComponents.Red:
                        s[0] = 1.0f;
                        break;
                    case ColorComponents.Green:
                        s[1] = 1.0f;
                        break;
                    case ColorComponents.Blue:
                        s[2] = 1.0f;
                        break;
                    case ColorComponents.Alpha:
                        s[3] = 1.0f;
                        break;
                    case ColorComponents.Item0:
                        s[4] = 0.0f;
                        break;
                    case ColorComponents.Item1:
                        s[4] = 1.0f;
                        break;
                }
            }

            return s;
        }
    }
}
