﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using nw.g3d.iflib;
using nw.g3d.nw4f_3dif;

namespace EffectMaker.Foundation.Utility
{
    /// <summary>
    /// NW4F 中間ファイルユティリティ
    /// </summary>
    public class Nw4fIfUtility
    {
        /// <summary>
        /// スキーマのベースパス
        /// </summary>
        private static string XsdBasePath
        {
            get
            {
                var path = Path.Combine(Environment.GetEnvironmentVariable("NW4F_ROOT"), @"Tool\G3dTool\g3dxsd\");
                return Directory.Exists(path) ? path : null;
            }
        }

        /// <summary>
        /// 中間ファイルを読み込む
        /// </summary>
        /// <param name="filePath">中間ファイルパス</param>
        /// <returns>結果</returns>
        public static LoadResult LoadNw4F3Dif(string filePath)
        {
            //// iflib を使って実際に読みに行く
            //// 不正な中間ファイルなどので例外になることはある。
            //// この関数では例外をキャッチしない。この関数を呼ぶ側でキャッチする。

            List<G3dStream> streamArray = null;
            bool isIfUpdated;
            nw4f_3difType nw4F3Dif;
            if (G3dPath.IsStreamBinaryPath(filePath))
            {
                streamArray = new List<G3dStream>();
                nw4F3Dif = IfBinaryReadUtility.Read(streamArray, filePath, XsdBasePath, out isIfUpdated);
            }
            else
            {
                nw4F3Dif = IfReadUtility.Read(filePath, XsdBasePath, out isIfUpdated);
            }

            return new LoadResult
            {
                Nw4F3Dif = nw4F3Dif,
                BinaryStreams = streamArray ?? MakeBinaryStreams(nw4F3Dif)
            };
        }

        /// <summary>
        /// nw4f_3dif からバイナリストリームを取り出す
        /// </summary>
        /// <param name="nw4F3Dif">中間ファイル</param>
        /// <returns>バイナリストリーム</returns>
        private static List<G3dStream> MakeBinaryStreams(nw4f_3difType nw4F3Dif)
        {
            var binaryStreams = new List<G3dStream>();
            {
                // ストリームがあれば保持する
                if (G3dStreamUtility.HasStreamArray(nw4F3Dif))
                {
                    G3dStreamUtility.ToStreams(binaryStreams, nw4F3Dif.RootElement.stream_array);

                    // 改行をバイナリファイルを読み込んだ時に合わせる
                    foreach (var stream in binaryStreams)
                    {
                        switch (stream.type)
                        {
                            case stream_typeType.@string:
                            case stream_typeType.wstring:
                                stream.StringData = stream.StringData.Replace("\n", "\r\n");
                                break;
                        }
                    }
                }
            }

            return binaryStreams;
        }

        /// <summary>
        /// 読み込み結果
        /// </summary>
        public class LoadResult
        {
            /// <summary>
            /// 中間ファイルオブジェクト
            /// </summary>
            public nw4f_3difType Nw4F3Dif { get; set; }

            /// <summary>
            /// バイナリストリーム
            /// </summary>
            public List<G3dStream> BinaryStreams { get; set; }
        }
    }
}
