﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace EffectMaker.Foundation.Utility
{
    /// <summary>
    /// タスクの非同期実行と差し替え可能な継続キューを管理するクラスです。
    /// </summary>
    public class AsyncTaskQueueJustOne
    {
        /// <summary>
        /// キューに積む要素
        /// </summary>
        private class QueueElement
        {
            /// <summary>
            /// コンストラクタ
            /// </summary>
            /// <param name="key">キー文字列</param>
            /// <param name="proc">継続処理</param>
            /// <param name="cancelProc">キャンセル処理</param>
            public QueueElement(string key, Action proc, Action cancelProc)
            {
                this.Key = key;
                this.Proc = proc;
                this.CancelProc = cancelProc;
            }

            /// <summary>
            /// キー文字列を取得します。
            /// </summary>
            public string Key { get; private set; }

            /// <summary>
            /// 継続処理を取得します。
            /// </summary>
            public Action Proc { get; private set; }

            /// <summary>
            /// キャンセル処理を取得します。
            /// </summary>
            public Action CancelProc { get; private set; }
        }

        /// <summary>
        /// 非同期実行用タスク
        /// </summary>
        private Task currentProc = Task.Run(() => { });

        /// <summary>
        /// 継続処理のキュー
        /// </summary>
        private readonly List<QueueElement> procQueue = new List<QueueElement>();

        /// <summary>
        /// 実行中のタスクが無ければ処理を開始し、あれば継続処理のキューとして追加します。
        /// </summary>
        /// <param name="key">処理のターゲットを表すキー文字列</param>
        /// <param name="act">継続処理</param>
        /// <param name="cancelAct">継続処理のキャンセル処理</param>
        public void RunOrQueue(string key, Action act, Action cancelAct)
        {
            // 実行中ならキューに積む（同じキーの処理があればキャンセルして差し替え）
            if (!currentProc.IsCompleted)
            {
                lock (procQueue)
                {
                    // 同じキーで登録した処理が積まれていたら、キャンセルして削除
                    // 無ければ普通に追加する
                    var sameKeyProc = procQueue.FirstOrDefault(q => q.Key == key);
                    if (sameKeyProc != null)
                    {
                        sameKeyProc.CancelProc();
                        procQueue.Remove(sameKeyProc);
                    }

                    procQueue.Add(new QueueElement(key, act, cancelAct));
                }

                return;
            }

            // 完了している場合は即座に非同期実行を開始する
            currentProc = Task.Run(() =>
            {
                act();
                while (true)
                {
                    Action next = null;
                    lock (procQueue)
                    {
                        if (procQueue.Any())
                        {
                            // キューが空でなければ先頭要素を実行して登録を解除する
                            next = procQueue[0].Proc;
                            procQueue.RemoveAt(0);
                        }
                        else
                        {
                            // 空だったらタスクは終了（次に積まれる処理は即時実行になる）
                            break;
                        }
                    }

                    if (next != null)
                    {
                        next();
                    }

                    // next()の間に次のキューが積まれる可能性もあるので、
                    // whileループで更新を待つようにする
                }
            });
        }
    }
}
