﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Drawing.Imaging;
using System.IO;
using System.IO.Compression;
using System.Linq;
using System.Runtime.InteropServices;
using System.Threading.Tasks;

namespace EffectMaker.Foundation.Texture
{
    /// <summary>
    /// テクスチャデータ
    /// </summary>
    public class TextureData
    {
        /// <summary>
        /// 圧縮済みイメージデータです。
        /// </summary>
        private byte[,][] compressedImage;

        /// <summary>
        /// オリジナルイメージデータサイズです。
        /// </summary>
        private int[,] originalSize;

        /// <summary>
        /// OriginalImagesのデバイスコンテキストのロックオブジェクトです。
        /// エフェクトブラウザで並列読み込みを行ったときに発生する例外を回避するために使用します。
        /// </summary>
        private readonly object lockOriginalImagesHdc = new object();

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public TextureData()
        {
            this.Linear = new[] { false, false, false, false };
        }

        /// <summary>
        /// テクスチャファイル形式を表すタグ
        /// </summary>
        public string TextureTypeTag { get; set; }

        /// <summary>
        /// ファイルパスを取得または設定します。
        /// </summary>
        public string FilePath { get; set; }

        /// <summary>
        /// 変更日時を取得または設定します。
        /// </summary>
        public DateTime ModifyTime { get; set; }

        /// <summary>
        /// テクスチャタイプを取得または設定します。
        /// </summary>
        public TextureTypes TextureType { get; set; }

        /// <summary>
        /// ピクセルフォーマットを取得または設定します。
        /// </summary>
        public PixelFormats PixelFormat { get; set; }

        /// <summary>
        /// 浮動小数点型テクスチャーかどうか取得または設定します。
        /// </summary>
        public bool IsFloat { get; set; }

        /// <summary>
        /// 横幅を取得または設定します。
        /// </summary>
        public int Width { get; set; }

        /// <summary>
        /// 縦幅を取得または設定します。
        /// </summary>
        public int Height { get; set; }

        /// <summary>
        /// 奥行きを取得または設定します。
        /// </summary>
        public int Depth { get; set; }

        /// <summary>
        /// トリミング領域を取得または設定します。
        /// </summary>
        public Rectangle TrimRect { get; set; }

        /// <summary>
        /// 配列サイズを取得または設定します。
        /// </summary>
        public int ArraySize { get; set; }

        /// <summary>
        /// ミップマップ数を取得または設定します。
        /// </summary>
        public int MipmapCount { get; set; }

        /// <summary>
        /// ミップマップ最小サイズを取得します。
        /// </summary>
        public Size MipmapMinSize
        {
            get
            {
                return new Size(Math.Max(this.Width >> this.MipmapCount, 1), Math.Max(this.Height >> this.MipmapCount, 1));
            }
        }

        /// <summary>
        /// テクスチャデータのメモリーサイズを取得します。
        /// </summary>
        public virtual int MemorySize
        {
            get
            {
                if (this.IsValidOriginalImages)
                {
                    lock (this.lockOriginalImagesHdc)
                    {
                        return this.OriginalImages.Cast<Bitmap>().Sum(x => x.Width * x.Height * 4);
                    }
                }

                return this.compressedImage.Cast<byte[]>().Sum(x => x.Length);
            }
        }

        /// <summary>
        /// テクスチャのガンマをリニアとして扱いたいかどうか取得または設定します。
        /// </summary>
        public bool[] Linear { get; set; }

        /// <summary>
        /// アルファチャンネルを持つかどうか取得します。
        /// </summary>
        public virtual bool HasAlpha
        {
            get
            {
                return this.PixelFormat.HasAlpha();
            }
        }

        /// <summary>
        /// Gets or sets the comment.
        /// </summary>
        public string Comment { get; set; }

        /// <summary>
        /// Gets or sets the label color.
        /// </summary>
        public string LabelColor { get; set; }

        /// <summary>
        /// オリジナルイメージデータです。(なんちゃってミップマップ付き)
        /// </summary>
        public Bitmap[,] OriginalImages { get; set; }

        /// <summary>
        /// ArraySize プロパティと MipmapCount プロパティの値に合わせて配列データを初期化します。
        /// </summary>
        public void InitializeArray()
        {
            this.compressedImage = new byte[this.ArraySize, this.MipmapCount][];
            this.originalSize = new int[this.ArraySize, this.MipmapCount];
            this.OriginalImages = new Bitmap[this.ArraySize, this.MipmapCount];
        }

        /// <summary>
        /// オリジナルイメージが有効か否かを取得します。
        /// </summary>
        public bool IsValidOriginalImages
        {
            get
            {
                return this.OriginalImages != null && this.OriginalImages.Cast<Bitmap>().Select(image => image != null).FirstOrDefault();
            }
        }

        /// <summary>
        /// The resize bitmap.
        /// </summary>
        /// <param name="bitmap">
        /// The bitmap.
        /// </param>
        /// <param name="size">
        /// The size.
        /// </param>
        /// <param name="matrix">
        /// The matrix.
        /// </param>
        /// <param name="keepAspect">
        /// アスペクト比をキープするかしないかを指定します。デフォルトはしません。
        /// </param>
        /// <returns>
        /// The <see cref="Bitmap"/>.
        /// </returns>
        public static Bitmap ResizeBitmap(Bitmap bitmap, Size size, ColorMatrix matrix = null, bool keepAspect = false)
        {
            // 指定サイズでなければリサイズする
            if ((bitmap.Width != size.Width) || (bitmap.Height != size.Height))
            {
                var ia = new ImageAttributes();
                var resizedBitmap = new Bitmap(size.Width, size.Height, System.Drawing.Imaging.PixelFormat.Format32bppArgb);
                {
                    if (matrix != null)
                    {
                        ia.SetColorMatrix(matrix);
                    }

                    using (var g = Graphics.FromImage(resizedBitmap))
                    {
                        g.InterpolationMode = InterpolationMode.HighQualityBicubic;

                        // 拡大時はサンプリングポイントをずらす
                        var samplingX = (resizedBitmap.Width > bitmap.Width) ? -0.5f : 0.0f;
                        var samplingY = (resizedBitmap.Height > bitmap.Height) ? -0.5f : 0.0f;

                        var rectScale = new Rectangle(0, 0, resizedBitmap.Width, resizedBitmap.Height);

                        using (var brush = new SolidBrush(Color.Transparent))
                        {
                            if (keepAspect)
                            {
                                if (bitmap.Width > bitmap.Height)
                                {
                                    float scale = (float)resizedBitmap.Width / (float)bitmap.Width;
                                    rectScale.Height = (int)(bitmap.Height * scale);
                                    rectScale.Y = (rectScale.Width - rectScale.Height) / 2;
                                    g.FillRectangle(brush, 0.0f, 0.0f, rectScale.Width, rectScale.Y);
                                    g.FillRectangle(brush, 0.0f, rectScale.Y + rectScale.Height, rectScale.Width, rectScale.Y);
                                }
                                else if (bitmap.Width < bitmap.Height)
                                {
                                    float scale = (float)resizedBitmap.Height / (float)bitmap.Height;
                                    rectScale.Width = (int)(bitmap.Width * scale);
                                    rectScale.X = (rectScale.Height - rectScale.Width) / 2;
                                    g.FillRectangle(brush, 0.0f, 0.0f, rectScale.X, rectScale.Height);
                                    g.FillRectangle(brush, rectScale.X + rectScale.Width, 0.0f, rectScale.X, rectScale.Height);
                                }
                            }
                        }

                        g.DrawImage(
                            bitmap,
                            rectScale,
                            samplingX,
                            samplingY,
                            bitmap.Width - samplingX,
                            bitmap.Height - samplingX,
                            GraphicsUnit.Pixel,
                            ia);
                    }
                }

                bitmap = resizedBitmap;
            }

            return bitmap;
        }

        /// <summary>
        /// 整数型イメージの設定
        /// </summary>
        /// <param name="arrayIndex">配列インデックス</param>
        /// <param name="mipmapLevel">ミップマップレベル</param>
        /// <param name="image">イメージ</param>
        public void SetByteImage(int arrayIndex, int mipLevel, byte[] image)
        {
            Debug.Assert(this.IsFloat == false, "整数型である必要がある");

            // BC圧縮テクスチャは4x4ブロックでアライメントされているのでアライメント部分を削る
            if (this.PixelFormat.IsCompressed())
            {
                image = this.RemoveAlignPixel(mipLevel, image);
            }

            this.compressedImage[arrayIndex, mipLevel] = CompressArray(image);
            this.originalSize[arrayIndex, mipLevel] = image.Length;
        }

        /// <summary>
        /// 浮動小数点型イメージの設定
        /// </summary>
        /// <param name="arrayIndex">配列インデックス</param>
        /// <param name="mipmapLevel">ミップマップレベル</param>
        /// <param name="image">イメージ</param>
        public void SetFloatImage(int arrayIndex, int mipLevel, float[] image)
        {
            Debug.Assert(this.IsFloat, "浮動小数点型である必要がある");

            // float[] を一旦 byte[] にためる
            var tempByteArray = new byte[image.Length * 4];  // 4 は sizeof(float)/sizeof(byte)

            unsafe
            {
                fixed (byte* tempPtr = tempByteArray)
                {
                    Marshal.Copy(image, 0, (IntPtr)tempPtr, image.Length);
                }
            }

            this.compressedImage[arrayIndex, mipLevel] = CompressArray(tempByteArray);
            this.originalSize[arrayIndex, mipLevel] = tempByteArray.Length;
        }

        /// <summary>
        /// ビットマップを生成する
        /// </summary>
        /// <param name="arrayIndex">配列インデックス</param>
        /// <param name="mipLevel">ミップマップレベル</param>
        /// <param name="floatConverter">浮動小数点をどのようにbyteに落としこむか</param>
        /// <returns>生成したビットマップ</returns>
        public virtual Bitmap GenerateBitmap(int arrayIndex, int mipLevel, Func<float, byte> floatConverter = null)
        {
            Debug.Assert(arrayIndex < this.ArraySize, "配列インデックスの指定が不正");
            Debug.Assert(mipLevel < this.MipmapCount, "ミップマップの指定が不正");

            if (this.IsValidOriginalImages)
            {
                lock (this.lockOriginalImagesHdc)
                {
                    return (Bitmap)this.OriginalImages[arrayIndex, mipLevel].Clone();
                }
            }

            var mippedW = Math.Max(this.Width >> mipLevel, 1);
            var mippedH = Math.Max(this.Height >> mipLevel, 1);

            // ビットマップに画素をコピーする
            var bitmap = new Bitmap(mippedW, mippedH, System.Drawing.Imaging.PixelFormat.Format32bppArgb);

            var rect = new Rectangle(0, 0, mippedW, mippedH);
            var bitmapData = bitmap.LockBits(rect, ImageLockMode.WriteOnly, System.Drawing.Imaging.PixelFormat.Format32bppArgb);
            {
                try
                {
                    if (this.IsFloat)
                    {
                        var floatImage = this.GetDecompressFloatImage(arrayIndex, mipLevel);

                        unsafe
                        {
                            var dst = (byte*)bitmapData.Scan0.ToPointer();

                            if (floatConverter != null)
                            {
                                Parallel.For(0, floatImage.Length, i => dst[i] = floatConverter(floatImage[i]));
                            }
                            else
                            {
                                Parallel.For(0, floatImage.Length, i => dst[i] = (byte)(Math.Min(Math.Max(floatImage[i], 0.0f), 1.0f) * 255.0f));
                            }
                        }
                    }
                    else
                    {
                        var byteImage = this.GetDecompressByteImage(arrayIndex, mipLevel);
                        Marshal.Copy(byteImage, 0, bitmapData.Scan0, mippedW * mippedH * 4);
                    }
                }
                finally
                {
                    bitmap.UnlockBits(bitmapData);
                }
            }

            return bitmap;
        }

        /// <summary>
        /// プレビュー用のビットマップ画像を生成します。
        /// </summary>
        /// <remarks>
        /// プレビュー用のビットマップ画像は、配列テクスチャを1枚にまとめた形式になります。
        /// </remarks>
        /// <param name="size">画像サイズ</param>
        /// <param name="mipLevel">ミップマップレベル</param>
        /// <param name="floatConverter">浮動小数点をどのようにbyteに落としこむか</param>
        /// <returns>生成したビットマップ画像を返します。</returns>
        public virtual Bitmap GeneratePreviewBitmap(int mipLevel, Func<float, byte> floatConverter = null)
        {
            // 配列テクスチャを並べるときの列と行の数を計算
            int col = (int)Math.Ceiling(Math.Sqrt(this.ArraySize));
            int row = (this.ArraySize + col - 1) / col;

            Bitmap bitmap;

            // ビットマップ画像を生成
            if (this.ArraySize == 1)
            {
                bitmap = this.GenerateBitmap(0, mipLevel, floatConverter);
            }
            else
            {
                int mipWidth = this.Width >> mipLevel;
                int mipHeight = this.Height >> mipLevel;

                bitmap = new Bitmap(mipWidth * col, mipHeight * row, System.Drawing.Imaging.PixelFormat.Format32bppArgb);

                // 配列テクスチャを1枚の画像に並べる
                using (Graphics g = Graphics.FromImage(bitmap))
                {
                    for (int i = 0; i < this.ArraySize; ++i)
                    {
                        using (Bitmap bmp = this.GenerateBitmap(i, mipLevel, floatConverter))
                        {
                            if (bmp == null) continue;

                            int x = mipWidth * (i % col);
                            int y = mipHeight * (i / col);

                            g.DrawImage(bmp, x, y);
                        }
                    }
                }
            }

            return bitmap;
        }

        /// <summary>
        /// 指定された画像サイズに適合するミップマップから、プレビュー用のビットマップ画像を生成します。
        /// </summary>
        /// <remarks>
        /// プレビュー用のビットマップ画像は、配列テクスチャを1枚にまとめた形式になります。
        /// </remarks>
        /// <param name="size">画像サイズ</param>
        /// <param name="floatConverter">浮動小数点をどのようにbyteに落としこむか</param>
        /// <returns>生成したビットマップ画像を返します。</returns>
        public virtual Bitmap GeneratePreviewBitmap(Size size, Func<float, byte> floatConverter = null)
        {
            int mipLevel;

            // 配列テクスチャを並べるときの列と行の数を計算
            int col = (int)Math.Ceiling(Math.Sqrt(this.ArraySize));
            int row = (this.ArraySize + col - 1) / col;

            // サイズに適合するミップマップレベルを計算
            {
                var mippedW = this.Width / col;
                var mippedH = this.Height / row;

                for (mipLevel = 0; mippedW > size.Width && mippedH > size.Height; ++mipLevel)
                {
                    mippedW = Math.Max(mippedW >> 1, 1);
                    mippedH = Math.Max(mippedH >> 1, 1);
                }

                mipLevel = Math.Min(Math.Max(0, mipLevel - 1), this.MipmapCount - 1);
            }

            return GeneratePreviewBitmap(mipLevel, floatConverter);
        }

        /// <summary>
        /// 指定された画像サイズにリサイズしたプレビュー用のビットマップ画像を生成します。
        /// </summary>
        /// <remarks>
        /// プレビュー用のビットマップ画像は、配列テクスチャを1枚にまとめた形式になります。
        /// </remarks>
        /// <param name="size">画像サイズ</param>
        /// <param name="mipLevel">ミップマップレベル</param>
        /// <param name="matrix">カラーマトリックス</param>
        /// <param name="floatConverter">浮動小数点をどのようにbyteに落としこむか</param>
        /// <returns>生成したビットマップ画像を返します。</returns>
        public virtual Bitmap GeneratePreviewBitmapWithResize(Size size, int mipLevel, ColorMatrix matrix = null, Func<float, byte> floatConverter = null)
        {
            // 任意のミップマップレベルの画像を生成する
            Bitmap bitmap = GeneratePreviewBitmap(mipLevel, floatConverter);
            if (bitmap == null) return null;

            // 指定サイズでなければリサイズする
            return ResizeBitmap(bitmap, size, matrix);
        }

        /// <summary>
        /// 展開済み整数型イメージを得る
        /// </summary>
        /// <param name="arrayIndex">配列インデックス</param>
        /// <param name="mipLevel">ミップマップレベル</param>
        /// <returns>展開済みイメージ</returns>
        private byte[] GetDecompressByteImage(int arrayIndex, int mipLevel)
        {
            Debug.Assert(this.IsFloat == false, "整数型である必要がある");
            Debug.Assert(mipLevel < this.MipmapCount, "ミップマップの指定が不正");

            return DecompressArray(this.compressedImage[arrayIndex, mipLevel], this.originalSize[arrayIndex, mipLevel]);
        }

        /// <summary>
        /// 展開済み浮動小数点型イメージを得る
        /// </summary>
        /// <param name="arrayIndex">配列インデックス</param>
        /// <param name="mipLevel">ミップマップレベル</param>
        /// <returns>展開済みイメージ</returns>
        private float[] GetDecompressFloatImage(int arrayIndex, int mipLevel)
        {
            var byteArray = DecompressArray(this.compressedImage[arrayIndex, mipLevel], this.originalSize[arrayIndex, mipLevel]);
            var floatArray = new float[byteArray.Length / 4];  // 4 は sizeof(float) / sizeof(byte)

            unsafe
            {
                fixed (byte* tempPtr = byteArray)
                {
                    Marshal.Copy((IntPtr)tempPtr, floatArray, 0, floatArray.Length);
                }
            }

            return floatArray;
        }

        /// <summary>
        /// 配列を圧縮する
        /// </summary>
        /// <param name="src">圧縮するデータ</param>
        /// <returns>圧縮後データ</returns>
        private static byte[] CompressArray(byte[] src)
        {
            using (var tempStream = new MemoryStream(src.Length))
            {
                using (var gzip = new GZipStream(tempStream, CompressionMode.Compress))
                {
                    gzip.Write(src, 0, src.Length);
                }

                return tempStream.ToArray();
            }
        }

        /// <summary>
        /// 配列を展開する
        /// </summary>
        /// <param name="src">展開するデータ</param>
        /// <param name="dstSize">展開後サイズ</param>
        /// <returns>展開後データ</returns>
        private static byte[] DecompressArray(byte[] src, int dstSize)
        {
            var temp = new byte[dstSize];
            {
                using (var srcStream = new MemoryStream(src))
                using (var gzip = new GZipStream(srcStream, CompressionMode.Decompress))
                {
                    gzip.Read(temp, 0, temp.Length);
                }
            }

            return temp;
        }

        /// <summary>
        /// アライメント部分を削る
        /// </summary>
        /// <param name="mipmapLevel">ミップマップレベル</param>
        /// <param name="srcImage">イメージ</param>
        /// <returns>処理済みイメージ</returns>
        private byte[] RemoveAlignPixel(int mipmapLevel, byte[] srcImage)
        {
            Debug.Assert(this.PixelFormat.IsCompressed(), "this.IsCompressFormat");

            var width = Math.Max(1, this.Width >> mipmapLevel);
            var height = Math.Max(1, this.Height >> mipmapLevel);

            if (((width & 3) == 0) && ((height & 3) == 0))
            {
                return srcImage;
            }

            var srcWidth = (width + 3) & ~3;
            var image = new byte[width * height * 4];

            unsafe
            {
                //// 注意: この関数は並列実行されるので、関数内では並列処理を行わない

                fixed (byte* dstImage = &image[0])
                {
                    for (var y = 0; y != height; ++y)
                    {
                        var srcPtr = y * srcWidth * 4;
                        var dst = dstImage + (y * width * 4);

                        Marshal.Copy(srcImage, srcPtr, (IntPtr)dst, width * 4);
                    }
                }
            }

            return image;
        }
    }
}
