﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace EffectMaker.Foundation.Core
{
    /// <summary>
    /// An IEqualityComparer that route interface implemenation to delegates.
    /// </summary>
    /// <typeparam name="T">The type of object to check the equality.</typeparam>
    public class AnonymousEqualityComparer<T> : IEqualityComparer<T>
    {
        /// <summary>
        /// Stores the GetHashCode method delegate.
        /// </summary>
        private readonly Func<T, int> getHashCode;

        /// <summary>
        /// Stores the Equals method delegate.
        /// </summary>
        private readonly Func<T, T, bool> equals;

        /// <summary>
        /// Initializes the AnonymousEqualityComparer instance.
        /// </summary>
        /// <param name="getHashCode">The GetHashCode method delegate.</param>
        /// <param name="equals">The Equals method delegate.</param>
        public AnonymousEqualityComparer(Func<T, int> getHashCode, Func<T, T, bool> equals)
        {
            if (getHashCode == null)
            {
                throw new ArgumentNullException("getHashCode");
            }

            if (equals == null)
            {
                throw new ArgumentNullException("equals");
            }

            this.getHashCode = getHashCode;
            this.equals = equals;
        }

        /// <summary>
        /// Initializes the AnonymousEqualityComparer instance.
        /// </summary>
        /// <param name="equals">The Equals method delegate.</param>
        public AnonymousEqualityComparer(Func<T, T, bool> equals)
        {
            if (equals == null)
            {
                throw new ArgumentNullException("equals");
            }

            this.getHashCode = NullGetHashCode;
            this.equals = equals;
        }

        /// <summary>
        /// Check the equality of the two given instances.
        /// </summary>
        /// <param name="x">First instance.</param>
        /// <param name="y">Second instance.</param>
        /// <returns>Returns true if equal, false otherwise.</returns>
        public bool Equals(T x, T y)
        {
            return this.equals(x, y);
        }

        /// <summary>
        /// Computes a hash code for the given instance.
        /// </summary>
        /// <param name="obj">The instance to compute its hash code.</param>
        /// <returns>Returns the hash code representing the instance.</returns>
        public int GetHashCode(T obj)
        {
            return this.getHashCode(obj);
        }

        /// <summary>
        /// Always return 0.
        /// </summary>
        /// <param name="obj">The object to compute the hash code.</param>
        /// <returns>Returns 0.</returns>
        private static int NullGetHashCode(T obj)
        {
            return 0;
        }
    }
}
