﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.CodeDom.Compiler;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Reflection;
using EffectMaker.Foundation.Debugging.Profiling;
using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Log;
using Microsoft.CSharp;

namespace EffectMaker.Foundation.Compiler
{
    /// <summary>
    /// ランタイムコンパイラです。
    /// 動的にC#コードをコンパイルして、アセンブリを生成・ロードします。
    /// 速度最適化のためアセンブリを自動的にキャッシュします。
    /// </summary>
    public static class RuntimeCompiler
    {
        /// <summary>
        /// アセンブリフォルダパスです。
        /// このフォルダには実行時にロックして使用するdllを配置します。
        /// </summary>
        private static string assemblyFolderPath;

        /// <summary>
        /// アセンブリキャッシュです。
        /// </summary>
        private static Dictionary<string, AssemblyCacheInfo> assemblyCache;

        /// <summary>
        /// ランタイムコンパイラを初期化します。
        /// </summary>
        /// <param name="assemblyFolder">アセンブリフォルダパス</param>
        /// <returns>初期化に成功したときはtrue、それ以外はfalseを返します。</returns>
        public static bool Initialize(string assemblyFolder)
        {
            assemblyFolderPath = assemblyFolder;
            assemblyCache = new Dictionary<string, AssemblyCacheInfo>();

            return true;
        }

        /// <summary>
        /// C#のソースコードをコンパイルしてアセンブリを生成します。
        /// 生成元となるファイルに変更がない場合はキャッシュしたアセンブリを返します。
        /// </summary>
        /// <param name="assemblyKey">キャッシュを利用するためのアセンブリキー</param>
        /// <param name="sourceFilePaths">生成元となるファイル</param>
        /// <param name="referencedAssemblies">コンパイル時に参照するアセンブリ</param>
        /// <param name="codeFilePaths">読み込むソースコードのファイルパス</param>
        /// <returns>コンパイルして生成したアセンブリを返します。コンパイルに失敗したときはnullを返します。</returns>
        public static Assembly CompileFromFile(
            string assemblyKey,
            IEnumerable<string> sourceFilePaths,
            IEnumerable<string> referencedAssemblies,
            IEnumerable<string> codeFilePaths)
        {
            Func<IEnumerable<string>> getCodes = () =>
            {
                IEnumerable<string> codes = null;

                try
                {
                    codes = codeFilePaths.Select(x => File.ReadAllText(x)).ToArray();
                }
                catch (Exception e)
                {
                    Logger.Log(LogLevels.Warning, "RuntimeCompiler.CompileAndCache : Failed to read a source file. {0}", e.Message);
                }

                return codes;
            };

            return CompileInternal(assemblyKey, sourceFilePaths, referencedAssemblies, getCodes);
        }

        /// <summary>
        /// C#のソースコードをコンパイルしてアセンブリを生成します。
        /// 生成元となるファイルに変更がない場合はキャッシュしたアセンブリを返します。
        /// </summary>
        /// <param name="assemblyKey">キャッシュを利用するためのアセンブリキー</param>
        /// <param name="sourceFilePaths">生成元となるファイル</param>
        /// <param name="referencedAssemblies">コンパイル時に参照するアセンブリ</param>
        /// <param name="codes">オンメモリで生成したコードの配列</param>
        /// <returns>コンパイルして生成したアセンブリを返します。コンパイルに失敗したときはnullを返します。</returns>
        public static Assembly CompileFromMemory(
            string assemblyKey,
            IEnumerable<string> sourceFilePaths,
            IEnumerable<string> referencedAssemblies,
            IEnumerable<string> codes)
        {
            Func<IEnumerable<string>> getCodes = () =>
            {
                return codes;
            };

            return CompileInternal(assemblyKey, sourceFilePaths, referencedAssemblies, getCodes);
        }

        /// <summary>
        /// C#のソースコードをコンパイルしてアセンブリを生成します。
        /// 生成元となるファイルに変更がない場合はキャッシュしたアセンブリを返します。
        /// </summary>
        /// <param name="assemblyKey">キャッシュを利用するためのアセンブリキー</param>
        /// <param name="sourceFilePaths">生成元となるファイル</param>
        /// <param name="referencedAssemblies">コンパイル時に参照するアセンブリ</param>
        /// <param name="getCodes">ソースコードの取得関数</param>
        /// <returns>コンパイルして生成したアセンブリを返します。コンパイルに失敗したときはnullを返します。</returns>
        private static Assembly CompileInternal(
            string assemblyKey,
            IEnumerable<string> sourceFilePaths,
            IEnumerable<string> referencedAssemblies,
            Func<IEnumerable<string>> getCodes)
        {
            if (string.IsNullOrEmpty(assemblyKey) == true)
            {
                Logger.Log(LogLevels.Warning, "RuntimeCompiler.CompileAndCache : The assembly key is mandatory for caching the compiled assembly.");
                return null;
            }

            if (sourceFilePaths.IsNullOrEmpty() == true)
            {
                Logger.Log(LogLevels.Warning, "RuntimeCompiler.CompileAndCache : There is no source file to compile.");
                return null;
            }

            // アセンブリキャッシュが利用可能ならキャッシュを返す
            {
                string[] sourceFileArray = sourceFilePaths.ToArray();

                AssemblyCacheInfo cacheInfo;
                bool findCache = assemblyCache.TryGetValue(assemblyKey, out cacheInfo);

                if (findCache)
                {
                    if (IsAssemblyCacheValid(cacheInfo, sourceFileArray))
                    {
                        return cacheInfo.Assembly;
                    }
                    else
                    {
                        assemblyCache.Remove(assemblyKey);
                        cacheInfo = null;
                    }
                }
            }

            // ソースコードを取得
            // キャッシュがヒットした場合は不要なので、ファイルのロード処理などをここまで先延ばしにしておく
            IEnumerable<string> codes = getCodes();

            if (codes == null)
            {
                return null;
            }

            string assemblyPath;

            // dllのファイル名が被らないようにする
            do
            {
                string guid = Guid.NewGuid().ToString("D").Substring(0, 8);
                string assemblyName = string.Format("{0}_{1}.dll", assemblyKey, guid);
                assemblyPath = Path.Combine(assemblyFolderPath, assemblyName);
            }
            while (File.Exists(assemblyPath));

            // コンパイルパラメータを設定
            var param = new CompilerParameters
            {
                GenerateExecutable = false,
                GenerateInMemory = false,
                OutputAssembly = assemblyPath,
                #if DEBUG
                IncludeDebugInformation = true,
                #else
                CompilerOptions = "/optimize",
                IncludeDebugInformation = false,
                #endif
            };

            // 参照アセンブリを設定
            param.ReferencedAssemblies.AddRange(referencedAssemblies.ToArray());

            Assembly assembly = null;

            // コンパイル処理を行う
            using (new ProfileTimer("Compile assembly " + Path.GetFileName(assemblyPath)))
            {
                // プロバイダーオプション
                var providerOptions = new Dictionary<string, string>
                {
                    { "CompilerVersion", "v4.0" },
                };

                using (var codeProvider = new CSharpCodeProvider(providerOptions))
                {
                    // コンパイル！！！
                    var result = codeProvider.CompileAssemblyFromSource(param, codes.ToArray());

                    // エラー処理を行う
                    if (result.Errors.Count > 0)
                    {
                        Logger.Log(LogLevels.Error, "Failed compiling assembly.");

                        foreach (var err in result.Errors)
                        {
                            Logger.Log(LogLevels.Error, err.ToString());
                        }
                    }
                    else if (result.CompiledAssembly == null)
                    {
                        Logger.Log(LogLevels.Error, "Failed compiling the assembly due to unknown reason.");
                    }
                    // 成功時はアセンブリを取得する
                    else
                    {
                        assembly = result.CompiledAssembly;
                    }
                }
            }

            if (assembly == null)
            {
                return null;
            }

            // キャッシュ情報を追加する
            {
                AssemblySourceFileInfo[] sourceFileInfo = sourceFilePaths.Select(x => new AssemblySourceFileInfo()
                {
                    FilePath = x,
                    ModifyTime = File.GetLastWriteTimeUtc(x)
                }).ToArray();

                AssemblyCacheInfo cacheInfo = new AssemblyCacheInfo()
                {
                    Key = assemblyKey,
                    SourceFiles = sourceFileInfo,
                    AssemblyPath = assemblyPath,
                    Assembly = assembly
                };

                assemblyCache.Add(assemblyKey, cacheInfo);
            }

            return assembly;
        }

        /// <summary>
        /// アセンブリキャッシュが利用可能かどうかチェックします。
        /// </summary>
        /// <param name="cacheInfo">アセンブリキャッシュ情報</param>
        /// <param name="sourceFiles">生成元となるファイル</param>
        /// <returns>アセンブリキャッシュが利用可能であればtrue、それ以外はfalseを返します。</returns>
        private static bool IsAssemblyCacheValid(AssemblyCacheInfo cacheInfo, string[] sourceFiles)
        {
            // 生成元となるファイルについて重複を無視して比較するため、ハッシュセットを作成
            HashSet<string> sourceFileHashSet = new HashSet<string>(sourceFiles);

            IEnumerable<string> cacheSourceFiles = cacheInfo.SourceFiles.Select(x => x.FilePath);
            HashSet<string> cacheSourceFileHashSet = new HashSet<string>(cacheSourceFiles);

            // 生成元となるファイルの増減をチェック
            if (cacheSourceFileHashSet.Count != sourceFileHashSet.Count)
            {
                return false;
            }

            // 生成元となるファイルに差異がないかチェック
            foreach (string sourceFile in sourceFileHashSet)
            {
                if (cacheSourceFileHashSet.Contains(sourceFile) == false)
                {
                    return false;
                }
            }

            // 生成元となるファイルの更新時間をチェック
            foreach (AssemblySourceFileInfo srcInfo in cacheInfo.SourceFiles)
            {
                DateTime modifyTime = File.GetLastWriteTimeUtc(srcInfo.FilePath);

                if (modifyTime != srcInfo.ModifyTime)
                {
                    return false;
                }
            }

            return true;
        }

        /// <summary>
        /// アセンブリキャッシュ情報です。
        /// </summary>
        private class AssemblyCacheInfo
        {
            /// <summary>
            /// ハッシュキーを取得または設定します。
            /// </summary>
            public string Key { get; set; }

            /// <summary>
            /// アセンブリの生成元となるファイルの情報を取得または設定します。
            /// </summary>
            public AssemblySourceFileInfo[] SourceFiles { get; set; }

            /// <summary>
            /// アセンブリファイルのパスを取得または設定します。
            /// </summary>
            public string AssemblyPath { get; set; }

            /// <summary>
            /// アセンブリを取得または設定します。
            /// </summary>
            public Assembly Assembly { get; set; }
        }

        /// <summary>
        /// アセンブリの生成元となるファイルの情報です。
        /// </summary>
        private class AssemblySourceFileInfo
        {
            /// <summary>
            /// ファイルパスを取得または設定します。
            /// </summary>
            public string FilePath { get; set; }

            /// <summary>
            /// ファイルの変更日時を取得または設定します。
            /// </summary>
            public DateTime ModifyTime { get; set; }
        }
    }
}
