﻿using System;
using System.Collections.Generic;

namespace EffectMaker.Foundation.Collections.Generic
{
    /// <summary>
    /// キーと複数の値のコレクションを表します。
    /// </summary>
    /// <typeparam name="TKey">ディクショナリ内のキーの型。</typeparam>
    /// <typeparam name="TValue">ディクショナリ内の値の型。</typeparam>
    public class MultiDictionary<TKey, TValue> : Dictionary<TKey, List<TValue>>
    {
        /// <summary>
        /// 空で、既定の初期量を備え、キーの型の既定の等値比較子を使用する、MultiDictionary<TKey, TValue> クラスの新しいインスタンスを初期化します。
        /// </summary>
        public MultiDictionary()
            : base()
        {
        }

        /// <summary>
        /// 空で、指定した初期量を備え、キーの型の既定の等値比較子を使用する、MultiDictionary<TKey, TValue> クラスの新しいインスタンスを初期化します。
        /// </summary>
        /// <param name="capacity">MultiDictionary<TKey, TValue> が格納できる要素数の初期値。</param>
        public MultiDictionary(int capacity)
            : base(capacity)
        {
        }

        /// <summary>
        /// 指定したキーと値をディクショナリに追加します。
        /// </summary>
        /// <param name="key">追加する要素のキー。</param>
        /// <param name="value">追加する要素の値。参照型の場合は null の値を使用できます。</param>
        public void Add(TKey key, TValue value)
        {
            // キーに関連付けられているリストを取得
            List<TValue> list;
            this.TryGetValue(key, out list);

            // キーに関連付けられているリストがなければ作成
            if (list == null)
            {
                list = new List<TValue>();
                this[key] = list;
            }

            // キーに関連付けられているリストに値を追加
            list.Add(value);
        }

        /// <summary>
        /// 指定したキーと複数値をディクショナリに追加します。
        /// </summary>
        /// <param name="key">追加する要素のキー。</param>
        /// <param name="values">追加する要素の複数の値。参照型の場合は null の値を使用できます。</param>
        public void Add(TKey key, IEnumerable<TValue> values)
        {
            // キーに関連付けられているリストを取得
            List<TValue> list;
            this.TryGetValue(key, out list);

            // キーに関連付けられているリストがなければ作成
            if (list == null)
            {
                list = new List<TValue>();
                this[key] = list;
            }

            // キーに関連付けられているリストに複数の値を追加
            list.AddRange(values);
        }

        /// <summary>
        /// 指定したキーと値が MultiDictionary<TKey, TValue> に格納されているかどうかを判断します。
        /// </summary>
        /// <param name="key">MultiDictionary<TKey, TValue> 内で検索されるキー。</param>
        /// <param name="value">MultiDictionary<TKey, TValue> 内で検索される値。参照型の場合は null の値を使用できます。</param>
        /// <returns>指定した値を持つ要素が MultiDictionary<TKey, TValue> に格納されている場合は true。それ以外の場合は false。</returns>
        public bool Contains(TKey key, TValue value)
        {
            // キーに関連付けられているリストを取得
            List<TValue> list;
            this.TryGetValue(key, out list);

            // キーに関連付けられているリストがなければ false を返す
            if (list == null)
            {
                return false;
            }

            // リストから値を検索
            return list.Contains(value);
        }

        /// <summary>
        /// 指定したキーが MultiDictionary<TKey, TValue> に特定の値が格納されているかどうかを判断します。
        /// </summary>
        /// <param name="value">MultiDictionary<TKey, TValue> 内で検索される値。参照型の場合は null の値を使用できます。</param>
        /// <returns>指定した値を持つ要素が MultiDictionary<TKey, TValue> に格納されている場合は true。それ以外の場合は false。</returns>
        public bool ContainsValue(TValue value)
        {
            foreach (List<TValue> list in this.Values)
            {
                if (list.Contains(value))
                {
                    return true;
                }
            }

            return false;
        }

        /// <summary>
        /// 指定したキーと値を MultiDictionary<TKey, TValue> から削除します。
        /// </summary>
        /// <param name="key">削除する要素のキー。</param>
        /// <param name="value">削除する要素の値。</param>
        /// <returns>要素が見つかり、正常に削除された場合は true。それ以外の場合は false。このメソッドは、key と value が MultiDictionary<TKey, TValue> に見つからない場合、false を返します。</returns>
        public bool Remove(TKey key, TValue value)
        {
            // キーに関連付けられているリストを取得
            List<TValue> list;
            this.TryGetValue(key, out list);

            // キーに関連付けられているリストがなければ false を返す
            if (list == null)
            {
                return false;
            }

            // リストから値を削除
            bool resRemove = list.Remove(value);

            // リストの内容が 0 になったときリストを削除する
            if (resRemove == true && list.Count == 0)
            {
                resRemove = this.Remove(key);
            }

            return resRemove;
        }
    }
}
