﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;

namespace EffectMaker.Foundation.Collections.Generic
{
    /// <summary>
    /// LRUキャッシュです。
    /// このクラスを継承して GetValueSize メソッドや OnDisposingValue メソッドを
    /// オーバーライドすることにで、キャッシュデータに応じた機能拡張ができます。
    /// </summary>
    /// <typeparam name="TKey">キーの型</typeparam>
    /// <typeparam name="TValue">値の型</typeparam>
    public class LruCache<TKey, TValue>
    {
        /// <summary>
        /// キー値保存用リスト
        /// </summary>
        private readonly LinkedList<KeyValuePair> keyValueStorage;

        /// <summary>
        /// キー値検索用ディクショナリ
        /// </summary>
        private readonly Dictionary<TKey, LinkedListNode<KeyValuePair>> keyValueLookup;

        /// <summary>
        /// 同期オブジェクト
        /// </summary>
        private readonly object syncObject;

        /// <summary>
        /// 最大キャパシティ
        /// </summary>
        private int maxCapacity;

        /// <summary>
        /// 現在のサイズ
        /// </summary>
        private int currentSize;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="maxCapacity">最大キャパシティ</param>
        /// <param name="comparer">キーの比較時に使用する IEqualityComparer 実装</param>
        public LruCache(int maxCapacity, IEqualityComparer<TKey> comparer = null)
        {
            this.keyValueStorage = new LinkedList<KeyValuePair>();
            this.keyValueLookup = (comparer != null) ? new Dictionary<TKey, LinkedListNode<KeyValuePair>>(comparer) : new Dictionary<TKey, LinkedListNode<KeyValuePair>>();
            this.syncObject = new object();

            this.maxCapacity = maxCapacity;
            this.currentSize = 0;
        }

        /// <summary>
        /// キャッシュをクリアします。
        /// </summary>
        public void Clear()
        {
            lock (this.syncObject)
            {
                foreach (var valueNode in this.keyValueStorage)
                {
                    this.OnDisposingValue(valueNode.Key, valueNode.Value);
                }

                this.keyValueStorage.Clear();
                this.keyValueLookup.Clear();
            }
        }

        /// <summary>
        /// キーに対応する値を取得します。
        /// </summary>
        /// <param name="key">キー</param>
        /// <returns>値を返します。</returns>
        public TValue Get(TKey key)
        {
            lock (this.syncObject)
            {
                LinkedListNode<KeyValuePair> listNode;

                if (this.keyValueLookup.TryGetValue(key, out listNode))
                {
                    this.keyValueStorage.Remove(listNode);
                    this.keyValueStorage.AddFirst(listNode);

                    return listNode.Value.Value;
                }

                return default(TValue);
            }
        }

        /// <summary>
        /// 値を追加します。
        /// </summary>
        /// <param name="key">キー</param>
        /// <param name="value">値</param>
        public void Add(TKey key, TValue value)
        {
            lock (this.syncObject)
            {
                LinkedListNode<KeyValuePair> listNode;

                if (this.keyValueLookup.TryGetValue(key, out listNode))
                {
                    this.currentSize -= this.GetValueSize(listNode.Value.Value);

                    this.keyValueStorage.Remove(listNode);
                    this.keyValueStorage.AddFirst(listNode);

                    listNode.Value.Value = value;

                    this.currentSize += this.GetValueSize(listNode.Value.Value);
                }
                else
                {
                    var keyValue = new KeyValuePair { Key = key, Value = value };

                    listNode = this.keyValueStorage.AddFirst(keyValue);

                    this.keyValueLookup.Add(key, listNode);

                    this.currentSize += this.GetValueSize(listNode.Value.Value);
                }

                // キャッシュサイズの最大値を超えていたら、古いものから解放する。
                this.CheckCapacity();
            }
        }

        /// <summary>
        /// 値を削除します。
        /// </summary>
        /// <param name="key">値のキー</param>
        public void Remove(TKey key)
        {
            lock (this.syncObject)
            {
                LinkedListNode<KeyValuePair> listNode;

                if (!this.keyValueLookup.TryGetValue(key, out listNode))
                {
                    return;
                }

                this.OnDisposingValue(listNode.Value.Key, listNode.Value.Value);

                this.currentSize -= this.GetValueSize(listNode.Value.Value);

                this.keyValueStorage.Remove(listNode);
                this.keyValueLookup.Remove(key);
            }
        }

        /// <summary>
        /// キャッシュサイズの最大値を設定します。
        /// </summary>
        /// <param name="maxCapacity">キャッシュサイズの最大値(Byte単位)</param>
        public void SetMaxCapacity(int maxCapacity)
        {
            this.maxCapacity = maxCapacity;
            this.CheckCapacity();
        }

        /// <summary>
        /// キャッシュサイズが最大値をチェックし、超えていたら古いものから解放します。
        /// </summary>
        protected void CheckCapacity()
        {
            // キャッシュサイズの最大値を超える場合は、解放を行なう
            // キャッシュサイズの最大値を超える場合でも、キャッシュが１つしかない場合は開放処理は行わない。
            while (this.currentSize > this.maxCapacity && this.keyValueStorage.Count > 1)
            {
                var valueNode = this.keyValueStorage.Last;

                this.keyValueStorage.RemoveLast();
                this.keyValueLookup.Remove(valueNode.Value.Key);

                this.currentSize -= this.GetValueSize(valueNode.Value.Value);

                this.OnDisposingValue(valueNode.Value.Key, valueNode.Value.Value);
            }
        }

        /// <summary>
        /// 値のサイズを取得します。
        /// サイズの計算方法を変えるときは、このメソッドをオーバーライドしてください。
        /// </summary>
        /// <param name="value">値</param>
        /// <returns>値のサイズを返します。</returns>
        protected virtual int GetValueSize(TValue value)
        {
            return 1;
        }

        /// <summary>
        /// 値の解放時の処理を行います。
        /// 解放時の処理を追加するときは、このメソッドをオーバーライドしてください。
        /// </summary>
        /// <param name="key">キー</param>
        /// <param name="value">値</param>
        protected virtual void OnDisposingValue(TKey key, TValue value)
        {
        }

        /// <summary>
        /// キーと値のペアです。
        /// </summary>
        private class KeyValuePair
        {
            /// <summary>
            /// キー
            /// </summary>
            public TKey Key { get; set; }

            /// <summary>
            /// 値
            /// </summary>
            public TValue Value { get; set; }
        }
    }
}
