﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// コピー元ソースコード:
// Programs\Iris\Sources\Tools\Graphics\GraphicsTools\ShaderConverter\Sourcee\gfxTool_Main.cpp

#include <conio.h>
#include <filesystem>

#include <Windows.h>

#include <nn/util/util_BinaryFormat.h>

#include <nn/gfx/gfx_Shader.h>

#include <nn/gfxTool/gfxTool_Error.h>
#include <nn/gfxTool/gfxTool_Util.h>
#include <nn/gfxTool/gfxTool_ShaderCompilerApi.h>
#include <nn/gfxTool/gfxTool_ShaderCompilerApi-gx.h>
#include <nn/gfxTool/gfxTool_ShaderCompilerApi-nvn.h>
#include <nn/gfxTool/gfxTool_ShaderConverterApi.h>
#include <nn/gfxTool/gfxTool_ShaderUtils.h>

#include "gfxTool_ShaderConverterDll.h"
#include "gfxTool_ReadFile.h"
#include "gfxTool_Dump.h"
#include "gfxTool_Main.h"

namespace nn {
namespace gfxTool {

const char* GetShaderConverterDllName()
{
    return
#if defined(_M_IX86)
    "x86/ShaderConverter.dll";
#else
    "ShaderConverter.dll";
#endif
}

const char* s_ReflectionVariableTypeName[] =
{
    "Unknown",

    "Bool32",
    "Int8",
    "Int16",
    "Int32",
    "Int64",
    "Uint8",
    "Uint16",
    "Uint32",
    "Uint64",
    "Float16",
    "Float32",
    "Float64",

    "Bool32x2",
    "Bool32x3",
    "Bool32x4",
    "Int8x2",
    "Int8x3",
    "Int8x4",
    "Int16x2",
    "Int16x3",
    "Int16x4",
    "Int32x2",
    "Int32x3",
    "Int32x4",
    "Int64x2",
    "Int64x3",
    "Int64x4",
    "Uint8x2",
    "Uint8x3",
    "Uint8x4",
    "Uint16x2",
    "Uint16x3",
    "Uint16x4",
    "Uint32x2",
    "Uint32x3",
    "Uint32x4",
    "Uint64x2",
    "Uint64x3",
    "Uint64x4",
    "Float16x2",
    "Float16x3",
    "Float16x4",
    "Float32x2",
    "Float32x3",
    "Float32x4",
    "Float64x2",
    "Float64x3",
    "Float64x4",

    "Bool32x2x2",
    "Bool32x2x3",
    "Bool32x2x4",
    "Bool32x3x2",
    "Bool32x3x3",
    "Bool32x3x4",
    "Bool32x4x2",
    "Bool32x4x3",
    "Bool32x4x4",
    "Int32x2x2",
    "Int32x2x3",
    "Int32x2x4",
    "Int32x3x2",
    "Int32x3x3",
    "Int32x3x4",
    "Int32x4x2",
    "Int32x4x3",
    "Int32x4x4",
    "Uint32x2x2",
    "Uint32x2x3",
    "Uint32x2x4",
    "Uint32x3x2",
    "Uint32x3x3",
    "Uint32x3x4",
    "Uint32x4x2",
    "Uint32x4x3",
    "Uint32x4x4",
    "Float32x2x2",
    "Float32x2x3",
    "Float32x2x4",
    "Float32x3x2",
    "Float32x3x3",
    "Float32x3x4",
    "Float32x4x2",
    "Float32x4x3",
    "Float32x4x4",
    "Float64x2x2",
    "Float64x2x3",
    "Float64x2x4",
    "Float64x3x2",
    "Float64x3x3",
    "Float64x3x4",
    "Float64x4x2",
    "Float64x4x3",
    "Float64x4x4",

    "Sampler1D",
    "Sampler2D",
    "Sampler3D",
    "SamplerCube",
    "Sampler1DShadow",
    "Sampler2DShadow",
    "Sampler1DArray",
    "Sampler2DArray",
    "Sampler1DArrayShadow",
    "Sampler2DArrayShadow",
    "Sampler2DMultisample",
    "Sampler2DMultisampleArray",
    "SamplerCubeShadow",
    "SamplerBuffer",
    "Sampler2DRect",
    "Sampler2DRectShadow",
    "IntSampler1D",
    "IntSampler2D",
    "IntSampler3D",
    "IntSamplerCube",
    "IntSampler1DArray",
    "IntSampler2DArray",
    "IntSampler2DMultisample",
    "IntSampler2DMultisampleArray",
    "IntSamplerBuffer",
    "IntSampler2DRect",
    "UintSampler1D",
    "UintSampler2D",
    "UintSampler3D",
    "UintSamplerCube",
    "UintSampler1DArray",
    "UintSampler2DArray",
    "UintSampler2DMultisample",
    "UintSampler2DMultisampleArray",
    "UintSamplerBuffer",
    "UintSampler2DRect",
    "SamplerCubeMapArray",
    "SamplerCubeMapArrawShadow",
    "IntSamplerCubeMapArray",
    "UintSamplerCubeMapArray",

    "Image1D",
    "Image2D",
    "Image3D",
    "ImageCube",
    "Image1DArray",
    "Image2DArray",
    "Image2DMultisample",
    "Image2DMultisampleArray",
    "ImageCubeMapArray",
    "ImageBuffer",
    "Image2DRect",
    "IntImage1D",
    "IntImage2D",
    "IntImage3D",
    "IntImageCube",
    "IntImage1DArray",
    "IntImage2DArray",
    "IntImage2DMultisample",
    "IntImage2DMultisampleArray",
    "IntImageCubeMapArray",
    "IntImageBuffer",
    "IntImage2DRect",
    "UintImage1D",
    "UintImage2D",
    "UintImage3D",
    "UintImageCube",
    "UintImage1DArray",
    "UintImage2DArray",
    "UintImage2DMultisample",
    "UintImage2DMultisampleArray",
    "UintImageCubeMapArray",
    "UintImageBuffer",
    "UintImage2DRect",
};

bool WriteFileCallback( const void* pFileData, size_t fileDataSize, const char* pFilename, void* )
{
    std::ofstream ofs( pFilename, std::ios::out | std::ios::binary );
    if( ofs.is_open() )
    {
        ofs.write( static_cast< const char* >( pFileData ), fileDataSize );
        ofs.close();
        return true;
    }
    return false;
}

void DumpReflection( std::ostream* pOs, const nngfxToolShaderCompilerOptionOutputProgramCommon* pOutput )
{
    if( pOs == nullptr || pOutput == nullptr )
    {
        return;
    }

    static const nngfxToolShaderCompilerReflectionStageReference s_StageReferences[] =
    {
        nngfxToolShaderCompilerReflectionStageReference_Vertex,
        nngfxToolShaderCompilerReflectionStageReference_Hull,
        nngfxToolShaderCompilerReflectionStageReference_Domain,
        nngfxToolShaderCompilerReflectionStageReference_Geometry,
        nngfxToolShaderCompilerReflectionStageReference_Pixel,
        nngfxToolShaderCompilerReflectionStageReference_Compute
    };
    static const char* s_StageNames[] =
    {
        "VS", "HS", "DS", "GS", "PS", "CS"
    };

    auto WriteName = []( std::ostream* pOs, const nngfxToolString& name )
    {
        *pOs << "\n\tName: ";
        pOs->write( name.pValue, name.length );
    };

    auto WriteStages = [ & ]( std::ostream* pOs, int32_t stages )
    {
        *pOs << "\n\tStages:";
        for( int idxStage = 0; idxStage < nn::gfx::ShaderStage_End; ++idxStage )
        {
            if( stages & s_StageReferences[ idxStage ] )
            {
                *pOs << " " << s_StageNames[ idxStage ];
            }
        }
    };

    auto WriteSlots = [ & ]( std::ostream* pOs, const nngfxToolShaderCompilerShaderSlot& slot, int32_t stages )
    {
        static int32_t const nngfxToolShaderCompilerShaderSlot::* const s_pStageSlots[] =
        {
            &nngfxToolShaderCompilerShaderSlot::vertexShaderSlot,
            &nngfxToolShaderCompilerShaderSlot::hullShaderSlot,
            &nngfxToolShaderCompilerShaderSlot::domainShaderSlot,
            &nngfxToolShaderCompilerShaderSlot::geometryShaderSlot,
            &nngfxToolShaderCompilerShaderSlot::pixelShaderSlot,
            &nngfxToolShaderCompilerShaderSlot::computeShaderSlot
        };

        *pOs << "\n\tSlot:";
        for( int idxStage = 0; idxStage < nn::gfx::ShaderStage_End; ++idxStage )
        {
            if( stages & s_StageReferences[ idxStage ] )
            {
                *pOs << "\n\t\t" << s_StageNames[ idxStage ] << ": " << slot.*s_pStageSlots[ idxStage ];
            }
        }
    };

    if( auto pReflection = pOutput->pReflection )
    {
        auto& os = *pOs;
        for( int idxConstantBuffer = 0; idxConstantBuffer < static_cast< int >(
            pReflection->constantBufferCount ); ++idxConstantBuffer )
        {
            auto& target = pReflection->pConstantBufferArray[ idxConstantBuffer ];
            os << "ConstantBuffer " << idxConstantBuffer;
            WriteName( &os, target.name );
            WriteSlots( &os, target.shaderSlot, target.stages );
            os << "\n\tSize: " << target.size << "\n\tActiveVariableCount: " << target.activeVariableCount;
            WriteStages( &os, target.stages );
            os << "\n\n";
        }
        for( int idxUnorderedAccessBuffer = 0; idxUnorderedAccessBuffer < static_cast< int >(
            pReflection->unorderedAccessBufferCount ); ++idxUnorderedAccessBuffer )
        {
            auto& target = pReflection->pUnorderedAccessBufferArray[ idxUnorderedAccessBuffer ];
            os << "UnorderedAccessBuffer " << idxUnorderedAccessBuffer;
            WriteName( &os, target.name );
            WriteSlots( &os, target.shaderSlot, target.stages );
            os << "\n\tSize: " << target.size << "\n\tActiveVariableCount: " << target.activeVariableCount;
            WriteStages( &os, target.stages );
            os << "\n\n";
        }
        for( int idxShaderInput = 0; idxShaderInput < static_cast< int >(
            pReflection->shaderInputCount ); ++idxShaderInput )
        {
            auto& target = pReflection->pShaderInputArray[ idxShaderInput ];
            os << "ShaderInput " << idxShaderInput;
            WriteName( &os, target.name );
            os << "\n\tType: " << s_ReflectionVariableTypeName[ target.type ]
                << "\n\tShaderSlot: " << target.shaderSlot << "\n\tArrayCount: " << target.arrayCount;
            WriteStages( &os, target.stages );
            os << "\n\n";
        }
        for( int idxShaderOutput = 0; idxShaderOutput < static_cast< int >(
            pReflection->shaderOutputCount ); ++idxShaderOutput )
        {
            auto& target = pReflection->pShaderOutputArray[ idxShaderOutput ];
            os << "ShaderOutput " << idxShaderOutput;
            WriteName( &os, target.name );
            os << "\n\tShaderSlot: " << target.shaderSlot << "\n\tArrayCount: " << target.arrayCount;
            WriteStages( &os, target.stages );
            os << "\n\n";
        }
        for( int idxSampler = 0; idxSampler < static_cast< int >( pReflection->samplerCount ); ++idxSampler )
        {
            auto& target = pReflection->pSamplerArray[ idxSampler ];
            os << "Sampler " << idxSampler;
            WriteName( &os, target.name );
            os << "\n\tType: " << s_ReflectionVariableTypeName[ target.type ];
            WriteSlots( &os,target.shaderSlot, target.stages );
            WriteStages( &os, target.stages );
            os << "\n\n";
        }
        for( int idxConstantBufferVariable = 0; idxConstantBufferVariable < static_cast< int >(
            pReflection->constantBufferVariableCount ); ++idxConstantBufferVariable )
        {
            auto& target = pReflection->pConstantBufferVariableArray[ idxConstantBufferVariable ];
            os << "ConstantBufferVariable " << idxConstantBufferVariable;
            WriteName( &os, target.name );
            os << "\n\tType: " << s_ReflectionVariableTypeName[ target.type ];
            if( target.blockIndex >= 0 && target.blockIndex < static_cast<
                decltype( target.blockIndex ) >( pReflection->constantBufferCount ) )
            {
                auto& block = pReflection->pConstantBufferArray[ target.blockIndex ];
                os << "\n\tBlock: ";
                os.write( block.name.pValue, block.name.length );
            }
            os << "\n\tOffset: " << target.offset << "\n\tArrayCount: " << target.arrayCount;
            WriteStages( &os, target.stages );
            os << "\n\n";
        }
        for( int idxUnorderedAccessBufferVariable = 0; idxUnorderedAccessBufferVariable < static_cast< int >(
            pReflection->unorderedAccessBufferVariableCount ); ++idxUnorderedAccessBufferVariable )
        {
            auto& target = pReflection->pUnorderedAccessBufferVariableArray[ idxUnorderedAccessBufferVariable ];
            os << "UnorderedAccessBufferVariable " << idxUnorderedAccessBufferVariable;
            WriteName( &os, target.name );
            os << "\n\tType: " << s_ReflectionVariableTypeName[ target.type ];
            if( target.blockIndex < static_cast< decltype( target.blockIndex ) >(
                pReflection->unorderedAccessBufferCount ) )
            {
                auto& block = pReflection->pUnorderedAccessBufferArray[ target.blockIndex ];
                os << "\n\tBlock: ";
                os.write( block.name.pValue, block.name.length );
            }
            os << "\n\tOffset: " << target.offset << "\n\tArrayCount: " << target.arrayCount;
            WriteStages( &os, target.stages );
            os << "\n\n";
        }
        for( int idxImage = 0; idxImage < static_cast< int >( pReflection->imageCount ); ++idxImage )
        {
            auto& target = pReflection->pImageArray[ idxImage ];
            os << "Image " << idxImage;
            WriteName( &os, target.name );
            os << "\n\tType: " << s_ReflectionVariableTypeName[ target.type ];
            WriteSlots( &os,target.shaderSlot, target.stages );
            WriteStages( &os, target.stages );
            os << "\n\n";
        }
    }
} // NOLINT

void DumpProgram( std::ostream* pOs, const nngfxToolShaderConverterConvertArg* pArg,
    const nngfxToolShaderCompilerOptionOutputProgramCommon* pOutput, nn::gfx::ShaderStage stage )
{
    static nngfxToolShaderCompilerOptionOutputStageCommon*
        nngfxToolShaderCompilerOptionOutputProgramCommon::* const s_pStageOutput[] =
    {
        &nngfxToolShaderCompilerOptionOutputProgramCommon::pVertexStageOutput,
        &nngfxToolShaderCompilerOptionOutputProgramCommon::pHullStageOutput,
        &nngfxToolShaderCompilerOptionOutputProgramCommon::pDomainStageOutput,
        &nngfxToolShaderCompilerOptionOutputProgramCommon::pGeometryStageOutput,
        &nngfxToolShaderCompilerOptionOutputProgramCommon::pPixelStageOutput,
        &nngfxToolShaderCompilerOptionOutputProgramCommon::pComputeStageOutput
    };

    if( pOs == nullptr || pOutput == nullptr )
    {
        return;
    }

    if( auto pOutputStageCommon = pOutput->*s_pStageOutput[ stage ] )
    {
        auto& os = *pOs;
        const char* const separator = "------------------------";
        if( auto pShaderStatistics = pOutputStageCommon->pShaderStatistics )
        {
            os << separator << "Shader Statistics" << separator << "\n";
            if( pArg->pCompileArg->targetLowLevelApiType == nngfxToolShaderCompilerLowLevelApiType_Gx )
            {
                nn::gfxTool::DumpStatistics< static_cast< nngfxToolShaderCompilerLowLevelApiType >(
                    nngfxToolShaderCompilerLowLevelApiType_Gx ) >( &os, pShaderStatistics );
            }
            if( pArg->pCompileArg->targetLowLevelApiType == nngfxToolShaderCompilerLowLevelApiType_Nvn )
            {
                nn::gfxTool::DumpStatistics< static_cast< nngfxToolShaderCompilerLowLevelApiType >(
                    nngfxToolShaderCompilerLowLevelApiType_Nvn ) >( &os, pShaderStatistics );
            }
        }
        if( pOutputStageCommon->dump.pValue )
        {
            os << separator << "Dump" << separator << "\n\n";
            os.write( pOutputStageCommon->dump.pValue, pOutputStageCommon->dump.length );
            os << "\n\n";
        }
        if( pOutputStageCommon->infoLog.pValue && pOutputStageCommon->infoLog.length > 0 )
        {
            os << separator << "Info Log" << separator << "\n";
            os.write( pOutputStageCommon->infoLog.pValue, pOutputStageCommon->infoLog.length );
            os << "\n\n";
        }
    }
}

void Dump( const std::string& dumpDirectory, const nngfxToolShaderConverterConvertArg* pArg,
    const nngfxToolShaderCompilerCompileOutput* pOutput )
{
    if( pOutput == nullptr )
    {
        return;
    }

    static const char* s_pStageNames[] =
    {
        "vs", "hs", "ds", "gs", "ps", "cs"
    };

    static nngfxToolShaderCompilerShaderProgramOutput*
        nngfxToolShaderCompilerVariationOutput::*s_pProgramOutputs[] =
    {
        &nngfxToolShaderCompilerVariationOutput::pSourceOutput,
        &nngfxToolShaderCompilerVariationOutput::pIntermediateLanguageOutput,
        &nngfxToolShaderCompilerVariationOutput::pBinaryOutput
    };
    static const char* s_pProgramNames[] =
    {
        "src", "il", "bin"
    };

    char path[ MAX_PATH ];
    char format[ MAX_PATH ];
    char formatReflection[ MAX_PATH ];
    auto variationCount = static_cast< int >( pOutput->variationCount );
    auto digit = static_cast< int >( log10( std::max NN_PREVENT_MACRO_FUNC ( variationCount - 1, 1 ) ) ) + 1;
    sprintf_s( format, "%%s%%0%dd_%%s_%%s.glsl", digit );
    sprintf_s( formatReflection, "%%s%%0%dd_reflection.txt", digit );
    for( int idxVariation = 0; idxVariation < variationCount; ++idxVariation )
    {
        auto& variationOutput = pOutput->pVariationOutputArray[ idxVariation ];
        for( int idxProgram = 0; idxProgram < ARRAYSIZE( s_pProgramOutputs ); ++idxProgram )
        {
            if( auto pProgramOutput = variationOutput.*( s_pProgramOutputs[ idxProgram ] ) )
            {
                const nngfxToolShaderCompilerOptionOutputProgramCommon* pOutputProgramCommon = nullptr;
                const nngfxToolShaderCompilerOptionOutputProgramNvn* pOutputProgramNvn = nullptr;
                for( int idxOption = 0; idxOption < static_cast< int >(
                    pProgramOutput->optionOutputCount ); ++idxOption )
                {
                    auto& option = pProgramOutput->pOptionOutputArray[ idxOption ];
                    switch( option.optionOutputType )
                    {
                    case nngfxToolShaderCompilerOptionOutputType_ProgramCommon:
                        {
                            pOutputProgramCommon = nn::gfxTool::StaticCastAuto( option.pOutput );
                        }
                        break;
                    case nngfxToolShaderCompilerOptionOutputType_ProgramNvn:
                        {
                            pOutputProgramNvn = nn::gfxTool::StaticCastAuto( option.pOutput );
                        }
                        break;
                    default:
                        break;
                    }
                }
                if( pOutputProgramCommon )
                {
                    for( int idxStage = 0; idxStage < nn::gfx::ShaderStage_End; ++idxStage )
                    {
                        auto stage = static_cast< nn::gfx::ShaderStage >( idxStage );
                        std::stringstream ss;
                        if( pOutputProgramNvn )
                        {
                            nn::gfxTool::DumpOptionOutputProgram< static_cast<
                                nngfxToolShaderCompilerOptionOutputType >(
                                nngfxToolShaderCompilerOptionOutputType_ProgramNvn ) >(
                                &ss, pOutputProgramNvn, stage );
                        }
                        DumpProgram( &ss, pArg, pOutputProgramCommon, stage );
                        if( ss.rdbuf()->in_avail() )
                        {
                            sprintf_s( path, format, dumpDirectory.data(),
                                idxVariation, s_pStageNames[ idxStage ], s_pProgramNames[ idxProgram ] );
                            std::ofstream ofs( path, std::ios::out | std::ios::binary );
                            if( !ofs.fail() )
                            {
                                ofs << ss.rdbuf();
                            }
                        }
                    }
                    std::stringstream ss;
                    DumpReflection( &ss, pOutputProgramCommon );
                    if( ss.rdbuf()->in_avail() )
                    {
                        sprintf_s( path, formatReflection, dumpDirectory.c_str(), idxVariation, "", "reflection" );
                        std::ofstream ofs( path );
                        if( !ofs.fail() )
                        {
                            ofs << ss.rdbuf();
                        }
                    }
                }
            }
        }
    }
} // NOLINT

nngfxToolResultCode Execute( int argc, char** argv, ShaderConverterDll& shaderConverterDll )
{
    #if 0
    nn::gfxTool::ShaderConverterDll shaderConverterDll;
    auto path = std::tr2::sys::path( nn::gfxTool::GetModulePath(
        GetModuleHandle( nullptr ) ) ).parent_path().string();
    path.append( "/" ).append( GetShaderConverterDllName() );
    shaderConverterDll.Initialize( path.c_str() );
    #endif

    nngfxToolResultCode result;

    nngfxToolShaderConverterCreateHandleArg createHandleArg;
    nngfxToolHandle hShaderConverter = 0;
    result = shaderConverterDll.CreateHandle( &hShaderConverter, &createHandleArg );
    if( result != nngfxToolResultCode_Succeeded )
    {
        NN_GFXTOOL_PRINT_ERROR( "Failed to create handle.\n" );
        return result;
    }

    nngfxToolShaderConverterCreateConvertArgArg createConvertArgArg = {};
    nn::gfxTool::ReadFile readFile;
    createConvertArgArg.pReadInputFileCallback = nn::gfxTool::ReadFile::Callback;
    createConvertArgArg.pReadInputFileCallbackParam = &readFile;
    createConvertArgArg.pReadIncludeFileCallback = nn::gfxTool::ReadFile::Callback;
    createConvertArgArg.pReadIncludeFileCallbackParam = &readFile;
    createConvertArgArg.pWriteDebugInfoFileCallback = WriteFileCallback;
    createConvertArgArg.pWriteDebugInfoFileCallbackParam = nullptr;
    createConvertArgArg.pReadShaderCacheCallback = nn::gfxTool::ReadFile::Callback;
    createConvertArgArg.pReadShaderCacheCallbackParam = &readFile;
    createConvertArgArg.pWriteShaderCacheCallback = WriteFileCallback;
    createConvertArgArg.pWriteShaderCacheCallbackParam = nullptr;
    createConvertArgArg.optionCount = argc;
    createConvertArgArg.ppOptionArray = argv;
    nngfxToolShaderConverterConvertArg* pConvertArg;
    result = shaderConverterDll.CreateConvertArg( &pConvertArg, &createConvertArgArg );
    if( result != nngfxToolResultCode_Succeeded )
    {
        NN_GFXTOOL_PRINT_ERROR( "Failed to parse command-line options.\n" );
        shaderConverterDll.DeleteHandle( hShaderConverter );
        return result;
    }
    // TODO: パーサーを乗り換えたら別の場所に移す
    for( int idxOption = 0, optionCount = nn::gfxTool::NumericCastAuto(
        pConvertArg->pCompileArg->optionCount  ); idxOption < optionCount; ++idxOption )
    {
        if( pConvertArg->pCompileArg->pOptionArray[ idxOption ].optionType
            == nngfxToolShaderCompilerOptionType_Common )
        {
            if( static_cast< const nngfxToolShaderCompilerCompileOptionCommon* >(
                pConvertArg->pCompileArg->pOptionArray[ idxOption ].pOption )->logOutputLevel ==
                nngfxToolShaderCompilerLogOutputLevel_Silent )
            {
                nn::gfxTool::Logger::SetLogOutputLevel(
                    nn::gfxTool::Logger::LogType::Default, nn::gfxTool::Logger::LogLevel::Silent );
            }
        }
    }

    auto& information = *pConvertArg->pInformation;
    std::string outputPath( information.outputPath.pValue, information.outputPath.length );
    std::string dumpDirectory( information.dumpDirectory.pValue, information.dumpDirectory.length );

    nngfxToolShaderConverterConvertOutput convertOutput;
    result = shaderConverterDll.Convert( &convertOutput, hShaderConverter, pConvertArg );
    if( result != nngfxToolResultCode_Succeeded )
    {
        NN_GFXTOOL_PRINT_ERROR( "Conversion failed.\n" );
        shaderConverterDll.DeleteConvertArg( pConvertArg );
        shaderConverterDll.DeleteHandle( hShaderConverter );
        return result;
    }

    if( !dumpDirectory.empty() )
    {
        Dump( dumpDirectory, pConvertArg, convertOutput.pCompileOutput );
    }

    result = shaderConverterDll.DeleteConvertArg( pConvertArg );
    if( result != nngfxToolResultCode_Succeeded )
    {
        NN_GFXTOOL_PRINT_ERROR( "Failed to delete conversion arguments.\n" );
        shaderConverterDll.DeleteHandle( hShaderConverter );
        return result;
    }

    std::unique_ptr< char[] > pBinary( new char[ convertOutput.binarySize ] );
    result = shaderConverterDll.Serialize( pBinary.get(), convertOutput.binarySize, hShaderConverter );
    if( result != nngfxToolResultCode_Succeeded )
    {
        NN_GFXTOOL_PRINT_ERROR( "Serialization failed.\n" );
        shaderConverterDll.DeleteHandle( hShaderConverter );
        return result;
    }

    result = shaderConverterDll.DeleteHandle( hShaderConverter );
    if( result != nngfxToolResultCode_Succeeded )
    {
        NN_GFXTOOL_PRINT_ERROR( "Failed to delete handle.\n" );
        return result;
    }

    auto pBinaryFileHeader = reinterpret_cast< nn::util::BinaryFileHeader* >( pBinary.get() );
    auto fileSize = pBinaryFileHeader->GetFileSize();

    FILE* fp = nullptr;
    fopen_s( &fp, outputPath.data(), "wb" );
    if( fp )
    {
        fwrite( pBinary.get(), 1, fileSize, fp );
        fclose( fp );
        NN_GFXTOOL_PRINT( "Converted: %s\n", outputPath.c_str() );
    }
    else
    {
        NN_GFXTOOL_PRINT_ERROR( "Failed to output binary.\n" );
        return nngfxToolResultCode_Failed;
    }

    return nngfxToolResultCode_Succeeded;
} // NOLINT

}
}

#if false
int main( int argc, char** argv )
{
    auto hModule = GetModuleHandle( nullptr );
    char moduleName[ MAX_PATH ];
    GetModuleFileNameA( hModule, moduleName, MAX_PATH );
    char filename[ MAX_PATH ];
    char ext[ MAX_PATH ];
    _splitpath_s( moduleName, nullptr, 0, nullptr, 0, filename, MAX_PATH, ext, MAX_PATH );

    // TODO: silent を前にもってこれた際に復活させる
    // NN_GFXTOOL_PRINT( "%s%s Start\n", filename, ext );

    bool isPerformanceCounterEnable = true;
    LARGE_INTEGER start;
    isPerformanceCounterEnable &= ::QueryPerformanceCounter( &start ) != 0;

    auto result = Execute( argc - 1, argv + 1 );

    if( result == nngfxToolResultCode_Succeeded )
    {
        LARGE_INTEGER end;
        isPerformanceCounterEnable &= ::QueryPerformanceCounter( &end ) != 0;

        LARGE_INTEGER frequency;
        isPerformanceCounterEnable &= ::QueryPerformanceFrequency( &frequency ) != 0;

        NN_GFXTOOL_PRINT( "%s%s Finish", filename, ext );
        if( isPerformanceCounterEnable && frequency.QuadPart )
        {
            NN_GFXTOOL_PRINT( ": %lld ms", ( end.QuadPart - start.QuadPart ) * 1000 / frequency.QuadPart );
        }
        NN_GFXTOOL_PRINT( "\n" );
    }

#if defined( _DEBUG )
    _getch();
#endif

    return static_cast< int >( result );
}
#endif
