﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "ShaderConverterWrapper_PCH.h"
#include "ShaderConverterHelper.h"
#include "ShaderCodeListCs.h"

#include <cassert>

using namespace std;

using namespace System::Runtime::InteropServices;
using namespace System::Text;

using namespace EffectMaker::ShaderConverterLib;


namespace EffectMaker {
namespace ShaderConverterWrapper {

    static char* CsStringToCppChar(String ^pStr, u32 *pSize);

    /// <summary>
    /// Constructor.
    /// </summary>
    /// <param name="converterFolderPath">The converter folder path.</param>
    /// <param name="shaderFolderPath">The shader folder path.</param>
    /// <param name="pAppMsgForwarder">The message forwarder to forward message to the applicaiton.</param>
    /// <param name="isCommandLineMode">The flag indicating whether the application is running under command line.</param>
    /// <param name="shaderAlignment">シェーダアライメント</param>
    /// <param name="binaryPosition">バイナリ書き込み位置</param>
    ShaderConverterHelper::ShaderConverterHelper(String ^converterFolderPath,
                                                 String ^shaderFolderPath,
                                                 IDotNetMessageForwarder ^pAppMsgForwarder,
                                                 int convertMode,
                                                 bool isCommandLineMode,
                                                 int jobsNumber,
                                                 int shaderAlignment,
                                                 int binaryPosition,
                                                 bool isSubBinary ) :
        m_pConverter(NULL),
        m_pNativeMsgForwarder(NULL),
        m_pShaderCodes(NULL),
        m_inputEmitterCount(0),
        m_pInputBuffer(NULL),
        m_bCommandLineMode(convertMode),
        m_jobsNumber(jobsNumber),
        m_shaderAlignment(shaderAlignment),
        m_binaryPosition(binaryPosition),
        m_isSubBinaryConverting(isSubBinary)
    {
        // Create the buffer for shader binary data.
        this->m_pShaderBinaryBuffer = NULL;

        // The folder paths have to end with '\', append if they don't.
        if (converterFolderPath->EndsWith("\\") == false)
        {
            converterFolderPath += "\\";
        }

        if (shaderFolderPath->EndsWith("\\") == false)
        {
            shaderFolderPath += "\\";
        }

        // Save the pointers of the folder paths, we will release them in the destructor.
        IntPtr strPtr = Marshal::StringToHGlobalAnsi(converterFolderPath);
        this->m_szConverterFolderPath = new char[Encoding::GetEncoding("Shift_JIS")->GetByteCount(converterFolderPath) + 1];
        strcpy_s(
            this->m_szConverterFolderPath,
            Encoding::GetEncoding("Shift_JIS")->GetByteCount(converterFolderPath) + 1,
            (const char*)strPtr.ToPointer());

        Marshal::FreeHGlobal(strPtr);

        strPtr = Marshal::StringToHGlobalAnsi(shaderFolderPath);
        this->m_szShaderFolderPath = new char[Encoding::GetEncoding("Shift_JIS")->GetByteCount(shaderFolderPath) + 1];
        strcpy_s(
            this->m_szShaderFolderPath,
            Encoding::GetEncoding("Shift_JIS")->GetByteCount(shaderFolderPath) + 1,
            (const char*)strPtr.ToPointer());

        Marshal::FreeHGlobal(strPtr);

        // Create the logger and assign it to the shader converter.
        m_pNativeMsgForwarder = new NativeMessageForwarder(pAppMsgForwarder);
        EffectMaker::ShaderConverterLib::SetMessageForwarder(m_pNativeMsgForwarder);

        // Set CommandLine Mode
        m_bCommandLineMode |= isCommandLineMode << 16;

        // Create the shader converter.
        m_pConverter = new ShaderConverter(
            this->m_szConverterFolderPath,
            this->m_szShaderFolderPath,
            this->m_bCommandLineMode,
            m_jobsNumber,
            m_shaderAlignment,
            m_binaryPosition,
            this->m_isSubBinaryConverting);
    }

    /// <summary>
    /// Destructor.
    /// </summary>
    ShaderConverterHelper::~ShaderConverterHelper()
    {
        this->!ShaderConverterHelper();
    }

    /// <summary>
    /// Finalize.
    /// </summary>
    ShaderConverterHelper::!ShaderConverterHelper()
    {
        // Release the converter.
        if (this->m_pConverter != NULL)
        {
            delete this->m_pConverter;
            this->m_pConverter = NULL;
        }

        // Release the message forwarder.
        if (this->m_pNativeMsgForwarder != NULL)
        {
            delete this->m_pNativeMsgForwarder;
            this->m_pNativeMsgForwarder = NULL;
        }

        // Release the folder path strings.
        if (this->m_szConverterFolderPath != NULL)
        {
            delete[] this->m_szConverterFolderPath;
            this->m_szConverterFolderPath = NULL;
        }

        if (this->m_szShaderFolderPath != NULL)
        {
            delete[] this->m_szShaderFolderPath;
            this->m_szShaderFolderPath = NULL;
        }

        // Release the buffers.
        this->ReleaseInputDataBuffer();
        this->ReleaseShaderBinaryBuffer();

        // Release the shader codes.
        if (this->m_pShaderCodes != NULL)
        {
            if (this->m_pShaderCodes->mSpecDecShaderSrc.buf != NULL)
            {
                delete[] this->m_pShaderCodes->mSpecDecShaderSrc.buf;
                this->m_pShaderCodes->mSpecDecShaderSrc.buf = NULL;
                this->m_pShaderCodes->mSpecDecShaderSrc.size = 0;
            }

            if (this->m_pShaderCodes->mVertexShaderSrc.buf != NULL)
            {
                delete[] this->m_pShaderCodes->mVertexShaderSrc.buf;
                this->m_pShaderCodes->mVertexShaderSrc.buf = NULL;
                this->m_pShaderCodes->mVertexShaderSrc.size = 0;
            }

            if (this->m_pShaderCodes->mFragShaderSrc.buf != NULL)
            {
                delete[] this->m_pShaderCodes->mFragShaderSrc.buf;
                this->m_pShaderCodes->mFragShaderSrc.buf = NULL;
                this->m_pShaderCodes->mFragShaderSrc.size = 0;
            }

            if (this->m_pShaderCodes->mParticleDecVshSrc.buf != NULL)
            {
                delete[] this->m_pShaderCodes->mParticleDecVshSrc.buf;
                this->m_pShaderCodes->mParticleDecVshSrc.buf = NULL;
                this->m_pShaderCodes->mParticleDecVshSrc.size = 0;
            }

            if (this->m_pShaderCodes->mParticleDecFshSrc.buf != NULL)
            {
                delete[] this->m_pShaderCodes->mParticleDecFshSrc.buf;
                this->m_pShaderCodes->mParticleDecFshSrc.buf = NULL;
                this->m_pShaderCodes->mParticleDecFshSrc.size = 0;
            }

            if (this->m_pShaderCodes->mStreamOutDecVshSrc.buf != NULL)
            {
                delete[] this->m_pShaderCodes->mStreamOutDecVshSrc.buf;
                this->m_pShaderCodes->mStreamOutDecVshSrc.buf = NULL;
                this->m_pShaderCodes->mStreamOutDecVshSrc.size = 0;
            }

            if (this->m_pShaderCodes->mStreamOutVshSrc.buf != NULL)
            {
                delete[] this->m_pShaderCodes->mStreamOutVshSrc.buf;
                this->m_pShaderCodes->mStreamOutVshSrc.buf = NULL;
                this->m_pShaderCodes->mStreamOutVshSrc.size = 0;
            }

            if (this->m_pShaderCodes->mGeneralVshSrc.buf != NULL)
            {
                delete[] this->m_pShaderCodes->mGeneralVshSrc.buf;
                this->m_pShaderCodes->mGeneralVshSrc.buf = NULL;
                this->m_pShaderCodes->mGeneralVshSrc.size = 0;
            }

            if (this->m_pShaderCodes->mGeneralFshSrc.buf != NULL)
            {
                delete[] this->m_pShaderCodes->mGeneralFshSrc.buf;
                this->m_pShaderCodes->mGeneralFshSrc.buf = NULL;
                this->m_pShaderCodes->mGeneralFshSrc.size = 0;
            }

            if (this->m_pShaderCodes->mGeneralCshSrc.buf != NULL)
            {
                delete[] this->m_pShaderCodes->mGeneralCshSrc.buf;
                this->m_pShaderCodes->mGeneralCshSrc.buf = NULL;
                this->m_pShaderCodes->mGeneralCshSrc.size = 0;
            }

            for (int i = 0; i < 8; ++i)
            {
                if (this->m_pShaderCodes->mCustomVshSrc[i].buf != NULL)
                {
                    delete[] this->m_pShaderCodes->mCustomVshSrc[i].buf;
                    this->m_pShaderCodes->mCustomVshSrc[i].buf = NULL;
                    this->m_pShaderCodes->mCustomVshSrc[i].size = 0;
                }

                if (this->m_pShaderCodes->mCustomFshSrc[i].buf != NULL)
                {
                    delete[] this->m_pShaderCodes->mCustomFshSrc[i].buf;
                    this->m_pShaderCodes->mCustomFshSrc[i].buf = NULL;
                    this->m_pShaderCodes->mCustomFshSrc[i].size = 0;
                }
            }

            delete m_pShaderCodes;
            m_pShaderCodes = NULL;
        }
    }

    /// <summary>
    /// シェーダコードを設定します.
    /// </summary>
    /// <param name="pShaderCodes">シェーダコードです.</param>
    void ShaderConverterHelper::SetShaderCodes(ShaderCodeListCs^ pShaderCodes)
    {
        assert(m_pShaderCodes == NULL);

        m_pShaderCodes = new ShaderCodeListCpp();

        if (pShaderCodes->SpecDecSrc != nullptr)
        {
            this->m_pShaderCodes->mSpecDecShaderSrc.buf = CsStringToCppChar(
                pShaderCodes->SpecDecSrc,
                (u32*)&this->m_pShaderCodes->mSpecDecShaderSrc.size);
        }

        if (pShaderCodes->VertexShaderSrc != nullptr)
        {
            this->m_pShaderCodes->mVertexShaderSrc.buf = CsStringToCppChar(
                pShaderCodes->VertexShaderSrc,
                (u32*)&this->m_pShaderCodes->mVertexShaderSrc.size);
        }

        if (pShaderCodes->ParticleGlslSrc != nullptr)
        {
            this->m_pShaderCodes->mParticleGlslSrc.buf = CsStringToCppChar(
                pShaderCodes->ParticleGlslSrc,
                (u32*)&this->m_pShaderCodes->mParticleGlslSrc.size);
        }

        if (pShaderCodes->FragShaderSrc != nullptr)
        {
            this->m_pShaderCodes->mFragShaderSrc.buf = CsStringToCppChar(
                pShaderCodes->FragShaderSrc,
                (u32*)&this->m_pShaderCodes->mFragShaderSrc.size);
        }

        if (pShaderCodes->ParticleDecVshSrc != nullptr)
        {
            this->m_pShaderCodes->mParticleDecVshSrc.buf = CsStringToCppChar(
                pShaderCodes->ParticleDecVshSrc,
                (u32*)&this->m_pShaderCodes->mParticleDecVshSrc.size);
        }

        if (pShaderCodes->ParticleDecFshSrc != nullptr)
        {
            this->m_pShaderCodes->mParticleDecFshSrc.buf = CsStringToCppChar(
                pShaderCodes->ParticleDecFshSrc,
                (u32*)&this->m_pShaderCodes->mParticleDecFshSrc.size);
        }

        if (pShaderCodes->StripShaderSrc != nullptr)
        {
            this->m_pShaderCodes->mStripShaderSrc.buf = CsStringToCppChar(
                pShaderCodes->StripShaderSrc,
                (u32*)&this->m_pShaderCodes->mStripShaderSrc.size);
        }

        if (pShaderCodes->GeneralVshSrc != nullptr)
        {
            this->m_pShaderCodes->mGeneralVshSrc.buf = CsStringToCppChar(
                pShaderCodes->GeneralVshSrc,
                (u32*)&this->m_pShaderCodes->mGeneralVshSrc.size);
        }

        if (pShaderCodes->GeneralFshSrc != nullptr)
        {
            this->m_pShaderCodes->mGeneralFshSrc.buf = CsStringToCppChar(
                pShaderCodes->GeneralFshSrc,
                (u32*)&this->m_pShaderCodes->mGeneralFshSrc.size);
        }

        if (pShaderCodes->GeneralCshSrc != nullptr)
        {
            this->m_pShaderCodes->mGeneralCshSrc.buf = CsStringToCppChar(
                pShaderCodes->GeneralCshSrc,
                (u32*)&this->m_pShaderCodes->mGeneralCshSrc.size);
        }

        if (pShaderCodes->StreamOutDecVshSrc != nullptr)
        {
            this->m_pShaderCodes->mStreamOutDecVshSrc.buf = CsStringToCppChar(
                pShaderCodes->StreamOutDecVshSrc,
                (u32*)&this->m_pShaderCodes->mStreamOutDecVshSrc.size);
        }

        if (pShaderCodes->StreamOutVshSrc != nullptr)
        {
            this->m_pShaderCodes->mStreamOutVshSrc.buf = CsStringToCppChar(
                pShaderCodes->StreamOutVshSrc,
                (u32*)&this->m_pShaderCodes->mStreamOutVshSrc.size);
        }

        for (int i = 0; i < 8; ++i)
        {
            if (pShaderCodes->CustomVshSrc != nullptr &&
                pShaderCodes->CustomVshSrc[i] != nullptr)
            {
                this->m_pShaderCodes->mCustomVshSrc[i].buf = CsStringToCppChar(
                    pShaderCodes->CustomVshSrc[i],
                    (u32*)&this->m_pShaderCodes->mCustomVshSrc[i].size);
            }

            if (pShaderCodes->CustomFshSrc != nullptr &&
                pShaderCodes->CustomFshSrc[i] != nullptr)
            {
                this->m_pShaderCodes->mCustomFshSrc[i].buf = CsStringToCppChar(
                    pShaderCodes->CustomFshSrc[i],
                    (u32*)&this->m_pShaderCodes->mCustomFshSrc[i].size);
            }
        }

        for (int i = 0; i < 8; ++i)
        {
            if (pShaderCodes->ReservedVshSrc != nullptr &&
                pShaderCodes->ReservedVshSrc[i] != nullptr)
            {
                this->m_pShaderCodes->mReservedVshSrc[i].buf = CsStringToCppChar(
                    pShaderCodes->ReservedVshSrc[i],
                    (u32*)&this->m_pShaderCodes->mReservedVshSrc[i].size);
            }

            if (pShaderCodes->ReservedFshSrc != nullptr &&
                pShaderCodes->ReservedFshSrc[i] != nullptr)
            {
                this->m_pShaderCodes->mReservedFshSrc[i].buf = CsStringToCppChar(
                    pShaderCodes->ReservedFshSrc[i],
                    (u32*)&this->m_pShaderCodes->mReservedFshSrc[i].size);
            }
        }
    }

    /// <summary>
    /// Convert shader binary.
    /// </summary>
    /// <param name="pDataArray">The array of input emitter data.</param>
    /// <param name="pShaderBinary">The output shader binary data.</param>
    /// <param name="pErrorList">The array of compile error information.</param>
    /// <param name="shaderCount">The number of shaders being compiled.</param>
    /// <returns>True on success.</returns>
    bool ShaderConverterHelper::Convert(cli::array<ShaderConversionInputData^> ^pDataArray,
                                        [Out] cli::array<System::Byte>^% pShaderBinary,
                                        [Out] List<ShaderCompileErrorInfo^>^% pErrorList,
                                        [Out] int% shaderCount)
    {
        pErrorList    = gcnew List<ShaderCompileErrorInfo^>();
        pShaderBinary = nullptr;

        if (this->m_pConverter == NULL)
        {
            return false;
        }

        // メモリの解放忘れがあったらやっておく
        if ( this->m_pShaderBinaryBuffer != NULL )
        {
            delete[] this->m_pShaderBinaryBuffer;
        }

        this->ConvertInputData(pDataArray);
        if (this->m_pInputBuffer == NULL)
        {
            return false;
        }

        if (this->m_pShaderCodes == NULL)
        {
            return false;
        }

        this->m_pConverter->Initialize(m_pShaderCodes);

        ShaderCompileErrorList errorList;
        int outputDataSize = 0;
        int nativeShaderCount = 0;

        this->m_pShaderBinaryBuffer = this->m_pConverter->Convert(
            (ShaderConverterEmitterData*)this->m_pInputBuffer,
            &errorList,
            this->m_inputEmitterCount,
            &outputDataSize,
            &nativeShaderCount);

        shaderCount = nativeShaderCount;

        bool isWarning;

        const char *szErrorEmitterName    = NULL;
        const char *szErrorVertexShader   = NULL;
        const char *szErrorFragmentShader = NULL;
        const char *szErrorLog            = NULL;

        // The compile error information.
        errorList.BeginEnumeration();
        while (errorList.GetNextItem(&isWarning,
                                     szErrorEmitterName,
                                     szErrorVertexShader,
                                     szErrorFragmentShader,
                                     szErrorLog) == true)
        {
            pErrorList->Add(gcnew ShaderCompileErrorInfo(isWarning,
                                                         szErrorEmitterName,
                                                         szErrorVertexShader,
                                                         szErrorFragmentShader,
                                                         szErrorLog));
        }

        if (this->m_pShaderBinaryBuffer != NULL)
        {
            // Copy the converted shader binary to the managed output buffer.
            pShaderBinary = gcnew cli::array<System::Byte>(outputDataSize);
            Marshal::Copy(IntPtr(this->m_pShaderBinaryBuffer), pShaderBinary, 0, outputDataSize);

            this->SetupShaderIndexForOutput(pDataArray);
        }

        // Release the buffer.
        this->ReleaseInputDataBuffer();

        return this->m_pShaderBinaryBuffer != NULL;
    }

    /// <summary>
    /// Generate shader code for the given emitter.
    /// </summary>
    /// <param name="pInputData">The input emitter data.</param>
    /// <param name="pVertexShader">The output vertex shader code.</param>
    /// <param name="pFragmentShader">The output fragment shader code.</param>
    /// <param name="pComputeShader">The output compute shader code.</param>
    /// <returns>True on success.</returns>
    bool ShaderConverterHelper::GenerateShaderCode(ShaderConversionInputData ^pInputData,
                                                   [Out] System::String^% pVertexShader,
                                                   [Out] System::String^% pFragmentShader,
                                                   [Out] System::String^% pComputeShader)
    {
        pVertexShader   = nullptr;
        pFragmentShader = nullptr;
        pComputeShader  = nullptr;

        if (this->m_pConverter == NULL)
        {
            return false;
        }

        this->ConvertInputData(pInputData);
        if (this->m_pInputBuffer == NULL)
        {
            return false;
        }

        if (this->m_pShaderCodes == NULL)
        {
            return false;
        }

        this->m_pConverter->Initialize(m_pShaderCodes);

        int vertexShaderSize = 0;
        int fragmentShaderSize = 0;
        int computeShaderSize = 0;
        char *pNativeVertexShader = NULL;
        char *pNativeFragmentShader = NULL;
        char *pNativeComputeShader = NULL;

        bool result = this->m_pConverter->GenerateShaderCode(
            (ShaderConverterEmitterData*)this->m_pInputBuffer,
            &pNativeVertexShader,
            &vertexShaderSize,
            &pNativeFragmentShader,
            &fragmentShaderSize,
            &pNativeComputeShader,
            &computeShaderSize,
            pInputData->UserDefineIndex);

        // Release the buffer.
        this->ReleaseInputDataBuffer();

        if (result == false)
        {
            pVertexShader = nullptr;
            pFragmentShader = nullptr;
            pComputeShader = nullptr;
            return false;
        }

        Encoding ^pEncoding = Encoding::GetEncoding("Shift_JIS");

        pVertexShader = gcnew String(pNativeVertexShader, 0, vertexShaderSize, pEncoding);
        pFragmentShader = gcnew String(pNativeFragmentShader, 0, fragmentShaderSize, pEncoding);
        pComputeShader = gcnew String(pNativeComputeShader, 0, computeShaderSize, pEncoding);

        // Clean up.
        delete[] pNativeVertexShader;
        delete[] pNativeFragmentShader;
        delete[] pNativeComputeShader;

        return result;
    }

    /// <summary>
    /// Generate shader assembly for the given emitter.
    /// </summary>
    /// <param name="pInputData">The input emitter data.</param>
    /// <param name="pVertexShader">The output vertex shader assembly.</param>
    /// <param name="pFragmentShader">The output fragment shader assembly.</param>
    /// <param name="pComputeShader">The output compute shader assembly.</param>
    /// <param name="pErrorList">The array of compile error information.</param>
    /// <returns>True on success.</returns>
    bool ShaderConverterHelper::GenerateShaderAssembly(ShaderConversionInputData ^pInputData,
                                                       [Out] System::String^% pVertexShader,
                                                       [Out] System::String^% pFragmentShader,
                                                       [Out] System::String^% pComputeShader,
                                                       [Out] List<ShaderCompileErrorInfo^>^% pErrorList)
    {
        pVertexShader   = nullptr;
        pFragmentShader = nullptr;
        pComputeShader  = nullptr;
        pErrorList      = gcnew List<ShaderCompileErrorInfo^>();

        if (this->m_pConverter == NULL)
        {
            return false;
        }

        this->ConvertInputData(pInputData);
        if (this->m_pInputBuffer == NULL)
        {
            return false;
        }

        if (this->m_pShaderCodes == NULL)
        {
            return false;
        }

        this->m_pConverter->Initialize(m_pShaderCodes);

        int vertexShaderSize = 0;
        int fragmentShaderSize = 0;
        int computeShaderSize = 0;
        char *pNativeVertexShader = NULL;
        char *pNativeFragmentShader = NULL;
        char *pNativeComputeShader = NULL;
        ShaderCompileErrorList errorList;

        bool result = this->m_pConverter->GenerateAssemblyCode(
            (ShaderConverterEmitterData*)this->m_pInputBuffer,
            &errorList,
            &pNativeVertexShader,
            &vertexShaderSize,
            &pNativeFragmentShader,
            &fragmentShaderSize,
            &pNativeComputeShader,
            &computeShaderSize,
            pInputData->UserDefineIndex);

        // Release the buffer.
        this->ReleaseInputDataBuffer();

        bool isWarning;

        const char *szErrorEmitterName    = NULL;
        const char *szErrorVertexShader   = NULL;
        const char *szErrorFragmentShader = NULL;
        const char *szErrorLog            = NULL;

        // The compile error information.
        errorList.BeginEnumeration();
        while (errorList.GetNextItem(&isWarning,
                                     szErrorEmitterName,
                                     szErrorVertexShader,
                                     szErrorFragmentShader,
                                     szErrorLog) == true)
        {
            pErrorList->Add(gcnew ShaderCompileErrorInfo(isWarning,
                                                         szErrorEmitterName,
                                                         szErrorVertexShader,
                                                         szErrorFragmentShader,
                                                         szErrorLog));
        }

        if (result == false)
        {
            pVertexShader = nullptr;
            pFragmentShader = nullptr;
            pComputeShader = nullptr;
            return false;
        }

        Encoding ^pEncoding = Encoding::GetEncoding("Shift_JIS");

        pVertexShader = gcnew String(pNativeVertexShader, 0, vertexShaderSize, pEncoding);
        pFragmentShader = gcnew String(pNativeFragmentShader, 0, fragmentShaderSize, pEncoding);
        pComputeShader = gcnew String(pNativeComputeShader, 0, computeShaderSize, pEncoding);

        // Clean up.
        delete[] pNativeVertexShader;
        delete[] pNativeFragmentShader;
        delete[] pNativeComputeShader;

        return result;
    }

    /// <summary>
    /// Convert the input data into the native input data buffer
    /// so the shader converter can access it.
    /// </summary>
    /// <param name="pDataArray">The array of input emitter data.</param>
    void ShaderConverterHelper::ConvertInputData(cli::array<ShaderConversionInputData^> ^pDataArray)
    {
        // Release the buffer first.
        this->ReleaseInputDataBuffer();

        // Save the emitter count.
        this->m_inputEmitterCount = pDataArray->Length;

        // Compute the required size for the input data.
        size_t emitterDataBlockSize = sizeof(ShaderConverterEmitterData) * pDataArray->Length;
        size_t requiredBufferSize = emitterDataBlockSize;
        for each (ShaderConversionInputData ^pData in pDataArray)
        {
            requiredBufferSize += pData->Size;
        }

        // Create a native buffer for input data, in the following format :
        // [ShaderConverterEmitterData_0] <-- Emitter data block
        // [ShaderConverterEmitterData_1]
        // ...
        // [ShaderConverterEmitterData_N]
        // [Emitter binary, override shader code] <-- Raw data block
        // [Emitter binary, override shader code]
        // ...
        // [Emitter binary, override shader code]
        this->m_pInputBuffer = new unsigned char[requiredBufferSize];

        // The pointers.
        ShaderConverterEmitterData *pEmitterData =
            (ShaderConverterEmitterData*)this->m_pInputBuffer;

        unsigned char *pRawData =
            this->m_pInputBuffer + emitterDataBlockSize;

        // Fill in the data.
        for each (ShaderConversionInputData ^pData in pDataArray)
        {
            this->SetupNativeInputData(pData, &pEmitterData, &pRawData);
        }
    }

    /// <summary>
    /// Convert the input data into the native input data buffer
    /// so the shader converter can access it.
    /// </summary>
    /// <param name="pInputData">The input emitter data.</param>
    void ShaderConverterHelper::ConvertInputData(ShaderConversionInputData ^pInputData)
    {
        // Release the buffer first.
        this->ReleaseInputDataBuffer();

        // Save the emitter count.
        this->m_inputEmitterCount = 1;

        // Compute the required size for the input data.
        size_t emitterDataBlockSize = sizeof(ShaderConverterEmitterData);
        size_t requiredBufferSize = emitterDataBlockSize + pInputData->Size;

        // Create a native buffer for input data, in the following format :
        // [ShaderConverterEmitterData_0] <-- Emitter data block
        // [ShaderConverterEmitterData_1]
        // ...
        // [ShaderConverterEmitterData_N]
        // [Emitter binary, override shader code] <-- Raw data block
        // [Emitter binary, override shader code]
        // ...
        // [Emitter binary, override shader code]
        this->m_pInputBuffer = new unsigned char[requiredBufferSize];

        // The pointers.
        ShaderConverterEmitterData *pEmitterData =
            (ShaderConverterEmitterData*)this->m_pInputBuffer;

        unsigned char *pRawData =
            this->m_pInputBuffer + emitterDataBlockSize;

        // Fill in the data.
        this->SetupNativeInputData(pInputData, &pEmitterData, &pRawData);
    }

    /// <summary>
    /// Convert and copy the managed input data into the native memory,
    /// so the shader converter can access it.
    /// </summary>
    /// <param name="pManagedData">The managed input data.</param>
    /// <param name="ppEmitterData">The pointer to the native emitter data.</param>
    /// <param name="ppRawData">The pointer to the native emitter binary and custom shaders.</param>
    void ShaderConverterHelper::SetupNativeInputData(ShaderConversionInputData ^pManagedData,
                                                     ShaderConverterEmitterData **ppEmitterData,
                                                     unsigned char **ppRawData)
    {
        // Set up the emitter data.
        (*ppEmitterData)->pBinaryHeader = *ppRawData;

        (*ppEmitterData)->pResEmitter = (u8*)( *ppRawData + pManagedData->BinaryHeader->Length );

        (*ppEmitterData)->szOverrideShader =
            (char*)(*ppRawData + pManagedData->BinaryHeader->Length + pManagedData->OverrideShaderOffset);

        (*ppEmitterData)->iOverrideShaderLength = Encoding::GetEncoding("Shift_JIS")->GetByteCount(pManagedData->OverrideShaderCode);
        (*ppEmitterData)->vertexShaderIndex    = 0;
        (*ppEmitterData)->pixelShaderIndex     = 0;
        (*ppEmitterData)->shaderIndexOffset    = 0;
        (*ppEmitterData)->reservedShaderIndex  = pManagedData->ReservedShaderIndex;

        // Set up the raw data.
        // The emitter binary.
        pin_ptr<unsigned char> pUnmanagedHeader = &(pManagedData->BinaryHeader[0]);
        pin_ptr<unsigned char> pUnmanagedBuffer = &(pManagedData->EmitterBinary[0]);

        size_t copySize = pManagedData->BinaryHeader->Length;
        memcpy_s(*ppRawData,
                 copySize,
                 pUnmanagedHeader,
                 copySize);

        copySize = pManagedData->EmitterBinary->Length;
        memcpy_s(*ppRawData + pManagedData->BinaryHeader->Length,
                 copySize,
                 pUnmanagedBuffer,
                 copySize);

        // The vertex shader.
        IntPtr szOverrideShader =
            Marshal::StringToHGlobalAnsi(pManagedData->OverrideShaderCode);

        copySize = System::Text::Encoding::GetEncoding("Shift_JIS")->GetByteCount(pManagedData->OverrideShaderCode) + 1;
        memcpy_s(*ppRawData + pManagedData->BinaryHeader->Length + pManagedData->OverrideShaderOffset,
                 copySize,
                 (const char*)szOverrideShader.ToPointer(),
                 copySize);

        Marshal::FreeHGlobal(szOverrideShader);

        // Advance the pointer.
        ++(*ppEmitterData);
        (*ppRawData) += pManagedData->Size;
    }

    /// <summary>
    /// Set up the shader index and it's offset so we can send it back to the
    /// caller.
    /// </summary>
    /// <param name="pDataArray">The array of emitter data to be set up.</param>
    void ShaderConverterHelper::SetupShaderIndexForOutput(
        cli::array<ShaderConversionInputData^> ^pDataArray)
    {
        ShaderConverterEmitterData *pEmitterDataArray =
            (ShaderConverterEmitterData*)this->m_pInputBuffer;

        for (int i = 0; i < pDataArray->Length; ++i)
        {
            pDataArray[i]->ShaderIndexOffset =
                (System::Int64)pEmitterDataArray[i].shaderIndexOffset;

            pDataArray[i]->VertexShaderIndex      = pEmitterDataArray[i].vertexShaderIndex;
            pDataArray[i]->PixelShaderIndex       = pEmitterDataArray[i].pixelShaderIndex;
            pDataArray[i]->UserVertexShaderIndex1 = pEmitterDataArray[i].userVertexShaderIndex1;
            pDataArray[i]->UserPixelShaderIndex1  = pEmitterDataArray[i].userPixelShaderIndex1;
            pDataArray[i]->UserVertexShaderIndex2 = pEmitterDataArray[i].userVertexShaderIndex2;
            pDataArray[i]->UserPixelShaderIndex2  = pEmitterDataArray[i].userPixelShaderIndex2;
        }
    }

    /// <summary>
    /// Release the memory allocated for the native input data.
    /// </summary>
    void ShaderConverterHelper::ReleaseInputDataBuffer()
    {
        if (this->m_pInputBuffer != NULL)
        {
            delete[] this->m_pInputBuffer;
            this->m_pInputBuffer = NULL;
        }
    }

    /// <summary>
    /// Release the memory allocated for the shader binary data.
    /// </summary>
    void ShaderConverterHelper::ReleaseShaderBinaryBuffer()
    {
        if (this->m_pShaderBinaryBuffer != NULL)
        {
            delete[] this->m_pShaderBinaryBuffer;
            this->m_pShaderBinaryBuffer = NULL;
        }
    }

    /// <summary>
    /// C#のStringをC++のchar*に変換します.
    /// この関数で取得したchar*は不要になったらdelete[]してください.
    /// </summary>
    /// <param name="str">C#のString</param>
    /// <return>CPPのchar*を返します.</return>
    static char* CsStringToCppChar(String ^pStr, u32 *pSize)
    {
        int length = Encoding::GetEncoding("Shift_JIS")->GetByteCount(pStr);
        IntPtr strPtr = Marshal::StringToHGlobalAnsi(pStr);

        char *pChar = new char[length + 1];
        strcpy_s(
            pChar,
            length + 1,
            (const char*)strPtr.ToPointer());

        Marshal::FreeHGlobal(strPtr);

        if (pSize != NULL)
        {
            *pSize = length;
        }

        return pChar;
    }

} // namespace ShaderConverterWrapper
} // namespace EffectMaker
