﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once
#ifndef __EFT_SHADER_GSH_COMPILE_H__
#define __EFT_SHADER_GSH_COMPILE_H__

#ifdef __cplusplus
extern "C"
{
#endif // __cplusplus

#pragma warning ( push )
#pragma warning ( disable: 4302 4311 4312 )
#include "shaderUtils.h"
#pragma warning ( pop )

#include <sdk_ver.h>
#if CAFE_OS_SDK_VERSION < 20903
using namespace ShaderUtils;
#endif

#pragma warning ( push )
#pragma warning ( disable: 4302 4311 4312 )
#include "cafe/gx2/gx2Surface.h"
#include "cafe/gx2/gx2Texture.h"
#include "cafe/gfd.h"
#include "gfdMem.h"
#pragma warning ( pop )

class ShaderTable
{
private :
    char            *szEsetName;
    char            *szLog;
    u32             vertexShaderId;
    u32             pixelhaderId;
    GX2VertexShader* vsBinary;
    GX2PixelShader*  fsBinary;
public :
    #if CAFE_OS_SDK_VERSION >= 21002
    u8               shaderID[16];  // shader binary Guid
    #endif

    //-------------------------------------------------------------------------
    //
    //-------------------------------------------------------------------------
    ShaderTable() :
        szEsetName(NULL),
        szLog(NULL),
        vertexShaderId(0),
        pixelhaderId(0),
        vsBinary(NULL),
        fsBinary(NULL)
    {
        #if CAFE_OS_SDK_VERSION >= 21002
            memset( &shaderID, 0, 16 );
        #endif
    }

    //-------------------------------------------------------------------------
    //
    //-------------------------------------------------------------------------
    virtual ~ShaderTable()
    {
        if (this->szEsetName != NULL)
        {
            delete[] this->szEsetName;
            this->szEsetName = NULL;
        }

        if (this->szLog != NULL)
        {
            delete[] this->szLog;
            this->szLog = NULL;
        }
    };


    //-------------------------------------------------------------------------
    // Get & get emitter set name.
    //-------------------------------------------------------------------------
    void SetEmitterSetName(const char *szName)
    {
        // First clean up the memory.
        if (this->szEsetName != NULL)
        {
            delete[] this->szEsetName;
            this->szEsetName = NULL;
        }

        if (szName == NULL)
        {
            return;
        }

        size_t len = strlen(szName);
        if (len <= 0)
        {
            return;
        }

        // Copy the log message.
        this->szEsetName = new char[len + 1];
        memcpy_s(this->szEsetName, len + 1, szName, len + 1);
    };

    //-------------------------------------------------------------------------
    const char* GetEmitterSetName() const { return this->szEsetName; };

    //-------------------------------------------------------------------------
    // Get & get log message.
    //-------------------------------------------------------------------------
    void SetLog(const char *szLog)
    {
        // First clean up the memory.
        if (this->szLog != NULL)
        {
            delete[] this->szLog;
            this->szLog = NULL;
        }

        if (szLog == NULL)
        {
            return;
        }

        size_t len = strlen(szLog);
        if (len <= 0)
        {
            return;
        }

        // Copy the log message.
        this->szLog = new char[len + 1];
        memcpy_s(this->szLog, len + 1, szLog, len + 1);
    };

    //-------------------------------------------------------------------------
    const char* GetLog() const { return this->szLog; };

    //-------------------------------------------------------------------------
    // Set & get VertexShader ID.
    //-------------------------------------------------------------------------
    void SetVertexShaderId( u32 id )
    {
        this->vertexShaderId = id;
    };
    //-------------------------------------------------------------------------
    u32 GetVertexShaderId() { return this->vertexShaderId; };

    //-------------------------------------------------------------------------
    // Set & get PixelShader ID.
    //-------------------------------------------------------------------------
    void SetPixelShaderId( u32 id )
    {
        this->pixelhaderId = id;
    };
    //-------------------------------------------------------------------------
    u32 GetPixelShaderId() { return this->pixelhaderId; };

    //-------------------------------------------------------------------------
    // Set & get VertexShader.
    //-------------------------------------------------------------------------
    void SetVertexShader( GX2VertexShader* vsh )
    {
        vsBinary = vsh;
    }
    //-------------------------------------------------------------------------
    GX2VertexShader* GetVertexShader() { return this->vsBinary; };

    //-------------------------------------------------------------------------
    // Set & ge PixelShader.
    //-------------------------------------------------------------------------
    void SetPixelShader( GX2PixelShader* fsh )
    {
        fsBinary = fsh;
    }
    //-------------------------------------------------------------------------
    GX2PixelShader* GetPixelShader() { return this->fsBinary; };

    //-------------------------------------------------------------------------
    // Set Shader Table.
    //-------------------------------------------------------------------------
    void SetShaderTable( ShaderTable* newTable )
    {
        this->SetEmitterSetName( newTable->szEsetName );
        this->SetLog( newTable->szLog );
      //  this->pShader = (char*)malloc( newTable->shaderSize );
      //  memcpy_s( this->pShader, newTable->shaderSize, newTable->pShader, newTable->shaderSize );
      //  this->shaderSize     = newTable->shaderSize;
        memcpy( &shaderID, newTable->shaderID, 16 );
        this->vertexShaderId = newTable->vertexShaderId;
        this->pixelhaderId   = newTable->pixelhaderId;
    };
};

typedef struct _ShaderAsmTable {
    char* vertexAsmCode;
    char* fragmentAsmCode;
    char* GeometryAsmCode;
    _ShaderAsmTable()
    {
        vertexAsmCode = NULL;
        fragmentAsmCode = NULL;
        GeometryAsmCode = NULL;
    }
} ShaderAsmTable;

class CollectShaderBinary
{
public:
    //-------------------------------------------------------------------------
    //
    //-------------------------------------------------------------------------
    CollectShaderBinary():
       mVertexShaderArray(NULL),
       mPixelShaderArray(NULL),
       mVertexCnt(0),
       mPixelCnt(0)
    {
    }

    //-------------------------------------------------------------------------
    //
    //-------------------------------------------------------------------------
    ~CollectShaderBinary()
    {
        if ( mVertexShaderArray != NULL )
        {
            for ( u32 i = 0; i < mVertexCnt; i++ )
            {
                if ( mVertexShaderArray[i] )
                {
                    free( mVertexShaderArray[i] );
                    mVertexShaderArray[i] = NULL;
                }
            }
        }

        if ( mPixelShaderArray != NULL )
        {
            for ( u32 i = 0; i < mPixelCnt; i++ )
            {
                if ( mPixelShaderArray[i] )
                {
                    free( mPixelShaderArray[i] );
                    mPixelShaderArray[i] = NULL;
                }
            }
        }

        if ( mVertexShaderArray ) delete[] mVertexShaderArray;
        if ( mPixelShaderArray )  delete[] mPixelShaderArray;
        mVertexShaderArray = NULL;
        mPixelShaderArray  = NULL;
    }

    //-------------------------------------------------------------------------
    //
    //-------------------------------------------------------------------------
    void Initialize( u32 arraySize )
    {
        mVertexShaderArray = new GX2VertexShader*[arraySize * 3];
        mPixelShaderArray  = new GX2PixelShader*[arraySize  * 3];
        mVertexCnt         = 0;
        mPixelCnt          = 0;
    }

    //-------------------------------------------------------------------------
    //
    //-------------------------------------------------------------------------
    bool   CheckVertexShader( GX2VertexShader* srcShader, GX2VertexShader* dstShader );

    //-------------------------------------------------------------------------
    //
    //-------------------------------------------------------------------------
    bool   CheckPixelShader(  GX2PixelShader* srcShader,  GX2PixelShader* dstShader );

    //-------------------------------------------------------------------------
    //
    //-------------------------------------------------------------------------
    u32    CheckDupVertexShader( GX2VertexShader*   vertexShader );

    //-------------------------------------------------------------------------
    //
    //-------------------------------------------------------------------------
    u32    CheckDupPixelShader(  GX2PixelShader*    pixelShader  );

    //-------------------------------------------------------------------------
    //
    //-------------------------------------------------------------------------
    bool   GenerateGsh();

    //-------------------------------------------------------------------------
    //
    //-------------------------------------------------------------------------
    u32    GetGshBinarySize() { return mGshBinarySize; }

    //-------------------------------------------------------------------------
    //
    //-------------------------------------------------------------------------
    char*  GetGshBinary() { return mGshBinaryPointer; }

    //-------------------------------------------------------------------------
    //
    //-------------------------------------------------------------------------
    void   SetVertexShader( u32 idx, GX2VertexShader * shader )
    {
        mVertexShaderArray[idx] = shader;
    }

    //-------------------------------------------------------------------------
    //
    //-------------------------------------------------------------------------
    void   SetPixelShader( u32 idx, GX2PixelShader * shader )
    {
        mPixelShaderArray[idx] = shader;
    }

    u32    PushVertexShader( GX2VertexShader* vsh );
    u32    PushPixelShader( GX2PixelShader* fsh );

    u32    GetVertexShaderCnt() { return mVertexCnt; }
    u32    GetPixelShaderCnt()  { return mPixelCnt; }

    void updateShaderCnt()
    {
        mVertexCnt++;
        mPixelCnt++;
    }
private:
    GX2VertexShader**  mVertexShaderArray;     ///< Pointer to GX2VertexShader structure
    GX2PixelShader**   mPixelShaderArray;      ///< Pointer to GX2PixelShader structure
    u32                mVertexCnt;
    u32                mPixelCnt;
    u32                mGshBinarySize;
    char*              mGshBinaryPointer;

};


// Input DLLs
#ifdef _DEBUG
#define LIB_DLL_SHADERUTILS TEXT("shader\\shaderUtilsD.dll")
#define LIB_DLL_GFD         TEXT("gfdD.dll")
#else
#define LIB_DLL_SHADERUTILS TEXT("shader\\shaderUtils.dll")
#define LIB_DLL_GFD         TEXT("gfd.dll")
#endif

// Functions
typedef GSH2Handle   (*PGSH2Initialize)(GSH2Setup * pSetup);
typedef GX2Boolean (*PGSH2Destroy)(GSH2Handle h);
typedef GX2Boolean (*PGSH2CompileProgram)(GSH2Handle h, GSH2CompileSetup *pSetup, GSH2CompileOutput* pOutput);

#if CAFE_OS_SDK_VERSION >= 21104
typedef GX2Boolean (*PGSH2CompileProgram3)(GSH2Handle h, GSH2CompileSetup3 *pSetup, GSH2CompileOutput3* pOutput);
typedef GX2Boolean (*PGSH2DestroyGX2Program3)(GSH2Handle h, GSH2GX2Program3* pGx2Program);
#elif CAFE_OS_SDK_VERSION >= 21002
typedef GX2Boolean (*PGSH2CompileProgram2)(GSH2Handle h, GSH2CompileSetup2 *pSetup, GSH2CompileOutput* pOutput);
#endif
typedef GX2Boolean (*PGSH2DestroyGX2Program)(GSH2Handle h, GSH2GX2Program* pGx2Program);

typedef u32  (*PGSH2CalcFetchShaderSizeEx)(u32 num_attrib, GX2FetchShaderType fsType, GX2TessellationMode tessMode);
typedef void (*PGSH2InitFetchShaderEx)(GX2FetchShader* fs, void* fs_buffer, u32 num, const GX2AttribStream* attribs, GX2FetchShaderType type, GX2TessellationMode tessMode);

typedef u32 (*PGSH2GetVertexShaderGPRs)(const GX2VertexShader * pShader);
typedef u32 (*PGSH2GetGeometryShaderGPRs)(const GX2GeometryShader * pShader);
typedef u32 (*PGSH2GetPixelShaderGPRs)(const GX2PixelShader * pShader);

typedef u32 (*PGSH2GetVertexShaderStackEntries)(const GX2VertexShader * pShader);
typedef u32 (*PGSH2GetGeometryShaderStackEntries)(const GX2GeometryShader * pShader);
typedef u32 (*PGSH2GetPixelShaderStackEntries)(const GX2PixelShader * pShader);

typedef GX2Boolean (*PGFDWriteFileShader)(char* pFilename, GFDGPUVersion gpuVer, GFDEndianSwapMode swapMode, GFDAlignMode alignMode, u32 numShader, GFDShaders *pShaders);
typedef GX2Boolean (*PGFDAppendWriteFileShader)(char* pFilename, GFDGPUVersion gpuVer, GFDEndianSwapMode swapMode, GFDAlignMode alignMode, u32 numShader, GFDShaders *pShaders);
typedef GX2Boolean (*PGFDWriteFileShaderAsCode)(char* pFilename, GFDEndianSwapMode swapMode, GFDShaders *pShaders);

typedef struct
{
    PGSH2Initialize               Initialize;
    PGSH2Destroy                  Destroy;
#if CAFE_OS_SDK_VERSION >= 21104
    PGSH2CompileProgram3          CompileProgram3;
    PGSH2DestroyGX2Program3       DestroyGX2Program3;
#elif CAFE_OS_SDK_VERSION >= 21002
    PGSH2CompileProgram2          CompileProgram2;
#else
    PGSH2CompileProgram           CompileProgram;
#endif

    PGSH2DestroyGX2Program        DestroyGX2Program;

    PGSH2CalcFetchShaderSizeEx    CalcFetchShaderSizeEx;
    PGSH2InitFetchShaderEx        InitFetchShaderEx;

    PGSH2GetVertexShaderGPRs      GetVertexShaderGPRs;
    PGSH2GetGeometryShaderGPRs    GetGeometryShaderGPRs;
    PGSH2GetPixelShaderGPRs       GetPixelShaderGPRs;

    PGSH2GetVertexShaderStackEntries      GetVertexShaderStackEntries;
    PGSH2GetGeometryShaderStackEntries    GetGeometryShaderStackEntries;
    PGSH2GetPixelShaderStackEntries       GetPixelShaderStackEntries;
} GSH2Func;

typedef struct
{
    PGFDWriteFileShaderAsCode WriteFileShaderAsCode;
    PGFDWriteFileShader       WriteFileShader;
    PGFDAppendWriteFileShader AppendWriteFileShader;
} GFDFunc;

// Error codes
#define EC_SUCCESS                  0
#define EC_UNSUPPORTEDGPU           1
#define EC_UNSUPPORTEDSHADERTYPE    2
#define EC_INPUTFILEERROR           3
#define EC_OUTPUTFILEERROR          4
#define EC_BADPARAM                 5
#define EC_COMPILEFAILED            6
#define EC_GSH2INITFAILED           7
#define EC_LOADINGDLLFAILED         8

#define EC_OUTPUTWARNING            9  // added

// App config defaults
#define DEFAULT_GPU                 GPU_VERSION_GPU7
#define DEFAULT_ENDIANBUGFIX        false
#define DEFAULT_ALIGNMODE           false
#define DEFAULT_APPENDMODE          false
#define DEFAULT_OUTFILENAME         "output.gsh"
#define DEFAULT_FORCEUNIFORMBLOCK   false
#define DEFAULT_ASCODEMODE          false
#define DEFAULT_SHADINGLANGUAGE     SHADERLANG_GLSL
#if CAFE_OS_SDK_VERSION >= 21002
#define DEFAULT_NOSPARKINFO         false
#endif

#define MAX_INPUTFILE_COUNT 64

// App config
typedef struct
{
    char*   vs_source;   // VS shader source code
    char*   ps_source;   // PS shader source code
    char*   gs_source;   // GS shader source code

#if CAFE_OS_SDK_VERSION >= 21002
    char*   vs_source_filename;   // VS shader source code file name
    char*   ps_source_filename;   // PS shader source code file name
    char*   gs_source_filename;   // GS shader source code file name
#endif

    char**  so_varyings;   // stream out varyings
    u32     numSOVaryings; // number of stream out varyings in so_varyings

    GSH2GPUVersion     gpu;               // Target gpu
    char*              pOutFilename;      // name of output file
    GSH2ShaderLang     lang;              // shader language
    u32                endianbugfix;      // endian bug workaround for the prototype hardware
    u32                outfilealign;      // add alignment padding to output
    u32                append;            // append shaders to existing file
    u32                forceUniformBlock; // force uniform block usage
    u32                ascode;            // append shaders to existing file
    u32                dumpShaders;       // dump pre/post optimized file
#if CAFE_OS_SDK_VERSION >= 21002
    u64                optimizeFlags;     // what kinds of extra optimization to do (bitmap)
    u32                noSparkInfo;       // suppress Spark debug output

    char*              pSparkDir;         // override default output dir for Spark file
#if CAFE_OS_SDK_VERSION >= 21104
    char*	cs_source;                    // compute shader source code
#endif
#endif
} ShaderSourceInfo;

void gshInitialize(const char *path, u32 numShader);
void gshFinalize();
#if CAFE_OS_SDK_VERSION >= 21104
s32 gshCompile(
    char *vshader,
    u32  vsize,
    char *fshader,
    u32  fsize,
    char *gshader,
    u32  gsize,
    char *sodata,
    u32  sosize,
    char *cshader,
    u32  csize,
    char *outfile,
    ShaderTable *shaderTbl,
    ShaderAsmTable* asmCodeTable = NULL );
#else
s32 gshCompile(
    char *vshader,
    u32  vsize,
    char *fshader,
    u32  fsize,
    char *gshader,
    u32  gsize,
    char *sodata,
    u32  sosize,
    char *outfile,
    ShaderTable *shaderTbl,
    ShaderAsmTable* asmCodeTable = NULL );
#endif

void   gshGenerateShaderBinary();
u32    gshGetgshBinarySize();
char*  gshGetgshBinary();
u32   gshPushVertexShader( GX2VertexShader* vsh );
u32   gshPushPixelShader(  GX2PixelShader*  fsh );


void gshInitializMulti( const char *multiPath, const char *exePath, int jobsNumber );
#if CAFE_OS_SDK_VERSION >= 21104
s32 gshCompileMulti(
    char*           vshader,
    u32             vsize,
    char*           fshader,
    u32             fsize,
    char*           gshader,
    u32             gsize,
    char*           sodata,
    u32             sosize,
    char*           cshader,
    u32             csize,
    char*           outfile,
    ShaderTable*    shaderTbl,
    ShaderAsmTable* asmCodeTable );
#else
s32 gshCompileMulti(
    char*           vshader,
    u32             vsize,
    char*           fshader,
    u32             fsize,
    char*           gshader,
    u32             gsize,
    char*           sodata,
    u32             sosize,
    char*           outfile,
    ShaderTable*    shaderTbl,
    ShaderAsmTable* asmCodeTable );
#endif

void gshRemoveDuplicateShader( ShaderTable* shaderTbl, u32 numShaders );

void gshFinalizeMulti( const char *multiPath );

#ifdef __cplusplus
}
#endif // __cplusplus

#endif  // __EFT_SHADER_GSH_COMPILE_H__
