﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/vfx/vfx_Data.h>
#include <nn/vfx/vfx_Binary.h>

namespace nn {
namespace vfx {
namespace detail {

//---------------------------------------------------------------------------
//! @brief  64Bitフラグ管理クラス
//---------------------------------------------------------------------------
class Flag64
{
public:
    //---------------------------------------------------------------------------
    //! @brief  初期化
    //---------------------------------------------------------------------------
    void Initialize() NN_NOEXCEPT
    {
        flag0 = 0;
        flag1 = 0;
    }

    //---------------------------------------------------------------------------
    //! @brief  ビット桁を取得します。
    //! @param[in] bits TBD
    //! @return TBD
    //---------------------------------------------------------------------------
    uint32_t GetCarryCount( uint64_t bits ) NN_NOEXCEPT
    {
        uint32_t result = 63;
        uint64_t mask = 0x8000000000000000;
        for( ; mask != 0; mask = mask >> 1 )
        {
            if( bits & mask )
            {
                break;
            }
            result--;
        }
        return result;
    }

    //---------------------------------------------------------------------------
    //! @brief  指定ビットのフラグを立てます。
    //! @param[in] bit TBD
    //---------------------------------------------------------------------------
    void Set( uint64_t bit ) NN_NOEXCEPT
    {
        if( bit > ( static_cast< uint64_t >( 0x01 ) << static_cast< uint64_t >( 31 ) ) )
        {
            flag1 |= ( bit >> static_cast< uint64_t >( 32 ) );
        }
        else
        {
            flag0 |= bit;
        }
    }

    //---------------------------------------------------------------------------
    //! @brief  指定ビットのフラグをおろします。
    //! @param[in] bit TBD
    //---------------------------------------------------------------------------
    void Cancel( uint64_t bit ) NN_NOEXCEPT
    {
        if( bit > ( static_cast< uint64_t >( 0x01 ) << static_cast< uint64_t >( 31 ) ) )
        {
            flag1 &= ~( bit >> static_cast< uint64_t >( 32 ) );
        }
        else
        {
            flag0 &= ~bit;
        }
    }

    //---------------------------------------------------------------------------
    //! @brief  指定ビットのフラグが立っているか取得します。
    //! @param[in] bit TBD
    //! @return TBD
    //---------------------------------------------------------------------------
    bool Is( uint64_t bit ) const NN_NOEXCEPT
    {
        if( bit > ( static_cast< uint64_t >( 0x01 ) << static_cast< uint64_t >( 31 ) ) )
        {
            return ( flag1 & ( bit >> static_cast< uint64_t >( 32 ) ) ) ? 1 : 0;
        }
        else
        {
            return ( flag0 & bit ) ? 1 : 0;
        }
    }

    //---------------------------------------------------------------------------
    //! @brief  等価
    //! @param[in] flag TBD
    //! @return TBD
    //---------------------------------------------------------------------------
    bool operator== ( Flag64 flag ) NN_NOEXCEPT
    {
        if( ( flag.flag0 == flag0 ) &&
            ( flag.flag1 == flag1 ) )
        {
            return true;
        }
        return false;
    }

private:
    uint32_t flag0;  //!< TBD
    uint32_t flag1;  //!< TBD
};

//---------------------------------------------------------------------------
//! @brief  シェーダキー生成の為のビットフラグ
//---------------------------------------------------------------------------
enum
{
    VFX_SHADER_KEY_FIELD_RANDOM = ( static_cast< uint64_t >( 0x01 ) << static_cast< uint64_t >( 0 ) ),   //!< フィールドランダム
};

//---------------------------------------------------------------------------
//! @brief  シェーダキー
//---------------------------------------------------------------------------
class ShaderKey
{
    NN_DISALLOW_COPY( ShaderKey );
public:

    ShaderKey()
    {

    }

    //---------------------------------------------------------------------------
    //! @brief  初期化
    //! @param[in] resEmitter TBD
    //! @param[in] emitterPluginIndex TBD
    //! @param[in] emitterPluginData TBD
    //! @param[in] fieldFlag TBD
    //! @param[in] customFieldFlag TBD
    //! @param[in] userDefine TBD
    //---------------------------------------------------------------------------
    void Initialize( detail::ResEmitter* resEmitter, uint32_t emitterPluginIndex, void* emitterPluginData, uint32_t fieldFlag, detail::ResFieldCustom* resCustomField, const char* userDefine = NULL ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief  コンパイルセッティングを取得します。
    //! @param[in] isCafe TBD
    //! @return TBD
    //---------------------------------------------------------------------------
    const char* GetCompileSetting() NN_NOEXCEPT
    {
        Generate();
        return g_CompileSetting;
    }

    //---------------------------------------------------------------------------
    //! @brief  頂点シェーダキーを取得します。
    //! @return TBD
    //---------------------------------------------------------------------------
    Flag64 GetVertexShaderKey() const NN_NOEXCEPT
    {
        return m_VertexShaderFlag;
    }

    //---------------------------------------------------------------------------
    //! @brief  頂点シェーダキー2を取得します。
    //! @return TBD
    //---------------------------------------------------------------------------
    Flag64 GetVertexShaderKey2() const NN_NOEXCEPT
    {
        return m_VertexShaderFlag2;
    }

    //---------------------------------------------------------------------------
    //! @brief  ピクセルシェーダキーを取得します。
    //! @return TBD
    //---------------------------------------------------------------------------
    Flag64 GetPixelShaderKey() const NN_NOEXCEPT
    {
        return m_PixelShaderFlag;
    }

    //---------------------------------------------------------------------------
    //! @brief  ピクセルシェーダキーを取得します。
    //! @return TBD
    //---------------------------------------------------------------------------
    Flag64 GetPixelShaderKey2() const NN_NOEXCEPT
    {
        return m_PixelShaderFlag2;
    }

    //---------------------------------------------------------------------------
    //! @brief  頂点シェーダキーを取得します。（EmitterPlugin版）
    //! @return TBD
    //---------------------------------------------------------------------------
    Flag64 GetVertexShaderKeyEP() const NN_NOEXCEPT
    {
        return m_VertexShaderFlagEP;
    }

    //---------------------------------------------------------------------------
    //! @brief  ピクセルシェーダキーを取得します。（EmitterPlugin版）
    //! @return TBD
    //---------------------------------------------------------------------------
    Flag64 GetPixelShaderKeyEP() const NN_NOEXCEPT
    {
        return m_PixelShaderFlagEP;
    }

    //---------------------------------------------------------------------------
    //! @brief  描画パスを取得します。
    //! @return 描画パス
    //---------------------------------------------------------------------------
    uint32_t GetDrawPathFlag() const NN_NOEXCEPT
    {
        return m_DrawPathFlag;
    }

    //---------------------------------------------------------------------------
    //! @brief  アニメーションキーを取得します。
    //! @return TBD
    //---------------------------------------------------------------------------
    uint8_t GetColor0AnimKeyCount() const NN_NOEXCEPT
    {
        return m_VertexColor0AnimKey;
    }
    //---------------------------------------------------------------------------
    //! @brief  アニメーションキーを取得します。
    //! @return TBD
    //---------------------------------------------------------------------------
    uint8_t GetColor1AnimKeyCount() const NN_NOEXCEPT
    {
        return m_VertexColor1AnimKey;
    }
    //---------------------------------------------------------------------------
    //! @brief  アニメーションキーを取得します。
    //! @return TBD
    //---------------------------------------------------------------------------
    uint8_t GetAlpha0AnimKeyCount() const NN_NOEXCEPT
    {
        return m_VertexAlpha0AnimKey;
    }
    //---------------------------------------------------------------------------
    //! @brief  アニメーションキーを取得します。
    //! @return TBD
    //---------------------------------------------------------------------------
    uint8_t GetAlpha1AnimKeyCount() const NN_NOEXCEPT
    {
        return m_VertexAlpha1AnimKey;
    }
    //---------------------------------------------------------------------------
    //! @brief  アニメーションキーを取得します。
    //! @return TBD
    //---------------------------------------------------------------------------
    uint8_t GetScaleAnimKeyCount() const NN_NOEXCEPT
    {
        return m_VertexScaleAnimKey;
    }

    //---------------------------------------------------------------------------
    //! @brief  キーを比較します。
    //! @param[in] pShaderKey TBD
    //! @return TBD
    //---------------------------------------------------------------------------
    bool IsEqual( ShaderKey* pShaderKey ) NN_NOEXCEPT
    {
        if( ( m_VertexShaderFlag == pShaderKey->GetVertexShaderKey() ) &&
            ( m_VertexShaderFlag2 == pShaderKey->GetVertexShaderKey2() ) &&
            ( m_PixelShaderFlag == pShaderKey->GetPixelShaderKey() ) &&
            ( m_PixelShaderFlag2 == pShaderKey->GetPixelShaderKey2() ) &&
            ( m_VertexShaderFlagEP == pShaderKey->GetVertexShaderKeyEP() ) &&
            ( m_PixelShaderFlagEP == pShaderKey->GetPixelShaderKeyEP() ) &&
            ( m_VertexColor0AnimKey == pShaderKey->GetColor0AnimKeyCount() ) &&
            ( m_VertexColor1AnimKey == pShaderKey->GetColor1AnimKeyCount() ) &&
            ( m_VertexAlpha0AnimKey == pShaderKey->GetAlpha0AnimKeyCount() ) &&
            ( m_VertexAlpha1AnimKey == pShaderKey->GetAlpha1AnimKeyCount() ) &&
            ( m_VertexScaleAnimKey == pShaderKey->GetScaleAnimKeyCount() ) &&
            ( m_CustomShaderIndex == pShaderKey->m_CustomShaderIndex ) &&
            ( m_CustomShaderFlag == pShaderKey->m_CustomShaderFlag ) &&
            ( m_CustomFieldFlag == pShaderKey->m_CustomFieldFlag ) &&
            ( m_CustomShaderSwitch == pShaderKey->m_CustomShaderSwitch ) &&
            ( m_CombinerShaderIndex == pShaderKey->m_CombinerShaderIndex ) &&
            ( strcmp( m_CustomShaderDefine, pShaderKey->m_CustomShaderDefine ) == 0 ) )
        {
            return true;
        }
        return false;
    }

    //---------------------------------------------------------------------------
    //! @brief  キーを比較します。
    //! @return TBD
    //---------------------------------------------------------------------------
    bool IsStreamOut() const NN_NOEXCEPT;

public:
    //---------------------------------------------------------------------------
    //! @brief  キーを無効化します。
    //---------------------------------------------------------------------------
    void Invalidate() NN_NOEXCEPT
    {
        m_VertexShaderFlag.Initialize();
        m_VertexShaderFlag2.Initialize();
        m_PixelShaderFlag.Initialize();
        m_PixelShaderFlag2.Initialize();
        m_VertexShaderFlagEP.Initialize();
        m_PixelShaderFlagEP.Initialize();
        m_DrawPathFlag        = 0;
        m_CustomShaderIndex   = 0;
        m_CustomShaderFlag    = 0;
        m_CustomFieldFlag     = 0;
        m_CustomShaderSwitch  = 0;
        m_CombinerShaderIndex = static_cast< uint32_t >( -1 );
        m_VertexColor0AnimKey = 0;
        m_VertexColor1AnimKey = 0;
        m_VertexAlpha0AnimKey = 0;
        m_VertexAlpha1AnimKey = 0;
        m_VertexScaleAnimKey  = 0;
        memset( m_CustomShaderDefine, 0, 16 );
    }

    //---------------------------------------------------------------------------
    //! @brief  描画パスを設定します。
    //! @param[in] drawPath 描画パス
    //---------------------------------------------------------------------------
    void SetDrawPath( uint32_t drawPath ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief  エミッタ挙動タイプを設定します。
    //! @param[in] type TBD
    //---------------------------------------------------------------------------
    void SetEmitterCalculationType( EmitterCalculationMode type ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief  パーティクル形状タイプを設定します。
    //! @param[in] type TBD
    //---------------------------------------------------------------------------
    void SetFigureType( ParticleFigureType type ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief  パーティクルタイプを設定します。
    //! @param[in] type TBD
    //---------------------------------------------------------------------------
    void SetParticleType( ParticleBillboardType type ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief  世界座標系で重力を適用 を設定します。
    //! @param[in] worldGravity TBD
    //---------------------------------------------------------------------------
    void SetParticleWorldGravity( bool worldGravity ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief  パーティクル回転タイプを設定します。
    //! @param[in] type TBD
    //---------------------------------------------------------------------------
    void SetParticleRotateType( ParticleRotationType type ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief  エミッタ追従タイプを設定します。
    //! @param[in] type TBD
    //---------------------------------------------------------------------------
    void SetEmitterFollowType( ParticleFollowType type ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief  パーティクルカラー計算タイプを設定します。
    //! @param[in] color0Type TBD
    //! @param[in] color1Type TBD
    //---------------------------------------------------------------------------
    void SetParticleColorCalculationType( ParticleColorCalculationMode color0Type, ParticleColorCalculationMode color1Type ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief  パーティクルアルファ計算タイプを設定します。
    //! @param[in] alpha0Type TBD
    //! @param[in] alpha1Type TBD
    //---------------------------------------------------------------------------
    void SetParticleAlphaCalculationType( ParticleColorCalculationMode alpha0Type, ParticleColorCalculationMode alpha1Type ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief  パーティクルテクスチャパターンアニメ有効無効を設定します。
    //! @param[in] texture0 TBD
    //! @param[in] texture1 TBD
    //! @param[in] texture2 TBD
    //---------------------------------------------------------------------------
    void SetParticleTexturePatternAnim( bool texture0, bool texture1, bool texture2 ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief  パーティクルテクスチャサーフェイスアニメ有効無効を設定します。
    //! @param[in] texture0 TBD
    //! @param[in] texture1 TBD
    //! @param[in] texture2 TBD
    //---------------------------------------------------------------------------
    void ShaderKey::SetParticleTextureSurfaceAnim( bool texture0, bool texture1, bool texture2 ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief  パーティクルテクスチャ座標アニメ有効無効を設定します。
    //! @param[in] texture0 TBD
    //! @param[in] texture1 TBD
    //! @param[in] texture2 TBD
    //---------------------------------------------------------------------------
    void SetParticleTextureShiftAnim( bool texture0, bool texture1, bool texture2 ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief  パーティクルスフィアマップ有効無効を設定します。
    //! @param[in] texture0 TBD
    //! @param[in] texture1 TBD
    //! @param[in] texture2 TBD
    //---------------------------------------------------------------------------
    void SetParticleSphereMap( bool texture0, bool texture1, bool texture2 ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief  ピクセルシェーダを標準設定にします。
    //---------------------------------------------------------------------------
    void SetDefaultPixelShaderSetting() NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief  color/alpha 8keyアニメを設定します。
    //! @param[in] color0Anim TBD
    //! @param[in] color1Anim TBD
    //---------------------------------------------------------------------------
    void SetColor8KeyAnim( uint8_t color0Anim, uint8_t color1Anim ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief 揺らぎを設定します。
    //! @param[in] fluctuation TBD
    //---------------------------------------------------------------------------
    void SetFluctuation( uint32_t fluctuation ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief 全てのパーティクルタイプをクリアする。
    //---------------------------------------------------------------------------
    void ClearAllParticleType() NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief 全ての追従タイプをクリアする。
    //---------------------------------------------------------------------------
    void ClearAllFollowType() NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief 全ての挙動計算タイプをクリアする。
    //---------------------------------------------------------------------------
    void ClearAllCalculationType() NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief 全ての挙動計算タイプをクリアする。
    //---------------------------------------------------------------------------
    void ClearAllFluctuation() NN_NOEXCEPT;

private:
    //---------------------------------------------------------------------------
    //! @brief  コンパイルセッティングを生成します。
    //! @param[in] isCafe TBD
    //---------------------------------------------------------------------------
    void Generate() NN_NOEXCEPT;

    Flag64      m_VertexShaderFlag;                                     //!< 頂点シェーダフラグ
    Flag64      m_VertexShaderFlag2;                                    //!< 頂点シェーダフラグ（2本目）
    Flag64      m_PixelShaderFlag;                                      //!< ピクセルシェーダフラグ
    Flag64      m_PixelShaderFlag2;                                     //!< ピクセルシェーダフラグ
    Flag64      m_VertexShaderFlagEP;                                   //!< Emitter Plugin 頂点シェーダフラグ
    Flag64      m_PixelShaderFlagEP;                                    //!< Emitter Plugin ピクセルシェーダフラグ
    uint64_t    m_CustomShaderFlag;                                     //!< カスタムシェーダフラグ
    uint64_t    m_CustomShaderSwitch;                                   //!< カスタムシェーダスイッチ
    uint32_t    m_DrawPathFlag;                                         //!< 描画パス
    uint32_t    m_CustomShaderIndex;                                    //!< カスタムシェーダインデックス
    uint32_t    m_CustomFieldFlag;                                      //!< カスタムフィールドフラグ
    uint32_t    m_CombinerShaderIndex;                                  //!< コンバイナシェーダインデックス
    char        m_CustomShaderDefine[ 16 ];                             //!< カスタムシェーダユーザー定義
    uint8_t     m_CustomFieldEnabled;                                   //!< カスタムフィールドの有無
    uint8_t     m_VertexColor0AnimKey;                                  //!< カラー0アニメ
    uint8_t     m_VertexColor1AnimKey;                                  //!< カラー1アニメ
    uint8_t     m_VertexAlpha0AnimKey;                                  //!< アルファ0アニメ
    uint8_t     m_VertexAlpha1AnimKey;                                  //!< アルファ1アニメ
    uint8_t     m_VertexScaleAnimKey;                                   //!< スケールアニメ

    enum ShaderCompileSettings
    {
        ShaderCompileSettings_BufferSize = 8192,                        //<! シェーダコンパイル定義用バッファサイズ
    };
    static char g_CompileSetting[ ShaderCompileSettings_BufferSize ];   //<! シェーダコンパイル定義用バッファ
};

} // namespace detail
} // namespace vfx
} // namespace nn
