﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Text.RegularExpressions;
using nw.g3d.iflib;
using nw.g3d.nw4f_3dif;

namespace G3dIfLib
{
    /// <summary>
    /// G3D 中間ファイルユーティリティ
    /// </summary>
    public class G3dIfUtility
    {
        /// <summary>
        /// 中間ファイルスキーマのベースパスを取得または設定します。
        /// </summary>
        public static string XsdBasePath { get; set; }

        /// <summary>
        /// 中間ファイルを読み込む
        /// </summary>
        /// <param name="filePath">中間ファイルパス</param>
        /// <returns>結果</returns>
        public static LoadResult LoadG3dIf(string filePath)
        {
            //// iflib を使って実際に読みに行く
            //// 不正な中間ファイルなどので例外になることはある。
            //// この関数では例外をキャッチしない。この関数を呼ぶ側でキャッチする。

            List<G3dStream> streamArray = null;
            bool isIfUpdated;
            nw4f_3difType g3dIf;
            if (G3dPath.IsStreamBinaryPath(filePath))
            {
                streamArray = new List<G3dStream>();
                g3dIf = IfBinaryReadUtility.Read(streamArray, filePath, XsdBasePath, out isIfUpdated);
            }
            else
            {
                g3dIf = IfReadUtility.Read(filePath, XsdBasePath, out isIfUpdated);
            }

            return new LoadResult
            {
                G3dIf = g3dIf,
                BinaryStreams = streamArray ?? MakeBinaryStreams(g3dIf)
            };
        }

        /// <summary>
        /// nw4f_3dif からバイナリストリームを取り出す
        /// </summary>
        /// <param name="g3dIf">中間ファイル</param>
        /// <returns>バイナリストリーム</returns>
        private static List<G3dStream> MakeBinaryStreams(nw4f_3difType g3dIf)
        {
            var binaryStreams = new List<G3dStream>();
            {
                // ストリームがあれば保持する
                if (G3dStreamUtility.HasStreamArray(g3dIf))
                {
                    G3dStreamUtility.ToStreams(binaryStreams, g3dIf.RootElement.stream_array);

                    // 改行をバイナリファイルを読み込んだ時に合わせる
                    foreach (var stream in binaryStreams)
                    {
                        switch (stream.type)
                        {
                        case stream_typeType.@string:
                        case stream_typeType.wstring:
                            stream.StringData = stream.StringData.Replace("\n", "\r\n");
                            break;
                        }
                    }
                }
            }

            return binaryStreams;
        }

        /// <summary>
        /// 読み込み結果
        /// </summary>
        public class LoadResult
        {
            /// <summary>
            /// 中間ファイルオブジェクト
            /// </summary>
            public nw4f_3difType G3dIf { get; set; }

            /// <summary>
            /// バイナリストリーム
            /// </summary>
            public List<G3dStream> BinaryStreams { get; set; }
        }

        // TODO: G3D XSD の仕様変更でファイルバージョンが正常に取れなくなったため無理やり動かす
#if false

        /// <summary>
        /// G3Dファイルの最新のバージョン番号を取得します。
        /// バージョン番号はG3D XSDのフォルダから取得します。
        /// </summary>
        /// <returns>バージョンを取得できたときはそのバージョンを、取得できなかったときはnullを返します。</returns>
        public static Version GetLatestFileVersion()
        {
            // G3D XSDのパスをチェック
            if (string.IsNullOrEmpty(XsdBasePath) == true ||
                Path.IsPathRooted(XsdBasePath) == false ||
                Directory.Exists(XsdBasePath) == false)
            {
                return null;
            }

            Version latestVersion = new Version();

            // フォルダの名前をパースして最新のバージョン番号を取得
            string[] dirPaths = Directory.GetDirectories(XsdBasePath);
            foreach (string path in dirPaths)
            {
                Version version;
                string dirName = Path.GetFileName(path);

                dirName = dirName.Replace("ver_", string.Empty);
                dirName = dirName.Replace("_", ".");

                if (Version.TryParse(dirName, out version) == true)
                {
                    if (version.CompareTo(latestVersion) > 0)
                    {
                        latestVersion = version;
                    }
                }
            }

            return latestVersion;
        }

        /// <summary>
        /// G3Dファイルのバージョンが最新版になっているかチェックします。
        /// </summary>
        /// <param name="filePath">ファイルパス</param>
        /// <param name="latestVersion">最新のバージョン番号</param>
        /// <returns>最新版のときtrue、それ以外はfalseを返します。</returns>
        public static bool CheckG3DFileVersion(string filePath, Version latestVersion)
        {
            // <nw4f_3dif version="**.**.**" にマッチする正規表現
            Regex regex = new Regex("^[ ]*<nw4f_3dif version=\"([0-9]+\\.[0-9]+\\.[0-9]+)\"");

            try
            {
                using (StreamReader reader = File.OpenText(filePath))
                {
                    while (reader.EndOfStream == false)
                    {
                        // ファイルから1行読み込み、マッチング
                        string line = reader.ReadLine();
                        Match match = regex.Match(line);

                        if (match.Groups.Count < 2)
                        {
                            continue;
                        }

                        // バージョン番号を取得
                        Version version;
                        string versionStr = match.Groups[1].Value;

                        if (Version.TryParse(versionStr, out version) == false)
                        {
                            continue;
                        }

                        // バージョン番号を比較
                        if (version.CompareTo(latestVersion) != 0)
                        {
                            return false;
                        }
                        else
                        {
                            break;
                        }
                    }
                }
            }
            catch
            {
            }

            return true;
        }

#else

        /// <summary>
        /// G3Dファイルの最新のバージョン番号を取得します。
        /// バージョン番号はG3D XSDのフォルダから取得します。
        /// 現在、最新のバージョン番号が正しく取得できないため常に "0.0.0.0" を返します。
        /// </summary>
        /// <returns>バージョンを取得できたときはそのバージョンを、取得できなかったときはnullを返します。</returns>
        public static Version GetLatestFileVersion()
        {
            return new Version("0.0.0.0");
        }

        /// <summary>
        /// G3Dファイルのバージョンが最新版になっているかチェックします。
        /// 現在、G3Dファイルのバージョンが正しく読み込めないため常に true を返します。
        /// </summary>
        /// <param name="filePath">ファイルパス</param>
        /// <param name="latestVersion">最新のバージョン番号</param>
        /// <returns>最新版のときtrue、それ以外はfalseを返します。</returns>
        public static bool CheckG3DFileVersion(string filePath, Version latestVersion)
        {
            return true;
        }

#endif
    }
}
