﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Extensions;

namespace EffectMaker.Application.OptionPanes
{
    /// <summary>
    /// World RenderScale options
    /// </summary>
    [DisplayOrder(3)]
    public partial class WorldScaleOptionPane : UserControl, IOptionPane
    {
        /// <summary>
        /// Stores a flag to prevent controls to interlock.
        /// </summary>
        private bool isLocked;

        /// <summary>
        /// Initializes the WorldScaleOptionPane instance.
        /// </summary>
        public WorldScaleOptionPane()
        {
            this.InitializeComponent();

            this.trkWorldScalePositionSpeedRange.ValueChanged += this.OnWorldScalePositionSpeedRangeTrackBarValueChanged;
            this.numWorldScalePositionSpeedRange.ValueChanged += this.OnWorldScalePositionSpeedRangeNumericUpDownValueChanged;

            this.trkWorldScaleTime.ValueChanged += this.OnWorldScaleTimeTrackBarValueChanged;
            this.numWorldScaleTime.ValueChanged += this.OnWorldScaleTimeNumericUpDownValueChanged;

            this.trkWorldScaleDisplayScale.ValueChanged += this.OnWorldScaleDisplayScaleTrackBarValueChanged;
            this.numWorldScaleDisplayScale.ValueChanged += this.OnWorldScaleDisplayScaleNumericUpDownValueChanged;
        }

        /// <summary>
        /// Gets the identifier.
        /// </summary>
        public string Identifier
        {
            get { return "WorldScaleOptionPane"; }
        }

        /// <summary>
        /// Gets the parent identifier.
        /// </summary>
        public string ChildOf
        {
            get { return null; }
        }

        /// <summary>
        /// Gets the display name.
        /// </summary>
        public string DisplayName
        {
            get { return Properties.Resources.OptionPaneWorldScale; }
        }

        /// <summary>
        /// Called when initialized.
        /// </summary>
        public void OnInitialize()
        {
            WorldScaleOptions options = OptionStore.RootOptions.WorldScale;

            this.numWorldScalePositionSpeedRange.SetValueClamped(options.PositionSpeedSize);
            this.numWorldScaleTime.SetValueClamped(options.Time);
            this.numWorldScaleDisplayScale.SetValueClamped(options.DisplayScale);

            // ラベル設定
            this.grpWorldScalePositionSpeedRange.Text = Properties.Resources.OptionWorldScalePosition;
            this.grpWorldScaleTime.Text = Properties.Resources.OptionWorldScaleTime;
            this.grpWorldScaleDisplayScale.Text = Properties.Resources.OptionWorldScaleGridScale;
        }

        /// <summary>
        /// Called when terminated.
        /// </summary>
        public void OnTerminate()
        {
        }

        /// <summary>
        /// Called when the Accept button is clicked.
        /// </summary>
        public void OnAccept()
        {
            WorldScaleOptions options = OptionStore.RootOptions.WorldScale;

            options.PositionSpeedSize = (float)this.numWorldScalePositionSpeedRange.Value;
            options.Time = (int)this.numWorldScaleTime.Value;
            options.DisplayScale = (float)this.numWorldScaleDisplayScale.Value;
        }

        /// <summary>
        /// Called when the Cancel button is clicked.
        /// </summary>
        public void OnCancel()
        {
        }

        /// <summary>
        /// Called when the WorldScalePositionSpeedRange TrackBar value changes.
        /// </summary>
        /// <param name="sender">The WorldScalePositionSpeedRange TrackBar.</param>
        /// <param name="e">Event argument.</param>
        private void OnWorldScalePositionSpeedRangeTrackBarValueChanged(object sender, EventArgs e)
        {
            if (this.isLocked)
            {
                return;
            }

            this.isLocked = true;
            try
            {
                double value = this.trkWorldScalePositionSpeedRange.Value / 1000.0;

                var min = decimal.ToDouble(this.numWorldScalePositionSpeedRange.Minimum);
                var max = decimal.ToDouble(this.numWorldScalePositionSpeedRange.Maximum);

                value = Math.Max(min, Math.Min(value, max));

                this.numWorldScalePositionSpeedRange.Value = new decimal(value);
            }
            finally
            {
                this.isLocked = false;
            }
        }

        /// <summary>
        /// Called when the WorldScalePositionSpeedRange NumericUpDown value changes.
        /// </summary>
        /// <param name="sender">The WorldScalePositionSpeedRange NumericUpDown.</param>
        /// <param name="e">Event argument.</param>
        private void OnWorldScalePositionSpeedRangeNumericUpDownValueChanged(object sender, EventArgs e)
        {
            if (this.isLocked)
            {
                return;
            }

            this.isLocked = true;
            try
            {
                decimal value = this.numWorldScalePositionSpeedRange.Value * 1000m;
                this.trkWorldScalePositionSpeedRange.Value = (int)value;
            }
            finally
            {
                this.isLocked = false;
            }
        }

        /// <summary>
        /// Called when the WorldScaleTime TrackBar value changes.
        /// </summary>
        /// <param name="sender">The WorldScaleTime TrackBar.</param>
        /// <param name="e">Event argument.</param>
        private void OnWorldScaleTimeTrackBarValueChanged(object sender, EventArgs e)
        {
            if (this.isLocked)
            {
                return;
            }

            this.isLocked = true;
            try
            {
                this.numWorldScaleTime.Value = this.trkWorldScaleTime.Value;
            }
            finally
            {
                this.isLocked = false;
            }
        }

        /// <summary>
        /// Called when the WorldScaleTime NumericUpDown value changes.
        /// </summary>
        /// <param name="sender">The WorldScaleTime NumericUpDown.</param>
        /// <param name="e">Event argument.</param>
        private void OnWorldScaleTimeNumericUpDownValueChanged(object sender, EventArgs e)
        {
            if (this.isLocked)
            {
                return;
            }

            this.isLocked = true;
            try
            {
                this.trkWorldScaleTime.Value = (int)this.numWorldScaleTime.Value;
            }
            finally
            {
                this.isLocked = false;
            }
        }

        /// <summary>
        /// Called when the WorldScaleDisplayScale TrackBar value changes.
        /// </summary>
        /// <param name="sender">The WorldScaleDisplayScale TrackBar.</param>
        /// <param name="e">Event argument.</param>
        private void OnWorldScaleDisplayScaleTrackBarValueChanged(object sender, EventArgs e)
        {
            if (this.isLocked)
            {
                return;
            }

            this.isLocked = true;
            try
            {
                double value = this.trkWorldScaleDisplayScale.Value / 1000.0;

                var min = decimal.ToDouble(this.numWorldScaleDisplayScale.Minimum);
                var max = decimal.ToDouble(this.numWorldScaleDisplayScale.Maximum);

                value = Math.Max(min, Math.Min(value, max));

                this.numWorldScaleDisplayScale.Value = new decimal(value);
            }
            finally
            {
                this.isLocked = false;
            }
        }

        /// <summary>
        /// Called when the WorldScaleDisplayScale NumericUpDown value changes.
        /// </summary>
        /// <param name="sender">The WorldScaleDisplayScale NumericUpDown.</param>
        /// <param name="e">Event argument.</param>
        private void OnWorldScaleDisplayScaleNumericUpDownValueChanged(object sender, EventArgs e)
        {
            if (this.isLocked)
            {
                return;
            }

            this.isLocked = true;
            try
            {
                decimal value = this.numWorldScaleDisplayScale.Value * 1000m;
                this.trkWorldScaleDisplayScale.Value = (int)value;
            }
            finally
            {
                this.isLocked = false;
            }
        }
    }
}
