﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using EffectMaker.Application.AutoBackup;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Extensions;

namespace EffectMaker.Application.OptionPanes
{
    /// <summary>
    /// Details options.
    /// </summary>
    [DisplayOrder(4)]
    public partial class DetailsOptionPane : UserControl, IOptionPane
    {
        /// <summary>
        /// Initializes the DetailsOptionPane instance.
        /// </summary>
        public DetailsOptionPane()
        {
            this.InitializeComponent();

            this.chkAutoBackupIsEnabled.CheckedChanged += this.OnAutoBackupIsEnabledCheckBoxCheckedChanged;
            this.chkAutoBackupDeleteOldBackupAtStartup.CheckedChanged += this.OnAutoBackupDeleteOldBackupAtStartupCheckBoxCheckedChanged;

            this.btnAutoBackupDeleteAllBackupFiles.Click += this.OnAutoBackupDeleteAllBackupFilesButtonClick;
            this.btnAutoBackupOpenFolder.Click += this.OnOpenAutoBckupButtonClick;
        }

        /// <summary>
        /// オプションウィンドウのTreeViewを再描画するdelegate
        /// </summary>
        public Action DrawTreeView
        {
            get;
            set;
        }

        /// <summary>
        /// Gets the identifier.
        /// </summary>
        public string Identifier
        {
            get { return "DetailsOptionPane"; }
        }

        /// <summary>
        /// Gets the parent identifier.
        /// </summary>
        public string ChildOf
        {
            get { return "UserConfig"; }
        }

        /// <summary>
        /// Gets the display name.
        /// </summary>
        public string DisplayName
        {
            get { return Properties.Resources.OptionPaneDetails; }
        }

        /// <summary>
        /// Called when initialized.
        /// </summary>
        public void OnInitialize()
        {
            DetailsOptions options = OptionStore.RootOptions.Details;

            this.numMemoryManagementMaxCacheSize.SetValueClamped(options.MaximumTextureCacheSize);

            this.chkAutoBackupIsEnabled.Checked = options.IsAutoBackupActive;
            this.numAutoBackupInterval.SetValueClamped(options.BackupInterval);

            this.chkAutoBackupDeleteOldBackupAtStartup.Checked = options.DeleteOldBackupFilesAtStartup;
            this.numAutoBackupDaysDelay.SetValueClamped(options.DeleteDaysLater);
        }

        /// <summary>
        /// Called when terminated.
        /// </summary>
        public void OnTerminate()
        {
        }

        /// <summary>
        /// Called when the Accept button is clicked.
        /// </summary>
        /// <returns>入力に不正がなければtrue,入力が不正であればfalse.</returns>
        public bool OnAccept()
        {
            DetailsOptions options = OptionStore.RootOptions.Details;

            options.MaximumTextureCacheSize = (int)this.numMemoryManagementMaxCacheSize.Value;

            // TextureManagerのキャッシュサイズの最大値を変更する
            EffectMaker.BusinessLogic.Manager.TextureManager.Instance.SetMaxCapacity(options.MaximumTextureCacheSize * 1024 * 1024);

            options.IsAutoBackupActive = this.chkAutoBackupIsEnabled.Checked;
            options.BackupInterval = (int)this.numAutoBackupInterval.Value;

            options.DeleteOldBackupFilesAtStartup = this.chkAutoBackupDeleteOldBackupAtStartup.Checked;
            options.DeleteDaysLater = (int)this.numAutoBackupDaysDelay.Value;

            return true;
        }

        /// <summary>
        /// Called when the Cancel button is clicked.
        /// </summary>
        public void OnCancel()
        {
        }

        /// <summary>
        /// ユーザーコンフィグが読み込まれた際に、その内容にUIを更新します。
        /// </summary>
        public void OnProjectSettingChanged()
        {
        }

        /// <summary>
        /// パス設定項目がないので常時trueを返します。
        /// </summary>
        /// <returns>
        /// 常時true.
        /// </returns>
        public bool OnValidationCheck()
        {
            return true;
        }

        /// <summary>
        /// プロジェクトコンフィグをファイル出力する際に各UIの内容を出力用インスタンスに収集します。
        /// </summary>
        public void OnExportProjectSetting()
        {
        }

        /// <summary>
        /// Called when the AutoBackupIsEnabled CheckBox is checked or unchecked.
        /// </summary>
        /// <param name="sender">The AutoBackupIsEnabled CheckBox.</param>
        /// <param name="e">Event argument.</param>
        private void OnAutoBackupIsEnabledCheckBoxCheckedChanged(object sender, EventArgs e)
        {
            var autoBackupIsEnabled = this.chkAutoBackupIsEnabled.Checked;

            this.lblAutoBackupInterval.Enabled = autoBackupIsEnabled;
            this.numAutoBackupInterval.Enabled = autoBackupIsEnabled;
            this.lblAutoBackupIntervalMinutes.Enabled = autoBackupIsEnabled;

            var deleteOldBackupIsEnabled = this.chkAutoBackupDeleteOldBackupAtStartup.Checked;

            this.chkAutoBackupDeleteOldBackupAtStartup.Enabled = autoBackupIsEnabled;
            this.numAutoBackupDaysDelay.Enabled = autoBackupIsEnabled && deleteOldBackupIsEnabled;
            this.lblAutoBackupDaysLater.Enabled = autoBackupIsEnabled && deleteOldBackupIsEnabled;
        }

        /// <summary>
        /// Called when the AutoBackupDeleteOldBackupAtStartup CheckBox is checked or unchecked.
        /// </summary>
        /// <param name="sender">The AutoBackupDeleteOldBackupAtStartup CheckBox.</param>
        /// <param name="e">Event argument.</param>
        private void OnAutoBackupDeleteOldBackupAtStartupCheckBoxCheckedChanged(object sender, EventArgs e)
        {
            var isEnabled = this.chkAutoBackupDeleteOldBackupAtStartup.Checked;

            this.numAutoBackupDaysDelay.Enabled = isEnabled;
            this.lblAutoBackupDaysLater.Enabled = isEnabled;
        }

        /// <summary>
        /// Called when the AutoBackupDeleteAllBackupFiles Button is clicked.
        /// </summary>
        /// <param name="sender">The AutoBackupDeleteAllBackupFiles Button.</param>
        /// <param name="e">Event argument.</param>
        private void OnAutoBackupDeleteAllBackupFilesButtonClick(object sender, EventArgs e)
        {
            AppData.AutoBackupManager.DeleteAutoBackupFiles(false);
        }

        /// <summary>
        /// オートバックアップフォルダを開きます。
        /// </summary>
        /// <param name="sender">使用しません</param>
        /// <param name="e">使用しません</param>
        private void OnOpenAutoBckupButtonClick(object sender, EventArgs e)
        {
            string path = AppData.AutoBackupManager.AutoBackupFolderPath;
            System.Diagnostics.Process.Start("EXPLORER.EXE", path);
        }
    }
}
