﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Imaging;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;

namespace EffectMaker.Application
{
    /// <summary>
    /// オプションウィンドウ向けTreeView
    /// </summary>
    partial class OptionPaneTreeView : TreeView
    {
        /// <summary>
        /// Initializes the OptionPaneTreeView instance.
        /// </summary>
        public OptionPaneTreeView()
        {
            this.DrawMode = TreeViewDrawMode.OwnerDrawText;
        }

        /// <summary>
        /// Called when owner draing node is required.
        /// </summary>
        /// <param name="e">Event argument.</param>
        protected override void OnDrawNode(DrawTreeNodeEventArgs e)
        {
            base.OnDrawNode(e);

            OptionPaneTreeViewItem treeItem = e.Node as OptionPaneTreeViewItem;

            if (treeItem == null)
            {
                return;
            }

            e.DrawDefault = false;

            // Use the font for the node or for the tree view?
            var nameFont = treeItem.NodeFont;
            if (nameFont == null)
            {
                nameFont = this.Font;
            }

            // Decide the brush color.
            Brush nameBrush = Brushes.Black;
            if ((e.State & TreeNodeStates.Focused) != 0)
            {
                nameBrush = SystemBrushes.HighlightText;
            }

            if (treeItem.Validate() == false)
            {
                nameBrush = Brushes.Red;
            }

            // Measure text width for the file name.
            int fileNameWidth =
                (int)Math.Ceiling(e.Graphics.MeasureString(treeItem.Text, nameFont).Width);

            // テキストを表示する座標
            Point textPoint = new Point(treeItem.Bounds.Location.X, treeItem.Bounds.Location.Y);

            // paneの入力項目が不正であり、子ノードを持つ場合は、テキストを右に寄せる。
            if (treeItem.Nodes.Count > 0 && treeItem.Validate() == false)
            {
                textPoint.X += Properties.Resources.SymbolWarning.Width;
            }

            // Compute the render rectangles.
            Rectangle fileNameRect = new Rectangle(textPoint, new Size(fileNameWidth, e.Bounds.Height));

            // Create the string format for rendering.
            var fileNameFormat = new StringFormat()
            {
                Alignment = StringAlignment.Near,
                LineAlignment = StringAlignment.Center,
            };

            // Render!
            e.Graphics.DrawString(
                treeItem.Text,
                nameFont,
                nameBrush,
                fileNameRect,
                fileNameFormat);

            // paneの入力項目が不正な場合は、警告アイコンを表示する
            if (treeItem.Validate() == false)
            {
                ImageAttributes imgAttr = new ImageAttributes();
                imgAttr.SetColorKey(System.Drawing.Color.Magenta, System.Drawing.Color.Magenta);
                e.Graphics.DrawImage(
                                     Properties.Resources.SymbolWarning,
                                     new Rectangle(
                                         new Point(fileNameRect.Left - Properties.Resources.SymbolWarning.Width, fileNameRect.Top),
                                         Properties.Resources.SymbolWarning.Size),
                                     0,
                                     0,
                                     Properties.Resources.SymbolWarning.Width,
                                     Properties.Resources.SymbolWarning.Height,
                                     GraphicsUnit.Pixel,
                                     imgAttr);
            }
        }
    }
}
