﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using EffectMaker.BusinessLogic.IO;
using EffectMaker.BusinessLogic.Serializer;
using EffectMaker.DataModel.DataModels;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.Application.CommandLine
{
    /// <summary>
    /// コマンドラインからUDD2.0定義をデータモデル・Xamlファイル・バイナリコンバータに変換するプロセッサです。
    /// </summary>
    public class Udd2ConvertProcessor : CommandLineProcessorBase
    {
        /// <summary>
        /// 入力ファイルパス
        /// </summary>
        private string inputFile;

        /// <summary>
        /// 出力フォルダパス
        /// </summary>
        private string outputPath;

        /// <summary>
        /// エラーログ
        /// </summary>
        private string errorLog;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="workingData">作業データ</param>
        /// <param name="inputFile">入力ファイル</param>
        public Udd2ConvertProcessor(CommandLineWorkingData workingData, string inputFile)
            : base(workingData)
        {
            this.inputFile = PathUtility.ToAbsolutePath(inputFile, Directory.GetCurrentDirectory());
            this.outputPath = Path.GetDirectoryName(this.inputFile);
        }

        /// <summary>
        /// エラーログを取得します。
        /// </summary>
        public override string ErrorReport { get { return this.errorLog; } }

        /// <summary>
        /// 設定処理を行います。
        /// </summary>
        /// <returns>処理が正常に完了したときtrue, それ以外はfalseを返します。</returns>
        public override bool Setup()
        {
            return true;
        }

        /// <summary>
        /// コマンドライン処理を行います。
        /// </summary>
        /// <returns>処理が正常に完了したときtrue, それ以外はfalseを返します。</returns>
        public override bool Process()
        {
            Logger.Log(
                "Console",
                LogLevels.Information,
                "Start Udd 2.0 convert...");

            bool result = true;
            var conv = new Udd2Converter.Udd2Converter();
            if (!conv.LoadDefinition(this.inputFile))
            {
                this.errorLog = conv.ErrorReport;
                result = false;
            }
            else if (!conv.ConvertToFile(this.outputPath))
            {
                this.errorLog = conv.ErrorReport;
                result = false;
            }
            else
            {
                Logger.Log(
                    "Console",
                    LogLevels.Information,
                    "Convert succeed: {0}",
                    this.inputFile);
            }

            return result;
        }

        /// <summary>
        /// 書き換えた設定などを復元します。
        /// </summary>
        public override void Cleanup()
        {
        }
    }
}
