﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.Application.CommandLine
{
    /// <summary>
    /// 静的解析オプションを処理します。
    /// </summary>
    public class ProfileModeProcessor : CommandLineProcessorBase
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="workingData">作業データ</param>
        /// <param name="outputPath">出力ファイルパス</param>
        public ProfileModeProcessor(CommandLineWorkingData workingData, string outputPath)
            : base(workingData)
        {
            this.OutputPath = outputPath;
        }

        /// <summary>
        /// エラーログを取得します。
        /// </summary>
        public override string ErrorReport { get { return string.Empty; } }

        /// <summary>
        /// 解析結果の出力パスを取得します。
        /// </summary>
        public string OutputPath { get; }

        /// <summary>
        /// 設定処理を行います。
        /// </summary>
        /// <returns>処理が正常に完了したときtrue, それ以外はfalseを返します。</returns>
        public override bool Setup()
        {
            // 出力ファイルパスを設定
            this.WorkingData.Profiler.OutputPath = PathUtility.ToAbsolutePath(this.OutputPath, TheApp.WorkingFolder);
            return true;
        }

        /// <summary>
        /// コマンドライン処理を行います。
        /// </summary>
        /// <returns>処理が正常に完了したときtrue, それ以外はfalseを返します。</returns>
        public override bool Process()
        {
            CommandLineBinaryConverter converter = this.WorkingData.Converter;
            StaticProfiler profiler = this.WorkingData.Profiler;

            // バイナリ出力スイッチが指定されていなかったとき、静的解析用にバイナリコンバートを行う
            if (converter.OutputBinaryEnabled == false)
            {
                bool convResult = converter.Convert();
                if (convResult == false)
                {
                    return false;
                }
            }

            // 静的解析を実行
            profiler.Analyze(converter.EmitterSetList, converter.Session);

            // 解析結果を出力
            profiler.Output();

            return true;
        }

        /// <summary>
        /// 書き換えた設定などを復元します。
        /// </summary>
        public override void Cleanup()
        {
        }
    }
}
