﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Linq;
using System.Text;
using EffectMaker.Application.Properties;
using EffectMaker.BusinessLogic.SpecDefinitions;
using Nintendo.Foundation.IO;

namespace EffectMaker.Application.CommandLine
{
    /// <summary>
    /// コマンドラインパラメータです。
    /// </summary>
    public class CommandLineParams
    {
        /// <summary>
        /// 位置引数（オプションではない、名前のない「引数」、ToolFoundation 参照）を取得または設定します。
        /// GUI モードで起動時に開く中間ファイルの指定に利用します。
        /// </summary>
        [CommandLineValues(ValueName = "startupFiles", Description = "ConsoleDescStartupFiles", DescriptionConverterName = "LocalizeDescription")]
        public string[] StartupFiles { get; set; }

        /// <summary>
        /// 出力バイナリファイル名を取得または設定します。
        /// </summary>
        [CommandLineOption('o', "output", Description = "ConsoleDescOutput", ValueName = "output.ptcl", DescriptionConverterName = "LocalizeDescription")]
        public string OutputBinaryFile { get; set; }

        /// <summary>
        /// コンバートリストファイルを取得または設定します。
        /// </summary>
        [CommandLineOption("convert-list", Description = "ConsoleDescConvertList", ValueName = "convert.xml", DescriptionConverterName = "LocalizeDescription")]
        public string ConvertListFile { get; set; }

        /// <summary>
        /// プロジェクト設定ファイルを取得または設定します。
        /// </summary>
        [CommandLineOption("team-setting-file", Description = "ConsoleDescProjectConfig", ValueName = "setting.xml", DescriptionConverterName = "LocalizeDescription")]
        public string ProjectConfigFile { get; set; }

        /// <summary>
        /// 出力ヘッダファイル名を取得または設定します。
        /// </summary>
        [CommandLineOption("output-c-header", Description = "ConsoleDescEsetHeader", ValueName = "header", DescriptionConverterName = "LocalizeDescription")]
        public string EsetHeaderFile { get; set; }

        /// <summary>
        /// ターゲットAPIタイプを取得または設定します。
        /// </summary>
        [CommandLineOption('a', "api-type", DescriptionPropertyName = "ApiTypeDescription", ValueName = "apiTypeName")]
        public string ApiType { get; set; }

        /// <summary>
        /// シェーダのコードタイプを取得または設定します。
        /// </summary>
        [CommandLineOption('c', "code-type", DescriptionPropertyName = "CodeTypeDescription", ValueName = "codeTypeName")]
        public string CodeType { get; set; }

        /// <summary>
        /// テクスチャタイリングモードを取得または設定します。
        /// </summary>
        [CommandLineOption("tile-mode", DescriptionPropertyName = "TileModeDescription", ValueName = "tileModeName")]
        public string TileMode { get; set; }

        /// <summary>
        /// テクスチャタイリング最適化モードを取得または設定します。
        /// </summary>
        [CommandLineOption("tile-optimize", DescriptionPropertyName = "TileOptimizeDescription", ValueName = "tileOptimizeName")]
        public string TileOptimize { get; set; }

        /// <summary>
        /// テクスチャタイリングサイズ最適化のしきい値を取得または設定します。
        /// </summary>
        [CommandLineOption("tile-size-threshold", Description = "ConsoleDescTileSizeThreshold", ValueName = "tileSizeThreshold", DescriptionConverterName = "LocalizeDescription")]
        public int? TileSizeThreshold { get; set; }

        /// <summary>
        /// 入力する常駐リストファイルを取得または設定します。
        /// </summary>
        [CommandLineOption("input-resident-asset-list", Description = "ConsoleDescInputResidentAssetList", ValueName = "assetlist.xml", DescriptionConverterName = "LocalizeDescription")]
        public string InputResidentAssetListFile { get; set; }

        /// <summary>
        /// 出力する常駐リストファイルを取得または設定します。
        /// </summary>
        [CommandLineOption("output-resident-asset-list", Description = "ConsoleDescOutputResidentAssetList", ValueName = "assetlist.xml", DescriptionConverterName = "LocalizeDescription")]
        public string OutputResidentAssetListFile { get; set; }

        /// <summary>
        /// 入力する改行形式の常駐リテクスチャファイルを取得または設定します。
        /// </summary>
        [CommandLineOption("resident-texture-list", Description = "ConsoleDescInputResidentTextureList", ValueName = "textureList.txt", DescriptionConverterName = "LocalizeDescription")]
        public string ResidentTextureListFile { get; set; }

        /// <summary>
        /// 出力シェーダバイナリのファイルパスを取得または設定します。
        /// </summary>
        [CommandLineOption("output-shader-binary", Description = "ShaderBinaryOutput", ValueName = "shaderBinary.bnsh", DescriptionConverterName = "LocalizeDescription")]
        public string OutputShaderBinaryName { get; set; }

        /// <summary>
        /// 出力コンピュートシェーダバイナリのファイルパスを取得または設定します。
        /// </summary>
        [CommandLineOption("output-compute-shader-binary", Description = "ComputeShaderBinaryOutput", ValueName = "computeShaderBinary.bnsh", DescriptionConverterName = "LocalizeDescription")]
        public string OutputComputeShaderBinaryName { get; set; }

        /// <summary>
        /// テクスチャバイナリのファイルパスを取得または設定します。
        /// </summary>
        [CommandLineOption("output-texture-binary", Description = "TextureBinaryOutput", ValueName = "textureBinary.bntx", DescriptionConverterName = "LocalizeDescription")]
        public string OutputTextureBinaryName { get; set; }

        /// <summary>
        /// G3Dプリミティブバイナリのファイルパスを取得または設定します。
        /// </summary>
        [CommandLineOption("output-g3d-primitive-binary", Description = "G3dPrimitiveBinaryOutput", ValueName = "g3dPrimitiveBinary.bfres", DescriptionConverterName = "LocalizeDescription")]
        public string OutputG3dPrimitiveBinaryName { get; set; }

        /// <summary>
        /// アップデートするesetファイルを取得または設定します。
        /// </summary>
        [CommandLineOption("update", Description = "ConsoleDescUpdateListFile", ValueName = "update.xml", DescriptionConverterName = "LocalizeDescription")]
        public string UpdateListFile { get; set; }

        /// <summary>
        /// アップデートするesetファイルを取得または設定します。
        /// </summary>
        [CommandLineOption("update-nw4c-to-nsdk", Description = "ConsoleDescUpdateCE2ListFile", ValueName = "convertlist.xml", DescriptionConverterName = "LocalizeDescription")]
        public string UpdateCE2ListFile { get; set; }

        /// <summary>
        /// 詳細表示するかどうか取得または設定します。
        /// </summary>
        [CommandLineOption('v', "verbose", Description = "ConsoleDescVerboseOutputMode", DescriptionConverterName = "LocalizeDescription")]
        public bool IsVerboseOutputMode { get; set; }

        /// <summary>
        /// サイレントモードかどうか取得または設定します。
        /// </summary>
        [CommandLineOption('s', "silent", Description = "ConsoleDescSilent", DescriptionConverterName = "LocalizeDescription")]
        public bool IsSilentMode { get; set; }

        /// <summary>
        /// 静的解析の解析結果ファイルの出力先パスを取得または設定します。
        /// </summary>
        [CommandLineOption("profile", Description="ConsoleDescProfileOutputPath", ValueName ="result.csv", DescriptionConverterName = "LocalizeDescription")]
        public string ProfileOutputPath { get; set; }

        /// <summary>
        /// 並列実行するジョブ数を取得または設定します。
        /// </summary>
        [CommandLineOption('j', "jobs", Description = "ConsoleDescJobsNumber", ValueName = "jobs", DescriptionConverterName = "LocalizeDescription")]
        public int JobsNumber { get; set; }

        /// <summary>
        /// シェーダコンバートオプションを取得または設定します。
        /// </summary>
        [CommandLineOption("shader-option", Description = "ConsoleDescShaderOption", ValueName = "option", DescriptionConverterName = "LocalizeDescription")]
        public string ShaderConvertOption { get; set; }

        /// <summary>
        /// シェーダソースコードを省略するかどうか取得または設定します。
        /// </summary>
        [CommandLineOption("no-shader-source", Description = "ConsoleDescNoShaderSource", DescriptionConverterName = "LocalizeDescription")]
        public bool NoShaderSource { get; set; }

        /// <summary>
        /// エミッタセットのファイルパスを含めるかどうか取得または設定します。
        /// </summary>
        [CommandLineOption("include-emitter-set-file-path-array", Description = "ConsoleDescNoShaderSource", DescriptionConverterName = "LocalizeDescription")]
        public bool IncludeEmitterSetFilePathArray { get; set; }

        /// <summary>
        /// シェーダキャッシュディレクトリを取得または設定します。
        /// </summary>
        [CommandLineOption("shader-cache-directory", Description = "ConsoleDescShaderCacheDirectory", ValueName="shaderCacheDirectory", DescriptionConverterName = "LocalizeDescription")]
        public string ShaderCacheDirectory { get; set; }

        /// <summary>
        /// UDD2.0のコンバートを行うかどうか取得または設定します。
        /// </summary>
        [CommandLineOption("udd2-convert", Description = "ConsoleDescUdd2Convert", ValueName = "definitionFile", DescriptionConverterName = "LocalizeDescription", IsHidden = true)]
        public string Udd2Convert { get; set; }

        /// <summary>
        /// バイナリファイルバージョンの上書きバージョンを取得または設定します。
        /// バージョン番号は16進数の文字列で表現します。
        /// </summary>
        [CommandLineOption("overwrite-binary-file-version", Description = "ConsoleDescOverwriteBinaryFileVesion", ValueName ="version", DescriptionConverterName = "LocalizeDescription", IsHidden = true)]
        public string OverwriteBinaryFileVersion { get; set; }

        /// <summary>
        /// api-typeオプションの説明を取得します。
        /// </summary>
        public static string ApiTypeDescription
        {
            get
            {
                bool first = true;
                var usage = new StringBuilder();
                usage.Append("[");

                var gfxSpecs = SpecManager.SpecDefinitions.Where(s => s.GraphicsApi == s.RecommendName).ToArray();

                foreach (var spec in gfxSpecs)
                {
                    if (!first)
                    {
                        usage.Append(" |");
                    }
                    else
                    {
                        first = false;
                    }

                    usage.Append(" " + spec.RecommendName.ToLower());
                }

                foreach (var spec in SpecManager.SpecDefinitions.Where(s => !gfxSpecs.Contains(s)))
                {
                    if (!first)
                    {
                        usage.Append(" |");
                    }
                    else
                    {
                        first = false;
                    }

                    usage.Append(" " + spec.RecommendName.ToLower());
                }

                usage.Append(" ]");

                return string.Format(Properties.Resources.ConsoleDescApiType, usage);
            }
        }

        /// <summary>
        /// code-typeオプションの説明を取得します。
        /// </summary>
        public static string CodeTypeDescription
        {
            get
            {
                var usage = new StringBuilder();
                var gfxSpecs = SpecManager.SpecDefinitions.Where(s => s.RecommendName == s.GraphicsApi).ToArray();

                if (gfxSpecs.Any())
                {
                    foreach (var spec in gfxSpecs.Where(s => s.ShaderConversionOption.ValidCodeTypes != null))
                    {
                        bool first = true;
                        usage.Append("api-type=" + spec.GraphicsApi.ToLower() + " [");
                        foreach (var codeType in spec.ShaderConversionOption.ValidCodeTypes)
                        {
                            if (!first)
                            {
                                usage.Append(" |");
                            }
                            else
                            {
                                first = false;
                            }

                            usage.Append(" " + codeType);
                        }

                        usage.AppendLine(" ]");
                    }
                }

                if (usage.Length == 0)
                {
                    usage.Append("[ " + Resources.CodeTypeDescriptionNotFoundGfxSpecs + " ]");
                }

                return string.Format(Properties.Resources.ConsoleDescCodeType, usage);
            }
        }

        /// <summary>
        /// tile-modeオプションの説明を取得します。
        /// </summary>
        public static string TileModeDescription
        {
            get
            {
                var usage = new StringBuilder();
                var gfxSpecs = SpecManager.SpecDefinitions.Where(s => s.RecommendName == s.GraphicsApi).ToArray();

                if (gfxSpecs.Any())
                {
                    foreach (var spec in gfxSpecs.Where(s => s.TexturePackingOption != null && s.TexturePackingOption.ValidTileModes != null))
                    {
                        bool first = true;
                        usage.Append("api-type=" + spec.GraphicsApi.ToLower() + " [");

                        foreach (var tileMode in spec.TexturePackingOption.ValidTileModes)
                        {
                            if (!first)
                            {
                                usage.Append(" |");
                            }
                            else
                            {
                                first = false;
                            }

                            usage.Append(" " + tileMode);
                        }

                        usage.AppendLine(" ]");
                    }
                }

                if (usage.Length == 0)
                {
                    usage.Append("[ " + Resources.TileModeDescriptionNotFoundGfxSpecs + " ]");
                }

                return string.Format(Properties.Resources.ConsoleDescTileMode, usage);
            }
        }

        /// <summary>
        /// tile-optimizeオプションの説明を取得します。
        /// </summary>
        public static string TileOptimizeDescription
        {
            get
            {
                var usage = new StringBuilder();
                var gfxSpecs = SpecManager.SpecDefinitions.Where(s => s.RecommendName == s.GraphicsApi).ToArray();

                if (gfxSpecs.Any())
                {
                    foreach (var spec in gfxSpecs.Where(s => s.TexturePackingOption != null && s.TexturePackingOption.ValidTileOptimizes != null))
                    {
                        bool first = true;
                        usage.Append("api-type=" + spec.GraphicsApi.ToLower() + " [");

                        foreach (var tileOptimize in spec.TexturePackingOption.ValidTileOptimizes)
                        {
                            if (!first)
                            {
                                usage.Append(" |");
                            }
                            else
                            {
                                first = false;
                            }

                            usage.Append(" " + tileOptimize);
                        }

                        usage.AppendLine(" ]");
                    }
                }

                if (usage.Length == 0)
                {
                    usage.Append("[ " + Resources. TileOptimizeDescriptionNotFoundGfxSpecs + " ]");
                }

                return string.Format(Properties.Resources.ConsoleDescTileOptimize, usage);
            }
        }

        /// <summary>
        /// ローカライズしたオプションの説明を取得します。
        /// </summary>
        /// <param name="description">Descriptionプロパティの値</param>
        /// <param name="valueName">引数名またはオプション名</param>
        /// <returns>ローカライズされたコマンドラインオプションの説明を返します。</returns>
        public static string LocalizeDescription(string description, string valueName)
        {
            string result = Properties.Resources.ResourceManager.GetString(description, Properties.Resources.Culture);
            System.Diagnostics.Debug.Assert(result != null, "コマンドオプションの Description が null");

            return result;
        }
    }

    /// <summary>
    /// 旧形式を含めたコマンドラインパラメータです。
    /// </summary>
    public class CompatibleCommandLineParams : CommandLineParams
    {
        /// <summary>
        /// 旧形式で記述されたConvertListFileを取得または設定します。
        /// </summary>
        [CommandLineOption("convlist", Description = "ConsoleDescConvertList", ValueName = "convert.xml")]
        public string ObsoleteConvertListFile
        {
            get
            {
                return this.ConvertListFile;
            }

            set
            {
                this.ConvertListFile = value;
                this.UsingObsoleteConvertListFile = true;
            }
        }

        /// <summary>
        /// 旧形式で記述されたConvertListFileが使われているかどうか取得します。
        /// </summary>
        public bool UsingObsoleteConvertListFile { get; private set; }

        /// <summary>
        /// 旧形式で記述されたProjectConfigFileを取得または設定します。
        /// </summary>
        [CommandLineOption("prjconfig", Description = "ConsoleDescProjectConfig")]
        public string ObsoleteProjectConfigFile
        {
            get
            {
                return this.ProjectConfigFile;
            }

            set
            {
                this.ProjectConfigFile = value;
                this.UsingObsoleteProjectConfigFile = true;
            }
        }

        /// <summary>
        /// 旧形式で記述されたProjectConfigFileが使われているかどうか取得します。
        /// </summary>
        public bool UsingObsoleteProjectConfigFile { get; private set; }

        /// <summary>
        /// 旧形式で記述されたShaderConvertOptionを取得または設定します。
        /// </summary>
        [CommandLineOption("shaderOpt", Description = "ConsoleDescShaderOption")]
        public string ObsoleteShaderConvertOption
        {
            get
            {
                return this.ShaderConvertOption;
            }

            set
            {
                this.ShaderConvertOption = value;
                this.UsingObsoleteShaderConvertOption = true;
            }
        }

        /// <summary>
        /// 旧形式で記述されたShaderConvertOptionが使われているかどうか取得します。
        /// </summary>
        public bool UsingObsoleteShaderConvertOption { get; private set; }

        /// <summary>
        /// 旧形式で記述されたNoShaderSourceを取得または設定します。
        /// </summary>
        [CommandLineOption("noShaderSrc", Description = "ConsoleDescNoShaderSource")]
        public bool ObsoleteNoShaderSource
        {
            get
            {
                return this.NoShaderSource;
            }

            set
            {
                this.NoShaderSource = value;
                this.UsingObsoleteNoShaderSource = true;
            }
        }

        /// <summary>
        /// 旧形式で記述されたNoShaderSourceが使われているかどうか取得します。
        /// </summary>
        public bool UsingObsoleteNoShaderSource { get; private set; }

        /// <summary>
        /// 旧形式で記述されたVerboseOutputModeを取得または設定します。
        /// </summary>
        [CommandLineOption('d', Description = "ConsoleDescVerboseOutputMode")]
        public bool ObsoleteVerboseOutputMode
        {
            get
            {
                return this.IsVerboseOutputMode;
            }

            set
            {
                this.IsVerboseOutputMode = value;
                this.UsingObsoleteVerboseMode = true;
            }
        }

        /// <summary>
        /// 旧形式で記述されたVerboseOutputModeが使われているかどうか取得します。
        /// </summary>
        public bool UsingObsoleteVerboseMode { get; private set; }

        /// <summary>
        /// 旧形式で記述されたPlatformが使われているかどうか取得します。
        /// </summary>
        [CommandLineOption("spec", DescriptionPropertyName = "ApiTypeDescription", ValueName = "apiTypeName")]
        public string ObsoleteSpec
        {
            get
            {
                return this.ApiType;
            }

            set
            {
                this.ApiType = value;
                this.UsingObsoleteSpec = true;
            }
        }

        /// <summary>
        /// 旧形式で記述されたSpecが使われているかどうか取得します。
        /// </summary>
        public bool UsingObsoleteSpec { get; private set; }

        /// <summary>
        /// 旧形式で記述されたPlatformが使われているかどうか取得します。
        /// </summary>
        [CommandLineOption('p', "platform", DescriptionPropertyName = "ApiTypeDescription", ValueName = "apiTypeName")]
        public string ObsoletePlatform
        {
            get
            {
                return this.ApiType;
            }

            set
            {
                this.ApiType = value;
                this.UsingObsoletePlatform = true;
            }
        }

        /// <summary>
        /// 旧形式で記述されたPlatformが使われているかどうか取得します。
        /// </summary>
        public bool UsingObsoletePlatform { get; private set; }
    }
}
