﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Globalization;
using System.IO;
using System.Linq;
using System.Text;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Forms;
using System.Windows.Input;
using CombinerNodeEditor.ViewModel;

namespace CombinerNodeEditor
{
    /// <summary>
    /// 返り値型の enum. そのまま使うとコード上、予約語に引っかかるので別名で定義して管理。
    /// </summary>
    public enum ValueType
    {
        Void,   // void
        Int,    // int
        Uint,   // uint
        Float,  // float
        Vec2,   // vec2
        Vec3,   // vec3
        Vec4,   // vec4
        Ivec2,  // ivec2
        Ivec3,  // ivec3
        Ivec4,  // ivec4
        Sampler2D,  // sampler2D
    };

    /// <summary>
    /// 実際の型名を文字列として取り出すための拡張
    /// </summary>
    public static class ValueTypeExt
    {
        public static string ToStringExt(this ValueType path)
        {
            switch (path)
            {
                case ValueType.Void: return "void";
                case ValueType.Int: return "int";
                case ValueType.Uint: return "uint";
                case ValueType.Float: return "float";
                case ValueType.Vec2: return "vec2";
                case ValueType.Vec3: return "vec3";
                case ValueType.Vec4: return "vec4";
                case ValueType.Ivec2: return "ivec2";
                case ValueType.Ivec3: return "ivec3";
                case ValueType.Ivec4: return "ivec4";
                case ValueType.Sampler2D: return "sampler2D";
                default: throw new ArgumentOutOfRangeException("e");
            }
        }

        public static ValueType FromString(string str)
        {
            switch (str)
            {
                case "void": return ValueType.Void;
                case "int": return ValueType.Int;
                case "uint": return ValueType.Uint;
                case "float": return ValueType.Float;
                case "vec2": return ValueType.Vec2;
                case "vec3": return ValueType.Vec3;
                case "vec4": return ValueType.Vec4;
                case "ivec2": return ValueType.Ivec2;
                case "ivec3": return ValueType.Ivec3;
                case "ivec4": return ValueType.Ivec4;
                case "sampler2D":return ValueType.Sampler2D;
                default: throw new ArgumentOutOfRangeException("e");
            }
        }
    };

    public enum UniformType
    {
        None,   // none
        File,   // file
        Int,    // int
        Uint,   // uint
        Float,  // float
        Vec2,   // vec2
        Vec3,   // vec3
        Vec4,   // vec4
    }

    /// <summary>
    /// 実際の型名を文字列として取り出すための拡張
    /// </summary>
    public static class UniformTypeExt
    {
        public static string ToStringExt(this UniformType path)
        {
            switch (path)
            {
                case UniformType.None: return "none";
                case UniformType.File: return "file";
                case UniformType.Int: return "int";
                case UniformType.Uint: return "uint";
                case UniformType.Float: return "float";
                case UniformType.Vec2: return "vec2";
                case UniformType.Vec3: return "vec3";
                case UniformType.Vec4: return "vec4";
                default: throw new ArgumentOutOfRangeException("e");
            }
        }

        public static UniformType FromString(string str)
        {
            switch (str)
            {
                case "none":return UniformType.None;
                case "file": return UniformType.File;
                case "int": return UniformType.Int;
                case "uint": return UniformType.Uint;
                case "float": return UniformType.Float;
                case "vec2": return UniformType.Vec2;
                case "vec3": return UniformType.Vec3;
                case "vec4": return UniformType.Vec4;
                default: throw new ArgumentOutOfRangeException("e");
            }
        }
    };

    public enum UniformExtType
    {
        TGA,
        FTXB
    }

    /// <summary>
    /// 実際の型名を文字列として取り出すための拡張
    /// </summary>
    public static class UniformExtTypeExt
    {
        public static string ToStringExt(this UniformExtType path)
        {
            switch (path)
            {
                case UniformExtType.TGA: return ".tga";
                case UniformExtType.FTXB: return ".ftxb";
                default: throw new ArgumentOutOfRangeException("e");
            }
        }

        public static UniformExtType FromString(string str)
        {
            switch (str)
            {
                case ".tga": return UniformExtType.TGA;
                case ".ftxb": return UniformExtType.FTXB;
                default: throw new ArgumentOutOfRangeException("e");
            }
        }
    };


    /// <summary>
    /// ノードの入力の定義
    /// </summary>
    public class InputDesc : ViewModelBase
    {
        private ValueType type;
        private string displayName;
        private string argTarget;
        private string semantics;
        private NodeDesc Parent { get; }

        /// <summary>
        /// 入力の型
        /// </summary>
        public ValueType Type
        {
            get { return this.type; }
            set
            {
                if (this.type == value) { return; }
                type = value;
                NotifyPropertyChanged("Type");

                // TODO: 理想的には ViewModel を用意し、それが親を知っているのでそちら側で呼び出すのが筋。
                //       今は ViewModel を用意するのを後に回すため、とりあえず雑にこれで実現する。
                //       InputDesc, OutputDesc の他のプロパティも同様。
                Parent.ReGenerateShaderCode();
            }
        }
        /// <summary>
        /// 表示名
        /// </summary>
        public string DisplayName
        {
            get { return this.displayName; }
            set
            {
                if (this.displayName == value) { return; }
                displayName = value;
                NotifyPropertyChanged("DisplayName");
                Parent.ReGenerateShaderCode();
            }
        }

        public string ArgTarget
        {
            get { return this.argTarget; }
            set
            {
                if (this.argTarget == value) { return; }
                argTarget = value;
                NotifyPropertyChanged("ArgTarget");
                Parent.ReGenerateShaderCode();
            }
        }

        public string Semantics
        {
            get { return this.semantics; }
            set
            {
                if (this.semantics == value) { return; }
                semantics = value;
                NotifyPropertyChanged("Semantics");
                Parent.ReGenerateShaderCode();
            }
        }

        /// <summary>
        /// 初期値を入れるためのコンストラクタ
        /// </summary>
        public InputDesc(NodeDesc parent)
        {
            this.type = ValueType.Float;
            this.displayName = "value";
            this.argTarget = "v";
            this.semantics = "";
            this.Parent = parent;
        }
    };

    public class OutputDesc : ViewModelBase
    {
        private ValueType type;
        private string displayName;
        private string returnTarget;
        private string semantics;
        private NodeDesc Parent { get; }

        /// <summary>
        /// 出力の型
        /// </summary>
        public ValueType Type
        {
            get { return this.type; }
            set
            {
                if (this.type == value) { return; }
                type = value;
                NotifyPropertyChanged("Type");
                Parent.ReGenerateShaderCode();
            }
        }

        /// <summary>
        /// 表示名
        /// </summary>
        public string DisplayName
        {
            get { return this.displayName; }
            set
            {
                if (this.displayName == value) { return; }
                displayName = value;
                NotifyPropertyChanged("DisplayName");
                Parent.ReGenerateShaderCode();
            }
        }

        /// <summary>
        /// 出力のマスク指定。基本的に "return" で返り値がそのまま渡される。
        /// 例えば、関数の返り値が vec4 のところで "return.xyz" のように記述すると、
        /// 返り値の xyz 要素で作られた vec3 を取り出すことができる。
        /// これを応用することで、
        /// ・返り値の一部分を取り出す（RGBのR要素だけ取り出す）
        /// ・返り値のチャンネル数別に出す（return.x / return.xxx で、同じ値の float / vec3 を両方出せる）
        /// ・（※処理にロスがあるが）複数の計算結果のマスク
        ///   最大 vec4 まで計算結果を返り値として持たせ、return.* で必要なものだけマスクして渡す
        /// といったトリックが可能。
        /// </summary>
        public string ReturnTarget
        {
            get { return this.returnTarget; }
            set
            {
                if (this.returnTarget == value) { return; }
                returnTarget = value;
                NotifyPropertyChanged("ReturnTarget");
                Parent.ReGenerateShaderCode();
            }
        }


        public string Semantics
        {
            get { return this.semantics; }
            set
            {
                if (this.semantics == value) { return; }
                semantics = value;
                NotifyPropertyChanged("Semantics");
                Parent.ReGenerateShaderCode();
            }
        }


        /// <summary>
        /// 初期値を入れるためのコンストラクタ
        /// </summary>
        public OutputDesc(NodeDesc parent)
        {
            this.type = ValueType.Float;
            this.displayName = "value";
            this.returnTarget = "return";
            this.semantics = "";
            this.Parent = parent;
        }
    };

    public class Preview
    {
        public bool IsEnabled { get; set; } = false;
        public float Width { get; set; } = 64;
        public float Height { get; set; } = 64;
    }

    public class UniformExtension : ViewModelBase
    {
        public UniformExtType Extension { get; set; }
        private bool enabled;
        public event EventHandler EnableValueChanged;

        public bool IsEnabled
        {
            get
            {
                return enabled;
            }
            set
            {
                if (enabled == value)
                    return;

                enabled = value;

                OnEnableValueChanged();
                NotifyPropertyChanged(nameof(IsEnabled));
            }
        }

        public UniformExtension(UniformExtType extension, bool enable)
        {
            Extension = extension;
            IsEnabled = enable;

        }

        private void OnEnableValueChanged()
        {
            EnableValueChanged?.Invoke(this, EventArgs.Empty);
        }
    }



    public class Uniform : ViewModelBase
    {
        public string Name { get; set; }
        public string DisplayName { get; set; }
        public ObservableCollection<UniformExtension> Extensions { get; set; }
        public UniformExtension ActiveExtensions
        {
            get
            {
                return Extensions.Where(p => p.IsEnabled).FirstOrDefault();
            }
        }


        private UniformType type;
        public UniformType Type
        {
            get
            {
                return type;
            }
            set
            {
                type = value;
                NotifyPropertyChanged(nameof(NameEnabled));
                NotifyPropertyChanged(nameof(DisplayNameEnabled));
                NotifyPropertyChanged(nameof(ExtensionEnabled));
            }
        }

        public bool NameEnabled
        {
            get
            {
                return Type != UniformType.None;
            }
        }

        public bool DisplayNameEnabled
        {
            get
            {
                return Type != UniformType.None && Type != UniformType.File;
            }
        }

        public bool ExtensionEnabled
        {
            get
            {
                return Type == UniformType.File;
            }
        }

        public Uniform()
        {
            Name = string.Empty;
            DisplayName = string.Empty;
            Type = UniformType.None;
            Extensions = new ObservableCollection<UniformExtension>();
            Extensions.Add(new UniformExtension(UniformExtType.FTXB, true));
            Extensions.Add(new UniformExtension(UniformExtType.TGA, false));

            foreach (var ext in Extensions)
            {
                ext.EnableValueChanged += Ext_EnableValueChanged;
            }
        }

        /// <summary>
        /// 必ず1つだけ選択できるように。（今後複数対応：CombinerEditorが複数に対応していない）
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void Ext_EnableValueChanged(object sender, EventArgs e)
        {
            // 指定したもの意外を全てfalseに。
            foreach (var ext in Extensions)
            {
                if(ext != sender)
                {
                    ext.IsEnabled = false;
                }
            }

            // リストが全てfalseなら、指定したものをtrueに
            if (Extensions.All(p => !p.IsEnabled))
            {
                UniformExtension ext = sender as UniformExtension;
                ext.IsEnabled = true;
            }
        }
        public void SetExtensionEnable(UniformExtType type, bool value)
        {
            foreach (var ext in Extensions)
            {
                if (ext.Extension == type)
                {
                    ext.IsEnabled = value;
                }
            }
        }
    }

    public class Block
    {
        public Guid Guid { get; set; } = Guid.Empty;
        public string DisplayName { get; set; } = "NewNodeName";
        public string Group { get; set; } = string.Empty;
    }

    public class NodeDesc : ViewModelBase, ITreeViewItem
    {
        private ValueType returnType;
        private string functionName;
        private bool isMacro;

        /// <summary>
        /// タグ名です。
        /// </summary>
        private string tagName;

        /// <summary>
        /// 選択状態です。
        /// </summary>
        private bool isSelected;

        /// <summary>
        /// func id, func name 共通。
        /// </summary>
        public string FunctionName
        {
            get { return functionName; }
            set
            {
                if (this.functionName == value) { return; }
                functionName = value;
                // シェーダ再生成
                ReGenerateShaderCode();
                NotifyPropertyChanged("FunctionName");
            }
        }

        public bool IsMacro
        {
            get { return isMacro; }
            set
            {
                if (this.isMacro == value) { return; }
                var prevValue = this.isMacro;
                this.isMacro = value;

                if (prevValue == false && this.ReturnType != ValueType.Void)
                {
                    // （返り値が void でない）関数→マクロのときは返り値があったら無くす
                    // ReturnType の out 出力を新設
                    var output = new OutputDesc(this);
                    output.DisplayName = "output";
                    output.ReturnTarget = "output";
                    output.Semantics = "";
                    output.Type = this.ReturnType;

                    this.InsertOutputDesc(0, output);
                    this.ReturnType = ValueType.Void;
                    for (int i = 0; i < this.OutputDescs.Count; ++i)
                    {
                        var o = this.OutputDescs[i];
                        if (o.ReturnTarget.Contains("return"))
                        {
                            // "return" 指定がある出力端子を削除。
                            this.RemoveAtOutputDesc(i);
                        }
                    }
                }
                else
                {
                    // マクロ→関数のときは
                    // - void で out を使ってやりとり
                    // - return でやりとり
                    // どちらでもあるのでとりあえずマクロのまま
                }

                // シェーダ再生成
                ReGenerateShaderCode();
                NotifyPropertyChanged("IsMacro");
            }
        }

        /// <summary>
        /// タグ名を取得します。
        /// </summary>
        public string TagName {
            get
            {
                return tagName;
            }

            set
            {
                if (this.tagName == value)
                {
                    return;
                }

                this.tagName = value;
                NotifyPropertyChanged(nameof(TagName));
            }
        }


        public string Description { get; set; }

        public ValueType ReturnType
        {
            get { return returnType; }
            set
            {
                if (this.returnType == value)
                {
                    return;
                }

                this.returnType = value;
                // シェーダ再生成
                ReGenerateShaderCode();
                NotifyPropertyChanged("ReturnType");
            }
        }

        public string OutputGlslFilePath { get; set; }

        public string ShaderCode { get; set; }

        public Preview Preview { get; set; } = new Preview();
        public Uniform Uniform { get; set; } = new Uniform();
        public Block Block { get; set; } = new Block();

        public void AddInputDesc(InputDesc desc)
        {
            this.InputDescs.Add(desc);
            // シェーダ再生成
            ReGenerateShaderCode();
            NotifyPropertyChanged("InputDescs");
        }

        public void RemoveAtInputDesc(int index)
        {
            this.InputDescs.RemoveAt(index);
            // シェーダ再生成
            ReGenerateShaderCode();
            NotifyPropertyChanged("InputDescs");
        }

        public void InsertInputDesc(int index, InputDesc desc)
        {
            this.InputDescs.Insert(index, desc);
            // シェーダ再生成
            ReGenerateShaderCode();
            NotifyPropertyChanged("InputDescs");
        }
        public void AddOutputDesc(OutputDesc desc)
        {
            this.OutputDescs.Add(desc);
            // シェーダ再生成
            ReGenerateShaderCode();
            NotifyPropertyChanged("OutputDescs");
        }

        public void RemoveAtOutputDesc(int index)
        {
            this.OutputDescs.RemoveAt(index);
            // シェーダ再生成
            ReGenerateShaderCode();
            NotifyPropertyChanged("OutputDescs");
        }

        public void InsertOutputDesc(int index, OutputDesc desc)
        {
            this.OutputDescs.Insert(index, desc);
            // シェーダ再生成
            ReGenerateShaderCode();
            NotifyPropertyChanged("OutputDescs");
        }

        public ObservableCollection<InputDesc> InputDescs { get; set; } = new ObservableCollection<InputDesc>();
        public ObservableCollection<OutputDesc> OutputDescs { get; set; } = new ObservableCollection<OutputDesc>();

        /// <summary>
        /// 選択状態を取得または設定します。
        /// </summary>
        public bool IsSelected
        {
            get
            {
                return isSelected;
            }
            set
            {
                if (this.isSelected != value)
                {
                    this.isSelected = value;
                    NotifyPropertyChanged(nameof(IsSelected));
                }
            }
        }

        public NodeDesc()
        {
            this.functionName = "NewFunctionName";
            TagName = "TagName";
            Description = "";
            OutputGlslFilePath = "";
            this.isMacro = false;
            this.returnType = ValueType.Void;

            // デフォルトのシェーダひな形だけ作っておく
            // MEMO: デフォルトではマクロではなく関数なのでそれで作成。
            var header = GenerateShaderFunctionHeader();
            ShaderCode =
                header +
                "{\n" +
                "    \n" +
                "}";
        }

        private string GenerateShaderFunctionHeader()
        {
            string args = "";

            // TODO: Input/Output を統合（でないと引数順を完全に制御できない）
            foreach (var i in this.OutputDescs)
            {
                // "return" を含まないものだけ out で出す
                if (!i.ReturnTarget.Contains("return"))
                {
                    args += string.Format(
                        "out {0} {1}, ",
                        i.Type.ToStringExt(),
                        i.ReturnTarget
                        );
                }
            }

            foreach (var i in this.InputDescs)
            {
                args += string.Format(
                    "{0} {1}, ",
                    i.Type.ToStringExt(),
                    i.ArgTarget
                    );
            }
            args = args.TrimEnd(new char[] { ',', ' ' });

            // 宣言部分を再構成
            var declaration = string.Format(
                "{0} {1}( {2} )\n",
                this.ReturnType.ToStringExt(),
                this.FunctionName,
                args);

            return declaration;
        }

        private string GenerateShaderMacroHeader()
        {
            string args = "";

            // TODO: Input/Output を統合（でないと引数順を完全に制御できない）
            foreach (var i in this.OutputDescs)
            {
                // "return" を含まないものだけ引数に含める
                if (!i.ReturnTarget.Contains("return"))
                {
                    args += string.Format(
                        "{0}, ",
                        i.ReturnTarget
                        );
                }
            }

            foreach (var i in this.InputDescs)
            {
                args += string.Format(
                    "{0}, ",
                    i.ArgTarget
                    );
            }
            args = args.TrimEnd(new char[] { ',', ' ' });

            // 宣言部分を再構成
            var declaration = string.Format(
                "#define {0}( {1} )\n",
                this.FunctionName,
                args);
            return declaration;
        }

        public void ReGenerateShaderCode()
        {
            // 今の設定に基づいて、 this.ShaderCode の宣言部分を更新
            if (this.ShaderCode.Length > 0)
            {
                if (this.IsMacro)
                {
                    // ========================================================
                    // MEMO: 以下の形式のマクロを想定し、テンプレート生成する
                    //       返り値は想定していない（宣言部分の抽出を共通化したいため）
                    //       中括弧の排除が必要になったら、 「#define で始まるかどうか」を基準に切り替えればよさそう。
                    // ========================================================
                    // #define MY_MACRO(argA, argB, argC) \
                    // {                                  \
                    //     <implementation>               \
                    // }                                  \
                    var splitPos = this.ShaderCode.IndexOf('{');
                    if (splitPos >= 0)
                    {
                        var implementation = this.ShaderCode.Substring(splitPos); // 残り
                        var declaration = GenerateShaderMacroHeader();
                        this.ShaderCode = declaration + implementation;

                        // "\\" を全部捨てて行末に付け直す（改行マクロ対応）
                        this.ShaderCode = this.ShaderCode.Replace("\r\n", "\n").Replace("\\", "").Replace("\n", "\\\n");

                        NotifyPropertyChanged("ShaderCode");
                    }
                    else
                    {
                        // この時は '{' が見つかっていないので何かおかしい
                    }
                }
                else
                {
                    // ========================================================
                    // MEMO: 以下の形式の関数を想定し、テンプレート生成を行う
                    // ========================================================
                    // <return_type> MyFucntion(<type> argA, <type> argB, <type> argC)
                    // {
                    //     <implementation>
                    // }
                    var splitPos = this.ShaderCode.IndexOf('{');
                    if (splitPos >= 0)
                    {
                        var implementation = this.ShaderCode.Substring(splitPos); // 残り
                        var declaration = GenerateShaderFunctionHeader();
                        this.ShaderCode = declaration + implementation;

                        // "\\" を捨てておく（マクロから戻ってきた場合の対応）
                        this.ShaderCode = this.ShaderCode.Replace("\r\n", "\n").Replace("\\", "");

                        NotifyPropertyChanged("ShaderCode");
                    }
                    else
                    {
                        // この時は '{' が見つかっていないので何かおかしい
                    }
                }
            }
        }
    }

    public class DataModel : ViewModelBase , ITreeViewItem
    {
        /// <summary>
        /// ワーキングディレクトリのパスです。
        /// </summary>
        private string workingDirPath;

        /// <summary>
        /// 選択アイテムです。
        /// </summary>
        private ITreeViewItem selectedItem;

        /// <summary>
        /// 選択状態です。
        /// </summary>
        private bool isSelected;

        /// <summary>
        /// 選択状態を取得または設定します。
        /// </summary>
        public bool IsSelected
        {
            get
            {
                return this.isSelected;
            }

            set
            {
                if (this.isSelected != value)
                {
                    this.isSelected = value;
                    this.NotifyPropertyChanged(nameof(this.IsSelected));
                }
            }
        }

        /// <summary>
        /// ワーキングディレクトリを取得または設定します。
        /// </summary>
        public string WorkingDirPath {
            get
            {
                return this.workingDirPath;
            }

            set
            {
                if (value != this.workingDirPath)
                {
                    this.workingDirPath = value;
                    this.NotifyPropertyChanged("WorkingDirPath");
                }
            }
        }

        /// <summary>
        /// ノード定義ファイルのリストです。
        /// </summary>
        public ObservableCollection<NodeFile> NodeFiles { get; set; } = new ObservableCollection<NodeFile>();

        public Dictionary<string, string> NoConvertCode { get; set; } = new Dictionary<string, string>();

        /// <summary>
        /// 選択アイテムを取得または設定します。
        /// </summary>
        public ITreeViewItem SelectedItem
        {
            get
            {
                return this.selectedItem;
            }

            set
            {
                if (this.selectedItem == value)
                {
                    return;
                }

                this.selectedItem = value;

                this.NotifyPropertyChanged(nameof(this.SelectedItem));
                this.NotifyPropertyChanged(nameof(this.SelectedNode));
                this.NotifyPropertyChanged(nameof(this.SelectedFile));
            }
        }

        /// <summary>
        /// 選択しているノードを取得します。
        /// </summary>
        public NodeDesc SelectedNode
        {
            get
            {
                if (this.SelectedItem is NodeDesc)
                {
                    return this.SelectedItem as NodeDesc;
                }

                return null;
            }
        }

        /// <summary>
        /// 選択しているファイルを取得します。
        /// </summary>
        public NodeFile SelectedFile
        {
            get
            {
                if (this.SelectedItem is NodeFile)
                {
                    return this.selectedItem as NodeFile;
                }

                if (this.SelectedItem is NodeDesc)
                {
                    return this.GetNodeFile((this.SelectedItem as NodeDesc).OutputGlslFilePath);
                }

                return null;
            }
        }

        /// <summary>
        /// ノード定義ファイルのインスタンスを取得します。
        /// </summary>
        /// <param name="filePath">ファイルパス</param>
        /// <returns>ノード定義ファイル</returns>
        public NodeFile GetNodeFile(string filePath)
        {
            return this.NodeFiles.Where(p => p.FilePath == filePath).FirstOrDefault();
        }
    }

    /// <summary>
    /// ノード定義ファイルのクラスです。
    /// </summary>
    public class NodeFile : ViewModelBase, ITreeViewItem
    {
        /// <summary>
        /// 選択状態です。
        /// </summary>
        private bool isSelected;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="filePath">ファイルパス</param>
        public NodeFile(string filePath)
        {
            this.FilePath = filePath;
        }

        /// <summary>
        /// 選択状態を取得または設定します。
        /// </summary>
        public bool IsSelected
        {
            get
            {
                return this.isSelected;
            }

            set
            {
                if (this.isSelected != value)
                {
                    this.isSelected = value;
                    this.NotifyPropertyChanged(nameof(this.IsSelected));
                }
            }
        }

        /// <summary>
        /// ファイルパスです。
        /// </summary>
        public string FilePath { get; private set; } = string.Empty;

        /// <summary>
        /// ノード定義リストです。
        /// </summary>
        public ObservableCollection<NodeDesc> NodeDescs { get; set; } = new ObservableCollection<NodeDesc>();

        /// <summary>
        /// ノードを追加します。
        /// </summary>
        /// <param name="nodeDesc">追加するノード</param>
        public void AddNode(NodeDesc nodeDesc)
        {
            this.NodeDescs.Add(nodeDesc);
        }

        /// <summary>
        /// ノードを削除します。
        /// </summary>
        /// <param name="nodeDesc">削除するノード</param>
        public void RemoveNode(NodeDesc nodeDesc)
        {
            this.NodeDescs.Remove(nodeDesc);
        }

        /// <summary>
        /// ノードを挿入します。
        /// </summary>
        /// <param name="index">要素番号</param>
        /// <param name="nodeDesc">挿入するノード</param>
        public void InsertNode(int index, NodeDesc nodeDesc)
        {
            this.NodeDescs.Insert(index, nodeDesc);
        }

        /// <summary>
        /// ノードを指定した要素番号に移動します。
        /// </summary>
        /// <param name="index">移動先の要素番号</param>
        /// <param name="nodeDesc">移動するノード</param>
        public void MoveNode(int index, NodeDesc nodeDesc)
        {
            if (index < 0 || this.NodeDescs.Count <= index)
            {
                return;
            }

            this.RemoveNode(nodeDesc);
            this.InsertNode(index, nodeDesc);
        }

        /// <summary>
        /// 一つ上のノードと指定したノードを置換します。
        /// </summary>
        /// <param name="nodeDesc">置換するノード</param>
        public void MoveUp(NodeDesc nodeDesc)
        {
            var index = this.NodeDescs.IndexOf(nodeDesc) - 1;

            this.MoveNode(index, nodeDesc);
        }

        /// <summary>
        /// 一つ下のノードと指定したノードを置換します。
        /// </summary>
        /// <param name="nodeDesc">置換するノード</param>
        public void MoveDown(NodeDesc nodeDesc)
        {
            var index = this.NodeDescs.IndexOf(nodeDesc) + 1;

            this.MoveNode(index, nodeDesc);
        }
    }

    /// <summary>
    /// MainWindow.xaml の相互作用ロジック
    /// </summary>
    public partial class MainWindow : Window
    {
        private DataModel dataModel = new DataModel();

        private void SearchWorkingDirectoryByDialog()
        {
            // ディレクトリ選択ダイアログを利用して作業ディレクトリを探す
            // TODO: FileBrowserDialog がいけてないので何とかする

            FolderBrowserDialog dialog = new FolderBrowserDialog();
            dialog.Description = "作業ディレクトリを選択してください。";
            dialog.ShowNewFolderButton = true;
            dialog.SelectedPath = Environment.CurrentDirectory;

            if (dialog.ShowDialog() == System.Windows.Forms.DialogResult.OK)
            {
                SetWorkingDirectory(dialog.SelectedPath);
            }
        }

        private void SetWorkingDirectory(string path)
        {
            this.dataModel.WorkingDirPath = path;
            ReloadNodeDefinition();
        }

        private void ReloadNodeDefinition()
        {
            // 作業ディレクトリ内の .glsl をロードしてノード定義の一覧を更新
            // TODO: 作業中のノードはあるけどリソース側を更新したい、とかのケースはあるはず

            if (!Directory.Exists(this.dataModel.WorkingDirPath))
            {
                return;
            }

            // NodeFiles を更新
            {
                this.dataModel.NodeFiles.Clear();
                // ディレクトリ内の .glsl を列挙
                // MEMO: .glsl でフィルタしないべき？
                // TODO: 階層構造に対応する？
                DirectoryInfo dir = new DirectoryInfo(this.dataModel.WorkingDirPath);
                IEnumerable<FileInfo> files = dir.GetFiles().Where(f => f.Extension == ".glsl");
                foreach (var f in files)
                {
                    this.dataModel.NodeFiles.Add(new NodeFile(f.Name));
                }
            }

            // 各 .glsl をパースして NodeDesc を更新
            {
                this.dataModel.NoConvertCode.Clear();
                for (int i = 0; i < this.dataModel.NodeFiles.Count; ++i)
                {
                    var filePath = this.dataModel.NodeFiles[i].FilePath;
                    string xmlText = null;
                    using (StreamReader sr = new StreamReader(this.dataModel.WorkingDirPath + "\\" + filePath))
                    {
                        var text = sr.ReadToEnd();
                        xmlText = Utility.NodeInfoHelper.PreprocessGlslToNodeInfoXml(text);
                    }
                    string noConvertCode = "";
                    var list = Utility.NodeInfoHelper.LoadNodeDesc(xmlText, ref noConvertCode);

                    this.dataModel.NoConvertCode[filePath] = noConvertCode;

                    // TODO: AddRange 的に処理した方が PropertyChanged も1度しか呼ばれないので高速？
                    foreach (var d in list)
                    {
                        // ここのタイミングで 保存先ファイル名のインデックスを付与
                        d.OutputGlslFilePath = filePath;
                        this.dataModel.GetNodeFile(filePath).AddNode(d);
                    }
                }
            }
        }

        private void SaveNodeDefinition(string extraExtension = "")
        {
            // ノード定義を保存
            // MEMO: extraExtension は別ファイルとして保存するときの追加の拡張子。

            // 不要なファイルが生じたかどうかのチェック用。
            Dictionary<string, bool> isFileNeeded = new Dictionary<string, bool>();
            foreach (var f in this.dataModel.NodeFiles)
            {
                // 否定文で初期化
                isFileNeeded[f.FilePath] = false;
            }

            // 出力ファイル別にリストに分類
            Dictionary<string, List<NodeDesc>> dict = new Dictionary<string, List<NodeDesc>>();
            foreach (var file in this.dataModel.NodeFiles)
            {
                foreach (var desc in file.NodeDescs)
                {
                    if (!dict.ContainsKey(desc.OutputGlslFilePath))
                    {
                        dict[desc.OutputGlslFilePath] = new List<NodeDesc>();
                    }
                    dict[desc.OutputGlslFilePath].Add(desc);
                    isFileNeeded[desc.OutputGlslFilePath] = true; // このファイルは必要
                }
            }

            // 出力ファイルごとにファイル保存
            foreach (var list in dict)
            {
                var path = this.dataModel.WorkingDirPath + "\\" + list.Key + extraExtension;
                using (StreamWriter sw = new StreamWriter(path, false, new UTF8Encoding(true)))
                {
                    // NoConvert タグを先頭に挿入（※存在する場合のみ）
                    if (this.dataModel.NoConvertCode.ContainsKey(list.Key) &&
                        this.dataModel.NoConvertCode[list.Key].Length > 0)
                    {
                        string noConvert = "/// <NoConvert>\n" +
                                           this.dataModel.NoConvertCode[list.Key] + "\n" +
                                           "/// </NoConvert>\n";
                        noConvert = noConvert.Replace("\r\n", "\n").Replace("\n", "\r\n");
                        sw.WriteLine(noConvert);
                    }

                    // そのあとにノード定義を並べる
                    foreach (var node in list.Value)
                    {
                        sw.WriteLine(Utility.NodeInfoHelper.SaveNodeDescToString(node));
                    }
                }
            }

            // 不要となったファイル（出力先が変わった結果何も保存されなくなったファイル）には、
            // 直前のノード定義が残っていて、 CombinerEditor 側でロードしたときに二重定義につながるので、
            // ここで空ファイルにして重複をなくしておく
            foreach (var cond in isFileNeeded)
            {
                if (cond.Value == false)
                {
                    // マーキングされなかったので不要
                    // MEMO: なくてもシステム的には困らないが、空のファイルを吐き出しておく。
                    var trashFilePath = this.dataModel.WorkingDirPath + "\\" + cond.Key + extraExtension;
                    using (StreamWriter sw = new StreamWriter(trashFilePath, false, new UTF8Encoding(true)))
                    {
                        sw.Write("");
                    }
                }
            }
        }

        /// <summary>
        /// 新しいノードを作成します。
        /// </summary>
        /// <returns>新しいノード</returns>
        private NodeDesc CreateNewNode(NodeFile nodeFile)
        {
            if (nodeFile == null)
            {
                return null;
            }

            var newNode = new NodeDesc();
            newNode.Block.Guid = Guid.NewGuid();

            if (nodeFile != null)
            {
                newNode.OutputGlslFilePath = nodeFile.FilePath;
            }

            return newNode;
        }

        /// <summary>
        /// リストにノードを追加します。
        /// </summary>
        /// <param name="node">追加するノード</param>
        private void AddNewNodeToList(NodeDesc node)
        {
            if (node == null)
                return;

            // ノードを作成し選択中のファイルの末尾に追加。
            var nodeFile = this.dataModel.SelectedFile;
            nodeFile.AddNode(node);
            node.IsSelected = true;
        }

        /// <summary>
        /// 選択中のノードの下にノードを追加します。
        /// </summary>
        /// <param name="node">追加するノード</param>
        private void InsertNewNodeToList(NodeDesc node)
        {
            if (node == null)
                return;

            //　ノードを選択中のノード下に追加
            var nodeFile = this.dataModel.SelectedFile;
            if (nodeFile == null)
            {
                return;
            }

            var selectNode = this.dataModel.SelectedNode;
            if (selectNode == null)
            {
                AddNewNodeToList(node);
            }
            else
            {
                var index = nodeFile.NodeDescs.IndexOf(selectNode) + 1;
                nodeFile.InsertNode(index, node);
                node.IsSelected = true;
            }
        }

        public MainWindow()
        {
            InitializeComponent();
            this.DataContext = this.dataModel;

            // 起動時引数の対応
            if (App.CommandLineArgs != null)
            {
                foreach (var arg in App.CommandLineArgs)
                {
                    if (arg.StartsWith("--working_dir"))
                    {
                        // MEMO: まずは以下の形式のみ対応
                        // --working_dir=<PATH>
                        var command = arg.Split('=');
                        var path = command[1];
                        if (Directory.Exists(path))
                        {
                            SetWorkingDirectory(path);
                        }
                    }
                }
            }
        }

        private void Button_OpenWorkingDir_Click(object sender, RoutedEventArgs e)
        {
            SearchWorkingDirectoryByDialog();
        }

        private void MenuItem_OpenWorkingDir_Click(object sender, RoutedEventArgs e)
        {
            SearchWorkingDirectoryByDialog();
        }

        private void TextBox_WorkingDirPath_KeyDown(object sender, System.Windows.Input.KeyEventArgs e)
        {
            if (e.Key == Key.Enter)
            {
                var path = this.TextBox_WorkingDirPath.Text;
                if (path != this.dataModel.WorkingDirPath)
                {
                    if (Directory.Exists(path))
                    {
                        SetWorkingDirectory(path);
                    }
                }
            }
        }

        private void Button_ReloadWorkingDir_Click(object sender, RoutedEventArgs e)
        {
            ReloadNodeDefinition();
        }

        private void MenuItem_SaveAll_Click(object sender, RoutedEventArgs e)
        {
            SaveNodeDefinition();
        }

        private void MenuItem_SaveAll_Click_1(object sender, RoutedEventArgs e)
        {
            SaveNodeDefinition();
        }

        private void Button_AddOutputSocket_Click(object sender, RoutedEventArgs e)
        {
            var node = this.dataModel.SelectedNode;
            if (node != null)
            {
                node.AddOutputDesc(new OutputDesc(node));
            }
        }

        private void Button_RemoveOutputSocket_Click(object sender, RoutedEventArgs e)
        {
            var node = this.dataModel.SelectedNode;
            var index = this.DataGrid_OutputDesc.SelectedIndex;

            if (node != null && index >= 0)
            {
                node.RemoveAtOutputDesc(index);
            }
        }

        private void Button_AddInputSocket_Click(object sender, RoutedEventArgs e)
        {
            var node = this.dataModel.SelectedNode;
            if (node != null)
            {
                node.AddInputDesc(new InputDesc(node));
            }
        }

        private void Button_RemoveInputSocket_Click(object sender, RoutedEventArgs e)
        {
            var node = this.dataModel.SelectedNode;
            var index = this.DataGrid_InputDesc.SelectedIndex;
            if (node != null && index >= 0)
            {
                node.RemoveAtInputDesc(index);
            }
        }
        private void Button_AddNode_Click(object sender, RoutedEventArgs e)
        {
            if (this.dataModel.SelectedFile == null)
            {
                return;
            }

            var newNode = CreateNewNode(this.dataModel.SelectedFile);
            if (newNode != null)
            {
                this.AddNewNodeToList(newNode);
            }
        }

        private void Button_InsertNode_Click(object sender, RoutedEventArgs e)
        {
            if (this.dataModel.SelectedFile == null)
            {
                return;
            }

            var newNode = CreateNewNode(this.dataModel.SelectedFile);
            if (newNode != null)
            {
                // 選択行の下に新規ノードを追加
                this.InsertNewNodeToList(newNode);
            }
        }

        private void Button_RemoveNode_Click(object sender, RoutedEventArgs e)
        {
            var nodeDesc = this.dataModel.SelectedNode;

            if (nodeDesc != null)
            {
                var nodeFile = this.dataModel.SelectedFile;
                var index = nodeFile.NodeDescs.IndexOf(nodeDesc);
                nodeFile.RemoveNode(nodeDesc);

                // ノードが全てなくなった場合は選択しない
                if (nodeFile.NodeDescs.Count == 0)
                {
                    return;
                }

                // 削除したノードと同じ添え字を選択
                if (index < nodeFile.NodeDescs.Count)
                {
                    nodeFile.NodeDescs[index].IsSelected = true;
                }
                else
                {
                    // 最後のノードだったら、一番下のノードを選択
                    if (index == nodeFile.NodeDescs.Count)
                    {
                        nodeFile.NodeDescs[index - 1].IsSelected = true;
                    }
                }
            }
        }

        private void Button_MoveUpNode_Click(object sender, RoutedEventArgs e)
        {
            //// 選択中のノードを上へ
            var nodeDesc = this.dataModel.SelectedNode;
            if (nodeDesc == null)
            {
                return;
            }

            var nodeFile = this.dataModel.SelectedFile;
            nodeFile.MoveUp(nodeDesc);
            nodeDesc.IsSelected = true;
        }

        private void Button_MoveDownNode_Click(object sender, RoutedEventArgs e)
        {
            //// 選択中のノードを下へ
            var nodeDesc = this.dataModel.SelectedNode;
            if (nodeDesc == null)
            {
                return;
            }

            var nodeFile = this.dataModel.SelectedFile;
            nodeFile.MoveDown(nodeDesc);
            nodeDesc.IsSelected = true;
        }

        private void Button_MoveUpInputDesc_Click(object sender, RoutedEventArgs e)
        {
            // 選択中の入力ノードを上へ
            var node = this.dataModel.SelectedNode;
            if (node != null)
            {
                var i = this.DataGrid_InputDesc.SelectedIndex;
                if (i > 0)
                {
                    // index == 0 の時は一番上なので何もしない
                    var input = node.InputDescs[i];
                    node.RemoveAtInputDesc(i);
                    node.InsertInputDesc(i - 1, input);
                    this.DataGrid_InputDesc.SelectedIndex = i - 1;
                }
            }
        }

        private void Button_MoveDownInputDesc_Click(object sender, RoutedEventArgs e)
        {
            // 選択中の入力ノードを下へ
            var node = this.dataModel.SelectedNode;
            if (node != null)
            {
                var i = this.DataGrid_InputDesc.SelectedIndex;
                if (0 <= i && i < node.InputDescs.Count - 1)
                {
                    // index が最後尾の時は一番下なので何もしない
                    var input = node.InputDescs[i];
                    node.RemoveAtInputDesc(i);
                    node.InsertInputDesc(i + 1, input);
                    this.DataGrid_InputDesc.SelectedIndex = i + 1;
                }
            }
        }

        private void Button_MoveUpOutputDesc_Click(object sender, RoutedEventArgs e)
        {
            // 選択中の出力ノードを上へ
            var node = this.dataModel.SelectedNode;
            if (node != null)
            {
                var i = this.DataGrid_OutputDesc.SelectedIndex;
                if (i > 0)
                {
                    // index == 0 の時は一番上なので何もしない
                    var output = node.OutputDescs[i];
                    node.RemoveAtOutputDesc(i);
                    node.InsertOutputDesc(i - 1, output);
                    this.DataGrid_OutputDesc.SelectedIndex = i - 1;
                }
            }
        }

        private void Button_MoveDownOutputDesc_Click(object sender, RoutedEventArgs e)
        {
            // 選択中の出力ノードを下へ
            var node = this.dataModel.SelectedNode;
            if (node != null)
            {
                var i = this.DataGrid_OutputDesc.SelectedIndex;
                if (0 <= i && i < node.OutputDescs.Count - 1)
                {
                    // index が最後尾の時は一番下なので何もしない
                    var output = node.OutputDescs[i];
                    node.RemoveAtOutputDesc(i);
                    node.InsertOutputDesc(i + 1, output);
                    this.DataGrid_OutputDesc.SelectedIndex = i + 1;
                }
            }
        }

        private void MenuItem_SaveAllAsAnotherFiles_Click(object sender, RoutedEventArgs e)
        {
            // デバッグ機能: すべてを別ファイルとして保存する
            // 入出力が正しく行われているかなどの確認用途。
            string extraExtension = ".tmp";
            SaveNodeDefinition(extraExtension);
        }

        private void MenuItem_Exit_Click(object sender, RoutedEventArgs e)
        {
            this.Close();
        }

        private void TextBox_WorkingDirPath_Drop(object sender, System.Windows.DragEventArgs e)
        {
            string[] data = e.Data.GetData(System.Windows.DataFormats.FileDrop) as string[];
            if (data.Length > 0)
            {
                var path = data[0];// 先頭だけ使う

                if (path != this.dataModel.WorkingDirPath)
                {
                    if (Directory.Exists(path))
                    {
                        SetWorkingDirectory(path);
                    }
                    else
                    {
                        this.TextBox_WorkingDirPath.Text = path;
                    }
                }
            }
        }

        private void TextBox_WorkingDirPath_PreviewDragOver(object sender, System.Windows.DragEventArgs e)
        {
            if (e.Data.GetDataPresent(System.Windows.DataFormats.FileDrop, true))
            {
                e.Effects = System.Windows.DragDropEffects.Copy;
            }
            else
            {
                e.Effects = System.Windows.DragDropEffects.None;
            }
            e.Handled = true;
        }

        private void Button_AddNewFile_Click(object sender, RoutedEventArgs e)
        {
            if (this.dataModel.WorkingDirPath == null)
            {
                System.Windows.MessageBox.Show("作業ディレクトリが設定されていないので、ファイルの追加は行えません。", "作業ディレクトリ未設定", MessageBoxButton.OK, MessageBoxImage.Error);
                return;
            }

            // 新規ファイル追加ダイアログを呼び出して、その結果を追加。
            Dialog.AddNewFileDialog dialog = new Dialog.AddNewFileDialog();
            if (dialog.ShowDialog() == true)
            {
                var path = dialog.NewFilePath;
                var newFilePath = this.dataModel.WorkingDirPath + "\\" + path;

                // ファイルパスのチェック
                var invalidPathChars = System.IO.Path.GetInvalidPathChars();
                var invalidFileNameChars = System.IO.Path.GetInvalidFileNameChars();
                if (invalidFileNameChars.Any(c => path.Contains(c)) ||
                     invalidPathChars.Any(c => newFilePath.Contains(c)))
                {
                    System.Windows.MessageBox.Show("ファイル名またはファイルパスが不正です。", "不正なファイルパス", MessageBoxButton.OK, MessageBoxImage.Error);
                    return;
                }
                else if (System.IO.File.Exists(newFilePath))
                {
                    System.Windows.MessageBox.Show("既に同名のファイルが存在します。", "ファイル名の重複", MessageBoxButton.OK, MessageBoxImage.Error);
                    return;
                }

                this.dataModel.NodeFiles.Add(new NodeFile(path));

                // MEMO: なくてもシステム的には困らないが、空のファイルを吐き出しておく。
                using (StreamWriter sw = new StreamWriter(newFilePath, false, new UTF8Encoding(true)))
                {
                    sw.Write("");
                }
            }
        }

        /// <summary>
        /// ノードのペースト情報です。
        /// </summary>
        public static class PasteInfo
        {
            /// <summary>
            /// ペーストの種類
            /// </summary>
            public enum CommandType
            {
                Cut,
                Copy,
            }

            /// <summary>
            /// ペーストの種類を取得または設定します。
            /// </summary>
            public static CommandType Type { get; set; }

            /// <summary>
            /// 選択したノードのビューモデルを取得または設定します。
            /// </summary>
            public static NodeDesc NodeDesc { get; set; }
        }

        /// <summary>
        /// キー入力の切り取りイベントです。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void Cut_KeyDown(object sender, ExecutedRoutedEventArgs e)
        {
            PasteInfo.Type = PasteInfo.CommandType.Cut;
            if (this.dataModel.SelectedNode != null)
            {
                PasteInfo.NodeDesc = this.dataModel.SelectedNode;
                var nodeFile = this.dataModel.SelectedFile;
                nodeFile.RemoveNode(this.dataModel.SelectedNode);
                nodeFile.IsSelected = true;
            }
        }

        /// <summary>
        /// キー入力のコピーイベントです。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void Copy_KeyDown(object sender, ExecutedRoutedEventArgs e)
        {
            PasteInfo.Type = PasteInfo.CommandType.Copy;
            if (this.dataModel.SelectedNode != null)
            {
                PasteInfo.NodeDesc = this.dataModel.SelectedNode;
            }
        }

        /// <summary>
        /// カット後のペースト処理を行います。
        /// </summary>
        private void CutPaste()
        {
            var removeFile = this.dataModel.GetNodeFile(PasteInfo.NodeDesc.OutputGlslFilePath);
            var insertFile = this.dataModel.SelectedFile;

            if (insertFile == null)
            {
                return;
            }

            var nodeDesc = this.dataModel.SelectedNode;
            if (nodeDesc == PasteInfo.NodeDesc)
            {
                return;
            }

            if (removeFile != insertFile)
            {
                // 違うファイルへ追加する場合は、出力ファイルを変える。
                PasteInfo.NodeDesc.OutputGlslFilePath = insertFile.FilePath;
            }

            if (nodeDesc == null)
            {
                insertFile.AddNode(PasteInfo.NodeDesc);
                PasteInfo.NodeDesc.IsSelected = true;
            }
            else
            {
                var index = insertFile.NodeDescs.IndexOf(nodeDesc);
                insertFile.InsertNode(index, PasteInfo.NodeDesc);
                PasteInfo.NodeDesc.IsSelected = true;
            }

            PasteInfo.NodeDesc = null;
        }

        /// <summary>
        /// コピー後のペースト処理を行います。
        /// </summary>
        private void CopyPaste()
        {
            var selectFile = this.dataModel.SelectedFile;
            if (selectFile == null)
            {
                return;
            }

            // 安く実装するため、コピー元ノードを1回ファイル出力用のxmlに形式に変換したものを
            // 新しいノードに適用する。
            var nodeDescStr = Utility.NodeInfoHelper.SaveNodeDescToString(PasteInfo.NodeDesc);
            var noConvertCode = string.Empty;
            nodeDescStr = Utility.NodeInfoHelper.PreprocessGlslToNodeInfoXml(nodeDescStr);
            var node = Utility.NodeInfoHelper.LoadNodeDesc(nodeDescStr, ref noConvertCode).FirstOrDefault();


            node.Block.Guid = Guid.NewGuid();
            int indexer = 1;
            var copyName = "_Copy" + indexer.ToString();
            var newName = node.FunctionName + copyName;

            while (true)
            {
                if (!ExistFunctionName(newName))
                {
                    break;
                }
                indexer++;
                copyName = "_Copy" + indexer.ToString();
                newName = node.FunctionName + copyName;
            }

            node.FunctionName = node.FunctionName + copyName;
            node.Block.DisplayName = node.Block.DisplayName + copyName;

            node.OutputGlslFilePath = selectFile.FilePath;
            InsertNewNodeToList(node);
        }

        /// <summary>
        /// キー入力のペーストイベントです。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void Paste_KeyDown(object sender, ExecutedRoutedEventArgs e)
        {
            if (PasteInfo.NodeDesc != null)
            {
                if (PasteInfo.Type == PasteInfo.CommandType.Cut)
                {
                    CutPaste();
                }
                else if (PasteInfo.Type == PasteInfo.CommandType.Copy)
                {
                    CopyPaste();
                }
            }
        }

        private void Save_KeyDown(object sender, ExecutedRoutedEventArgs e)
        {
            SaveNodeDefinition();
        }

        private bool ExistFunctionName(string name)
        {
            foreach (var file in this.dataModel.NodeFiles)
            {
                foreach (var node in file.NodeDescs)
                {
                    if (node.FunctionName == name)
                    {
                        return true;
                    }
                }
            }

            return false;
        }

        private void Button_ApplyTemplate_Click(object sender, RoutedEventArgs e)
        {
            // テンプレートを適用
            var node = this.dataModel.SelectedNode;
            if (node != null)
            {
                // MEMO: NodeDesc のディープコピーがあると理想的だが、
                //       今無い上に用意するのも手間なので、プレビュー用のコピーをここで安く作る
                var tempNode = new NodeDesc();

                // 必要な部分以外はコピーしない
                // 要するにノードのブロックに関する情報のみコピー
                tempNode.Block.DisplayName = node.Block.DisplayName;
                foreach (var input in node.InputDescs)
                {
                    var d = new InputDesc(tempNode);
                    d.ArgTarget = input.ArgTarget;
                    d.DisplayName = input.DisplayName;
                    d.Semantics = input.Semantics;
                    d.Type = input.Type;
                    tempNode.InputDescs.Add(d);
                }

                foreach (var output in node.OutputDescs)
                {
                    var d = new OutputDesc(tempNode);
                    d.DisplayName = output.DisplayName;
                    d.ReturnTarget = output.ReturnTarget;
                    d.Semantics = output.Semantics;
                    d.Type = output.Type;
                    tempNode.OutputDescs.Add(d);
                }

                // 新規ファイル追加ダイアログを呼び出して、その結果を追加。
                Dialog.ApplyTemplateDialog dialog = new Dialog.ApplyTemplateDialog(tempNode);
                if (dialog.ShowDialog() == true)
                {
                    // ダイアログの結果からテンプレートを上書き
                    // 既存の入出力定義の名前は引き継げる分は引き継ぐ
                    dialog.ApplyTemplate(node);
                    this.NodeViewer_CurrentNode.Invalidate();
                }
            }
        }

        private void Button_InvalidatePreview_Click(object sender, RoutedEventArgs e)
        {
            if (this.dataModel.SelectedNode != null)
            {
                // ノードのプレビューの更新ボタン
                // MEMO: 本来はデータバインドと変更通知を駆使して、
                //       自動でリロードが掛かるのがベストだが、
                //       今は DataModel の変更を拾い上げて UI 再構築を呼び出すための仕組みが
                //       作ろうとすると煩雑になるので、暫定で更新ボタンを置いておく。
                this.NodeViewer_CurrentNode.Invalidate();
            }
        }
    }

    /// <summary>
    /// 設定できる返却値か検証するクラスです。
    /// </summary>
    public class CanSetReturnValueValidationRule : ValidationRule
    {
        /// <summary>
        /// 値の検証を行います。
        /// </summary>
        /// <param name="value">検証する値</param>
        /// <param name="cultureInfo">カルチャ情報</param>
        /// <returns>検証結果</returns>
        public override ValidationResult Validate(object value, CultureInfo cultureInfo)
        {
            // TODO:SIGLO-68790の対応後に削除予定
            var binding = (System.Windows.Data.BindingExpression)value;

            if (binding.DataItem is NodeDesc)
            {
                var nodeDes = binding.DataItem as NodeDesc;

                if (nodeDes.ReturnType == ValueType.Sampler2D)
                {
                    return new ValidationResult(false, "EffectCombinerEditorでは、sampler2Dは評価中の機能です。");
                }
            }

            return ValidationResult.ValidResult;
        }
    }
}

