﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace Workflow.Core
{
    public interface IUniquelyIdentifiable
    {
        Guid Identifier { get; }
    }

    public class IdentifiableEventArgs : EventArgs
    {
        public IUniquelyIdentifiable OldIdentifiable { get; private set; }
        public IUniquelyIdentifiable NewIdentifiable { get; private set; }

        public IdentifiableEventArgs(IUniquelyIdentifiable oldIdentifiable, IUniquelyIdentifiable newIdentifiable)
        {
            OldIdentifiable = oldIdentifiable;
            NewIdentifiable = newIdentifiable;
        }
    }

    public class UniquelyIdentifiableBase : IUniquelyIdentifiable
    {
        public UniquelyIdentifiableBase()
        {
            Identifier = Guid.NewGuid();
        }

        public virtual Guid Identifier { get; private set; }

        public override string ToString()
        {
            return Identifier.ToString();
        }
    }

    public class UniquelyIdentifiableEqualityComparer : IEqualityComparer<IUniquelyIdentifiable>
    {
        public static readonly UniquelyIdentifiableEqualityComparer Default = new UniquelyIdentifiableEqualityComparer();

        public bool Equals(IUniquelyIdentifiable x, IUniquelyIdentifiable y)
        {
            if (x == null || y == null)
                return false;
            return x.Identifier == y.Identifier;
        }

        public int GetHashCode(IUniquelyIdentifiable obj)
        {
            if (obj == null)
                return 0;
            return obj.Identifier.GetHashCode();
        }
    }

    public static class UniquelyIdentifiableExtensions
    {
        public static bool Equals(this IUniquelyIdentifiable a, IUniquelyIdentifiable b)
        {
            if (a == null)
                throw new ArgumentNullException("a");
            return b != null && (ReferenceEquals(a, b) || a.Identifier == b.Identifier);
        }
    }
}
