﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using Workflow.Core;

namespace UnitTests
{
    public class ProducerWorkflowItem : WorkflowItem<object>
    {
        public ProducerWorkflowItem(string name)
        {
            Name = name;
            AddOutputPlug(new OutputPlug<object>());
            AddOutputPlug(new OutputPlug<object>());
        }

        public override string ToString()
        {
            return Name;
        }
    }

    public class ConsumerWorkflowItem : WorkflowItem<object>
    {
        public ConsumerWorkflowItem(string name)
        {
            Name = name;
            AddInputPlug(new InputPlug<object>());
            AddInputPlug(new InputPlug<object>());
        }

        public override string ToString()
        {
            return Name;
        }
    }

    public class DummyWorkflowItem : WorkflowItem<object>
    {
        public DummyWorkflowItem(string name)
        {
            Name = name;
            AddInputPlug(new InputPlug<object>());
            AddInputPlug(new InputPlug<object>());
            AddOutputPlug(new OutputPlug<object>());
            AddOutputPlug(new OutputPlug<object>());
        }

        public override string ToString()
        {
            return Name;
        }
    }

    public class ValueWorkflowItem : ValueWorkflowItem<double>
    {
        public ValueWorkflowItem(double initialValue)
            : base(initialValue)
        {
        }
    }

    public class ValueWorkflowItem<T> : WorkflowItem<T>
    {
        public void SetValue(T value)
        {
            SetOutputValue(0, value);
            Name = string.Format("Value ({0})", value);
        }

        public ValueWorkflowItem(T initialValue)
        {
            AddOutputPlug(new OutputPlug<T>());
            SetValue(initialValue);
        }
    }

    public class AddWorkflowItem : WorkflowItem<double>
    {
        public AddWorkflowItem()
        {
            Name = "Add";
            AddInputPlug(new InputPlug<double> { Name = "A" });
            AddInputPlug(new InputPlug<double> { Name = "B" });
            AddOutputPlug(new OutputPlug<double> { Name = "Result" });
        }

        protected override void OnInputsInvalidated()
        {
            base.OnInputsInvalidated();
            if (InputPlugs.Length == 2 && OutputPlugs.Length == 1)
                SetOutputValue(0, InputPlugs[0].Value + InputPlugs[1].Value);
        }
    }

    public class SubWorkflowItem : WorkflowItem<double>
    {
        public SubWorkflowItem()
        {
            Name = "Sub";
            AddInputPlug(new InputPlug<double> { Name = "A" });
            AddInputPlug(new InputPlug<double> { Name = "B" });
            AddOutputPlug(new OutputPlug<double> { Name = "Result" });
        }

        protected override void OnInputsInvalidated()
        {
            base.OnInputsInvalidated();
            if (InputPlugs.Length == 2 && OutputPlugs.Length == 1)
                SetOutputValue(0, InputPlugs[0].Value - InputPlugs[1].Value);
        }
    }

    public class MulWorkflowItem : WorkflowItem<double>
    {
        public MulWorkflowItem()
        {
            Name = "Mul";
            AddInputPlug(new InputPlug<double> { Name = "A" });
            AddInputPlug(new InputPlug<double> { Name = "B" });
            AddOutputPlug(new OutputPlug<double> { Name = "Result" });
        }

        protected override void OnInputsInvalidated()
        {
            base.OnInputsInvalidated();
            if (InputPlugs.Length == 2 && OutputPlugs.Length == 1)
                SetOutputValue(0, InputPlugs[0].Value * InputPlugs[1].Value);
        }
    }

    public class AddMulWorkflowItem : WorkflowItem<double>
    {
        public AddMulWorkflowItem()
        {
            Name = "Add & Mul";
            AddInputPlug(new InputPlug<double> { Name = "A" });
            AddInputPlug(new InputPlug<double> { Name = "B" });
            AddOutputPlug(new OutputPlug<double> { Name = "Add Result" });
            AddOutputPlug(new OutputPlug<double> { Name = "Mul Result" });
        }

        protected override void OnInputsInvalidated()
        {
            base.OnInputsInvalidated();

            if (InputPlugs.Length < 2)
                return;

            if (OutputPlugs.Length > 0)
            {
                SetOutputValue(0, InputPlugs[0].Value + InputPlugs[1].Value);
                if (OutputPlugs.Length > 1)
                    SetOutputValue(1, InputPlugs[0].Value * InputPlugs[1].Value);
            }
        }
    }

    public class TransferWorkflowItem<T> : WorkflowItem<T>
    {
        private readonly int ioCount;

        public TransferWorkflowItem(int inputOutputCount)
        {
            ioCount = Math.Max(1, Math.Min(inputOutputCount, 16));

            for (var i = 0; i < ioCount; i++)
            {
                AddInputPlug(new InputPlug<T>());
                AddOutputPlug(new OutputPlug<T>());
            }
        }

        protected override void OnInputsInvalidated()
        {
            base.OnInputsInvalidated();

            if (InputPlugs.Length < ioCount || OutputPlugs.Length < ioCount)
                return;

            for (var i = 0; i < ioCount; i++)
                SetOutputValue(i, InputPlugs[i].Value);
        }
    }
}
