﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Linq;
using System.Text;

namespace Renderer2D.GDI
{
    static class Conversions
    {
        public static Color ToColor(this Core.IColor c)
        {
            return Color.FromArgb(c.A, c.R, c.G, c.B);
        }

        #region Pens and brushes

        /*
        public static System.Drawing.Pen ToPen(this Core.IBrush b)
        {
            return new System.Drawing.Pen(b.ToBrush());
        }
        */

        public static DashCap ToDashCap(this Core.CapStyle style)
        {
            switch (style)
            {
                case Core.CapStyle.Round:    return DashCap.Round;
                case Core.CapStyle.Triangle: return DashCap.Triangle;

                case Core.CapStyle.Flat:
                case Core.CapStyle.Square:
                default:                     return DashCap.Flat;
            }
        }

        public static LineCap ToLineCap(this Core.CapStyle style)
        {
            switch (style)
            {
                case Core.CapStyle.Square:   return LineCap.Square;
                case Core.CapStyle.Round:    return LineCap.Round;
                case Core.CapStyle.Triangle: return LineCap.Triangle;
                case Core.CapStyle.Flat:
                default:                     return LineCap.Flat;
            }
        }

        public static DashStyle ToDashStyle(this Core.DashStyle style)
        {
            switch (style)
            {
                case Core.DashStyle.Dash:       return DashStyle.Dash;
                case Core.DashStyle.DashDot:    return DashStyle.DashDot;
                case Core.DashStyle.DashDotDot: return DashStyle.DashDotDot;
                case Core.DashStyle.Dot:        return DashStyle.Dot;

                case Core.DashStyle.Solid:
                case Core.DashStyle.Custom: // Custom is not implemented because I'm too lazy
                default:                        return DashStyle.Solid;
            }
        }

        public static LineJoin ToLineJoin(this Core.LineJoin style)
        {
            switch (style)
            {
                case Core.LineJoin.Bevel:        return LineJoin.Bevel;
                case Core.LineJoin.Miter:        return LineJoin.Miter;
                case Core.LineJoin.MiterOrBevel: return LineJoin.MiterClipped;

                case Core.LineJoin.Round:
                default:                         return LineJoin.Round;
            }
        }

        public static Pen ToPen(this Core.IBrush b, double width, Core.IStrokeStyle style)
        {
            Pen pen = new Pen(b.ToBrush(), (float)width);
            pen.DashCap = style.DashCap.ToDashCap();
            pen.DashStyle = style.DashStyle.ToDashStyle();
            pen.StartCap = style.StartCap.ToLineCap();
            pen.EndCap = style.EndCap.ToLineCap();
            pen.LineJoin = style.LineJoin.ToLineJoin();
            pen.DashOffset = (float)style.DashOffset;
            pen.MiterLimit = (float)style.MiterLimit;

            return pen;
        }

        public static Brush ToBrush(this Core.IBrush b)
        {
            if ((b is GDIBrushContainer) == false)
                throw new ArgumentException(string.Format("Invalid brush (Unknown type '{0}')", b.GetType().FullName));

            return ((GDIBrushContainer)b).NativeBrush;
        }

        #endregion

        #region Points and rectangles

        public static Point ToPoint(this Core.IPoint p)
        {
            return new Point((int)p.X, (int)p.Y);
        }

        public static PointF ToPointF(this Core.IPoint p)
        {
            return new PointF((float)p.X, (float)p.Y);
        }


        public static Rectangle ToRectangle(this Core.IRectangle r)
        {
            return new Rectangle((int)r.X, (int)r.Y, (int)r.Width, (int)r.Height);
        }

        public static RectangleF ToRectangleF(this Core.IRectangle r)
        {
            return new RectangleF((float)r.X, (float)r.Y, (float)r.Width, (float)r.Height);
        }

        public static RectangleF ToRectangleF(this Core.IEllipse e)
        {
            return new RectangleF((float)(e.Center.X - e.RadiusX),
                                  (float)(e.Center.Y - e.RadiusY),
                                  (float)(2.0 * e.RadiusX),
                                  (float)(2.0 * e.RadiusY));
        }

        #endregion

        #region Font

        public static FontStyle ToFontStyle(this Core.ITextFormat textFormat)
        {
            // Core.FontStretch stretch = textFormat.Font.Stretch;

            FontStyle convertedStyle = FontStyle.Regular;
            switch (textFormat.Font.Style)
            {
                case Core.FontStyle.Normal:
                    convertedStyle = FontStyle.Regular;
                    break;
                case Core.FontStyle.Oblique:
                case Core.FontStyle.Italic:
                    convertedStyle = FontStyle.Italic;
                    break;
            }

            Core.FontWeight weight = textFormat.Font.Weight;
            FontStyle convertedWeight = FontStyle.Regular;
            switch (weight)
            {
                case Core.FontWeight.UltraBlack:
                case Core.FontWeight.ExtraBlack:
                case Core.FontWeight.Black:
                case Core.FontWeight.Heavy:
                case Core.FontWeight.UltraBold:
                case Core.FontWeight.ExtraBold:
                case Core.FontWeight.Bold:
                case Core.FontWeight.DemiBold:
                case Core.FontWeight.SemiBold:
                    convertedWeight = FontStyle.Bold;
                    break;
                case Core.FontWeight.Medium:
                case Core.FontWeight.Normal:
                case Core.FontWeight.Regular:
                case Core.FontWeight.Thin:
                case Core.FontWeight.Light:
                case Core.FontWeight.ExtraLight:
                case Core.FontWeight.UltraLight:
                    convertedWeight = FontStyle.Regular;
                    break;
            }

            return convertedStyle | convertedWeight;
        }


        #endregion

        #region Matrix

        public static Core.IMatrix ToGenericMatrix(this Matrix matrix)
        {
            return new Core.Matrix(
                matrix.Elements[0],
                matrix.Elements[1],
                matrix.Elements[2],
                matrix.Elements[3],
                matrix.Elements[4],
                matrix.Elements[5]);
        }

        public static Matrix ToSpecificMatrix(this Core.IMatrix matrix)
        {
            return new Matrix(
                (float)matrix.M11,
                (float)matrix.M12,
                (float)matrix.M21,
                (float)matrix.M22,
                (float)matrix.M31,
                (float)matrix.M32);
        }

        #endregion // Matrix
    }
}
