﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using SharpDX.Direct2D1;
using SharpDX;

namespace Renderer2D.Direct2D1
{
    public abstract class D2D1BrushContainer : Core.IBrush
    {
        public Brush NativeBrush { get; protected set; }
        protected RenderTarget renderTarget;

        protected D2D1BrushContainer(RenderTarget renderTarget)
        {
            if (renderTarget == null)
                throw new ArgumentNullException("renderTarget");

            this.renderTarget = renderTarget;
        }

        public void Dispose()
        {
            if (NativeBrush != null)
            {
                NativeBrush.Dispose();
                NativeBrush = null;
            }
        }

        public virtual void SetBoundingBox(Core.IRectangle box) { }
    }

    public class D2D1SolidColorBrushContainer : D2D1BrushContainer, Core.ISolidColorBrush
    {
        public D2D1SolidColorBrushContainer(Core.IColor color, RenderTarget renderTarget)
            : base(renderTarget)
        {
            if (color == null)
                throw new ArgumentNullException("color");

            this.color = color;
            NativeBrush = new SolidColorBrush(renderTarget, color.ToD2D1Color());
        }

        private readonly Core.IColor color;
        public Core.IColor Color
        {
            get { return color; }
            set { throw new InvalidOperationException("Frozen"); }
        }
    }

    public class D2D1LinearGradientBrushContainer : D2D1BrushContainer, Core.ILinearGradientBrush
    {
        public D2D1LinearGradientBrushContainer(RenderTarget renderTarget, Core.IPoint startPoint, Core.IPoint endPoint, params Core.IGradientStop[] gradientStops)
            : base(renderTarget)
        {
            if (startPoint == null)
                throw new ArgumentNullException("startPoint");
            if (endPoint == null)
                throw new ArgumentNullException("endPoint");
            if (gradientStops == null)
                throw new ArgumentNullException("gradientStops");

            this.startPoint = startPoint;
            this.endPoint = endPoint;
            this.gradientStops = gradientStops;

            ProduceNativeBrush();
        }

        private readonly Core.IGradientStop[] gradientStops;
        private readonly Core.IPoint startPoint;
        private readonly Core.IPoint endPoint;

        private void ProduceNativeBrush()
        {
            Dispose();

            var gradients = from g in gradientStops
                            select new GradientStop
                            {
                                Position = (float)g.Offset,
                                Color = g.Color.ToD2D1Color()
                            };
            var collection = new GradientStopCollection(renderTarget, gradients.ToArray(), Gamma.StandardRgb, ExtendMode.Clamp);

            new LinearGradientBrush(renderTarget, new LinearGradientBrushProperties
            {
                StartPoint = startPoint.ToPoint2F(),
                EndPoint = endPoint.ToPoint2F(),
            }, collection);

            NativeBrush = new LinearGradientBrush(renderTarget,
                new LinearGradientBrushProperties
                {
                    StartPoint = startPoint.ToPoint2F(),
                    EndPoint = endPoint.ToPoint2F(),
                },
                collection);
        }

        public override void SetBoundingBox(Core.IRectangle box)
        {
            if (NativeBrush == null)
                throw new InvalidOperationException("NativeBrush not yet produced.");

            var left = (float)box.X;
            var top = (float)box.Y;
            var width = (float)box.Width;
            var height = (float)box.Height;

            var brush = (LinearGradientBrush)NativeBrush;
            brush.StartPoint = new Vector2(left + width * (float)startPoint.X, top + height * (float)startPoint.Y);
            brush.EndPoint = new Vector2(left + width * (float)endPoint.X, top + height * (float)endPoint.Y);
        }

        public Core.IGradientStop[] GradientStops
        {
            get { return gradientStops; }
            set { throw new InvalidOperationException("Frozen"); }
        }

        public Core.IPoint StartPoint
        {
            get { return startPoint; }
            set { throw new InvalidOperationException("Frozen"); }
        }

        public Core.IPoint EndPoint
        {
            get { return endPoint; }
            set { throw new InvalidOperationException("Frozen"); }
        }
    }

    public class D2D1RadialGradientBrushContainer : D2D1BrushContainer, Core.IRadialGradientBrush
    {
        public D2D1RadialGradientBrushContainer(RenderTarget renderTarget,
            Core.IPoint center, Core.IPoint gradientOriginOffset,
            double radiusX, double radiusY, params Core.IGradientStop[] gradientStops)
            : base(renderTarget)
        {
            if (center == null)
                throw new ArgumentNullException("center");
            if (gradientOriginOffset == null)
                throw new ArgumentNullException("gradientOriginOffset");
            if (gradientStops == null)
                throw new ArgumentNullException("gradientStops");

            this.center = center;
            this.gradientOriginOffset = gradientOriginOffset;
            this.radiusX = radiusX;
            this.radiusY = radiusY;
            this.gradientStops = gradientStops;

            ProduceNativeBrush();
        }

        private readonly Core.IGradientStop[] gradientStops;
        private readonly Core.IPoint center;
        private readonly Core.IPoint gradientOriginOffset;
        private readonly double radiusX;
        private readonly double radiusY;

        private void ProduceNativeBrush()
        {
            Dispose();

            var gradients = from g in gradientStops
                            select new GradientStop
                            {
                                Position = (float)g.Offset,
                                Color = g.Color.ToD2D1Color(),
                            };
            var collection = new GradientStopCollection(renderTarget, gradients.ToArray(), Gamma.StandardRgb, ExtendMode.Clamp);

            NativeBrush = new RadialGradientBrush(renderTarget,
                new RadialGradientBrushProperties
                {
                    Center = center.ToPoint2F(),
                    GradientOriginOffset = gradientOriginOffset.ToPoint2F(),
                    RadiusX = (float)radiusX,
                    RadiusY = (float)radiusY,
                },
                collection);
        }

        public override void SetBoundingBox(Core.IRectangle box)
        {
            if (NativeBrush == null)
                throw new InvalidOperationException("NativeBrush not yet produced.");

            var left = (float)box.X;
            var top = (float)box.Y;
            var width = (float)box.Width;
            var height = (float)box.Height;

            var brush = (RadialGradientBrush)NativeBrush;
            brush.Center = new Vector2(left + width * (float)center.X, top + height * (float)center.Y);
            brush.GradientOriginOffset = new Vector2((width * (float)gradientOriginOffset.X) / 2.0f, (height * (float)gradientOriginOffset.Y) / 2.0f);
            brush.RadiusX = (width * (float)radiusX) / 2.0f;
            brush.RadiusY = (height * (float)radiusY) / 2.0f;
        }

        public Core.IPoint Center
        {
            get { return center; }
            set { throw new InvalidOperationException("Frozen"); }
        }

        public Core.IPoint GradientOriginOffset
        {
            get { return gradientOriginOffset; }
            set { throw new InvalidOperationException("Frozen"); }
        }

        public double RadiusX
        {
            get { return radiusX; }
            set { throw new InvalidOperationException("Frozen"); }
        }

        public double RadiusY
        {
            get { return radiusY; }
            set { throw new InvalidOperationException("Frozen"); }
        }

        public Core.IGradientStop[] GradientStops
        {
            get { return gradientStops; }
            set { throw new InvalidOperationException("Frozen"); }
        }
    }
}
