﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Text;
using System.Collections.Generic;
using System.Linq;
using Microsoft.VisualStudio.TestTools.UnitTesting;

using EffectDefinitions;

namespace EffectDefinitionsUnitTests
{
    /// <summary>
    /// Unit tests for the Plug class
    /// </summary>
    [TestClass]
    public class PlugTests
    {
        [TestMethod]
        public void PlugConstructor()
        {
            var p = new Plug("myPlug", new ShaderTypeDefinition("int"), "return.u", false, null, null);

            var name = p.Name;
            if (name != "myPlug")
                throw new Exception(string.Format("Name should be \"myPlug\" but returned \"{0}\"", name));

            var type = p.Type.ToString();
            if (type != "int")
                throw new Exception(string.Format("Type should be \"int\" but returned \"{0}\"", type));

            var target = p.Target;
            if (target != "return.u")
                throw new Exception(string.Format("Target should be \"return.u\" but returned \"{0}\"", target));
        }

        [TestMethod]
        public void BaseTargetTest()
        {
            var baseTarget = "foo";
            for (int i = 0; i < 10; ++i)
            {
                var target = baseTarget + string.Concat(Enumerable.Repeat(".bar", i));
                var plug = new Plug("myPlug", new ShaderTypeDefinition("int"), target, false, null, null);
                if (plug.BaseTarget != baseTarget)
                    throw new Exception(string.Format("Plug with target \"{0}\" has base target \"{1}\" instead of \"{2}\"", plug.Target, plug.BaseTarget, baseTarget));
            }
        }

        [TestMethod]
        public void CanPlugVoid()
        {
            try
            {
                var voidPlug = new Plug("name", ShaderTypeDefinition.Void, "target", false, null, null);
            }
            catch (ArgumentException)
            {
                return;
            }

            throw new Exception("Creating a void plug should throw an exception");
        }

        [TestMethod]
        public void CanPlugIntInt()
        {
            var intPlug1 = new Plug("name1", new ShaderTypeDefinition("int"), "target1", false, null, null);
            var intPlug2 = new Plug("name2", new ShaderTypeDefinition("int"), "target2", true, null, null);

            if (!intPlug1.CanPlugTo(intPlug2))
                throw new Exception("CanPlug should return true: int / int");
        }

        [TestMethod]
        public void CanPlugFloatFloat()
        {
            var floatPlug1 = new Plug("name1", new ShaderTypeDefinition("float"), "target1", false, null, null);
            var floatPlug2 = new Plug("name2", new ShaderTypeDefinition("float"), "target2", true, null, null);

            if (!floatPlug1.CanPlugTo(floatPlug2))
                throw new Exception("CanPlug should return true: float / float");
        }

        [TestMethod]
        public void CanPlugIntFloat()
        {
            var intPlug = new Plug("name1", new ShaderTypeDefinition("int"), "target1", false, null, null);
            var floatPlug = new Plug("name2", new ShaderTypeDefinition("float"), "target2", true, null, null);

            if (intPlug.CanPlugTo(floatPlug))
                throw new Exception("CanPlug should return false: int / float");

            if (floatPlug.CanPlugTo(intPlug))
                throw new Exception("CanPlug should return false: float / int");
        }

        [TestMethod]
        public void PlugIntIsMatchingParameterInt()
        {
            var intPlug = new Plug("name", new ShaderTypeDefinition("int"), "target", false, null, null);
            var intParam = new ParameterDefinition(null, "target", new ShaderTypeDefinition("int"), ShaderTyping.ParameterDirection.Input, "description", null);

            if (!intPlug.IsMatchingParameter(intParam))
                throw new Exception("IsMatchingParameter should return true: int target / int target");
        }

        [TestMethod]
        public void PlugIntIsMatchingParameterFloat()
        {
            var intPlug = new Plug("name", new ShaderTypeDefinition("int"), "target", false, null, null);
            var floatParam = new ParameterDefinition(null, "target", new ShaderTypeDefinition("float"), ShaderTyping.ParameterDirection.Input, "description", null);

            if (intPlug.IsMatchingParameter(floatParam))
                throw new Exception("IsMatchingParameter should return false: int / float");
        }

        [TestMethod]
        public void PlugIsMatchingParameterField()
        {
            var vec2Plug = new Plug("name", new ShaderTypeDefinition("vec2"), "target.xy", false, null, null);
            var vec4Param = new ParameterDefinition(null, "target", new ShaderTypeDefinition("vec4"), ShaderTyping.ParameterDirection.Input, "description", null);

            if (!vec2Plug.IsMatchingParameter(vec4Param))
                throw new Exception("IsMatchingParameter should return true: vec2 target.xy / vec4 target");
        }

        [TestMethod]
        public void PlugIsMatchingParameterDifferentName()
        {
            var intPlug = new Plug("name", new ShaderTypeDefinition("int"), "target1", false, null, null);
            var intParam = new ParameterDefinition(null, "target2", new ShaderTypeDefinition("int"), ShaderTyping.ParameterDirection.Input, "description", null);

            if (intPlug.IsMatchingParameter(intParam))
                throw new Exception("IsMatchingParameter should return false: int target1 / int target2");
        }
    }
}
