﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Text;
using System.Collections.Generic;
using System.Linq;
using Microsoft.VisualStudio.TestTools.UnitTesting;

using EffectDefinitions;

namespace EffectDefinitionsUnitTests
{
    /// <summary>
    /// Unit tests for the FunctionDefinition class
    /// </summary>
    [TestClass]
    public class FunctionDefinitionUnitTest
    {
        #region ParameterDefinition

        [TestMethod]
        public void ParameterDefinitionConstructor()
        {
            var a = new ParameterDefinition(null, "myParamName", new ShaderTypeDefinition("int"), ShaderTyping.ParameterDirection.Reference, "My param usage", "My param description");

            var name = a.Name;
            if (name != "myParamName")
                throw new Exception(string.Format("Name should be \"myParamName\" but returned \"{0}\"", name));

            var type = a.Type.ToString();
            if (type != "int")
                throw new Exception(string.Format("Type should be \"int\" but returned \"{0}\"", type));

            var direction = a.Direction;
            if (direction != ShaderTyping.ParameterDirection.Reference)
                throw new Exception(string.Format("Direction should be reference but returned \"{0}\"", direction));

            var description = a.Description;
            var expectedDescription = "My param description";
            if (description != expectedDescription)
                throw new Exception(string.Format("Description should be \"{0}\" but returned \"{1}\"", expectedDescription, description));
        }

        [TestMethod]
        public void ParameterNameWithSpacesTest()
        {
            string name = "my Param Name";
            try
            {
                var a = new ParameterDefinition(null, name, new ShaderTypeDefinition("int"), ShaderTyping.ParameterDirection.Reference, null, null);
            }
            catch (System.ArgumentException)
            {
                return;
            }
            throw new Exception(string.Format("ParameterDefinition should not accept names with spaces, but it accepted [{0}]", name));
        }

        [TestMethod]
        public void ParameterNameStartingWithFigureTest()
        {
            string name = "1myParamName";
            try
            {
                var a = new ParameterDefinition(null, name, new ShaderTypeDefinition("int"), ShaderTyping.ParameterDirection.Reference, null, null);
            }
            catch (System.ArgumentException)
            {
                return;
            }
            throw new Exception(string.Format("ParameterDefinition should not accept names starting with numbers, but it accepted [{0}]", name));
        }

        [TestMethod]
        public void ParameterNameContainingInvalidCharsTest1()
        {
            string name = "my.Param.Name";
            try
            {
                var a = new ParameterDefinition(null, name, new ShaderTypeDefinition("int"), ShaderTyping.ParameterDirection.Reference, null, null);
            }
            catch (System.ArgumentException)
            {
                return;
            }
            throw new Exception(string.Format("ParameterDefinition should not accept names containing non alphanums, but it accepted [{0}]", name));
        }

        [TestMethod]
        public void ParameterNameContainingInvalidCharsTest2()
        {
            string name = "my+Param+Name";
            try
            {
                var a = new ParameterDefinition(null, name, new ShaderTypeDefinition("int"), ShaderTyping.ParameterDirection.Reference, null, null);
            }
            catch (System.ArgumentException)
            {
                return;
            }
            throw new Exception(string.Format("ParameterDefinition should not accept names containing non alphanums, but it accepted [{0}]", name));
        }

        #endregion

        #region FunctionDefinition

        [TestMethod]
        public void FunctionDefinitionConstructor()
        {
            var parameters = new List<ParameterDefinition>();
            string[] paramTypes = { "int", "float", "vec2", "vec3", "vec4" };
            foreach (string paramType in paramTypes)
            {
                string paramName = paramType + "Arg";
                string paramDesc = "Arg of type " + paramType;
                var parameter = new ParameterDefinition(null, paramName, new ShaderTypeDefinition(paramType), ShaderTyping.ParameterDirection.Input, null, paramDesc);
                parameters.Add(parameter);
            }

            var f = new FunctionDefinition(new DefinitionLocation(new Uri("file://mySource")), "myFunctionId", "myFunctionName", new ShaderTypeDefinition("int"), "My return usage", "My return description", "My description", parameters.ToArray());

            var name = f.Name;
            if (name != "myFunctionName")
                throw new Exception(string.Format("Name should be \"myFunctionName\" but returned \"{0}\"", name));

            var id = f.Id;
            if (id != "myFunctionId")
                throw new Exception(string.Format("Id should be \"myFunctionId\" but returned \"{0}\"", id));

            var returnType = f.ReturnType.ToString();
            if (returnType != "int")
                throw new Exception(string.Format("Return type should be \"int\" but returned \"{0}\"", returnType));

            var returnDescription = f.ReturnDescription;
            if (returnDescription != "My return description")
                throw new Exception(string.Format("Return description should be \"My return description\" but returned \"{0}\"", returnDescription));

            var description = f.Description;
            if (description != "My description")
                throw new Exception(string.Format("Description should be \"My description\" but returned \"{0}\"", returnDescription));

            var numberOfParams = f.Parameters.Length;
            if (numberOfParams != paramTypes.Length)
                throw new Exception(string.Format("There should be \"{1}\" arguments, but returned \"{0}\"", numberOfParams, paramTypes.Length));
        }

        [TestMethod]
        public void FunctionNameWithSpacesTest()
        {
            string name = "my Function Name";
            try
            {
                var f = new FunctionDefinition(new DefinitionLocation(new Uri("file://mySource")), "myFunctionId", name, new ShaderTypeDefinition("int"), "My return usage", "My return description", "My description");
            }
            catch (System.ArgumentException)
            {
                return;
            }
            throw new Exception(string.Format("FunctionDefinition should not accept names with spaces, but it accepted [{0}]", name));
        }

        [TestMethod]
        public void FunctionNameStartingWithFigureTest()
        {
            string name = "1myFunctionName";
            try
            {
                var f = new FunctionDefinition(new DefinitionLocation(new Uri("file://mySource")), "myFunctionId", name, new ShaderTypeDefinition("int"), "My return usage", "My return description", "My description");
            }
            catch (System.ArgumentException)
            {
                return;
            }
            throw new Exception(string.Format("FunctionDefinition should not accept names starting with numbers, but it accepted [{0}]", name));
        }

        [TestMethod]
        public void FunctionNameContainingInvalidCharsTest1()
        {
            string name = "my.Function.Name";
            try
            {
                var f = new FunctionDefinition(new DefinitionLocation(new Uri("file://mySource")), "myFunctionId", name, new ShaderTypeDefinition("int"), "My return usage", "My return description", "My description");
            }
            catch (System.ArgumentException)
            {
                return;
            }
            throw new Exception(string.Format("FunctionDefinition should not accept names containing non alphanums, but it accepted [{0}]", name));
        }

        [TestMethod]
        public void FunctionNameContainingInvalidCharsTest2()
        {
            string name = "my+Function+Name";
            try
            {
                var f = new FunctionDefinition(new DefinitionLocation(new Uri("file://mySource")), "myFunctionId", name, new ShaderTypeDefinition("int"), "My return usage", "My return description", "My description");
            }
            catch (System.ArgumentException)
            {
                return;
            }
            throw new Exception(string.Format("FunctionDefinition should not accept names containing non alphanums, but it accepted [{0}]", name));
        }


        #endregion
    }
}
