﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading;

namespace EffectCombiner.Core.IO
{
    /// <summary>
    /// ファイルIOユーティリティです。
    /// </summary>
    public static class IOUtility
    {
        /// <summary>
        /// テンポラリディレクトリ情報です。
        /// </summary>
        public static List<TemporaryDirectoryInfo> TemporaryDirectories { get; } = new List<TemporaryDirectoryInfo>();

        /// <summary>
        /// 例外が発生しないよう、安全にディレクトリを作成します。
        /// </summary>
        /// <param name="path">ディレクトリパス</param>
        /// <returns>ディレクトリを作成したときはtrue、それ以外はfalseを返します。</returns>
        public static bool SafeCreateDirectory(string path)
        {
            // パスの文字列が空でないかチェック
            if (string.IsNullOrEmpty(path))
            {
                return false;
            }

            // ディレクトリが存在するかチェック
            if (Directory.Exists(path))
            {
                return true;
            }

            // ディレクトリを作成
            try
            {
                Directory.CreateDirectory(path);
            }
            catch //(Exception e)
            {
                // Logger.LogException("Console", LogLevels.Warning, e);
            }

            // 途中で例外が発生しても、最終的にディレクトリが存在すれば成功とする
            // （並列実行時に、同時に別プロセスから作成されることがある）
            bool result = Directory.Exists(path);

            return result;
        }

        /// <summary>
        /// 例外が発生しないよう、安全にディレクトリを作成します。
        /// 既にディレクトリがある場合はは中身を空にします。
        /// </summary>
        /// <param name="path">ディレクトリパス</param>
        /// <returns>ディレクトリを作成したときはtrue、それ以外はfalseを返します。</returns>
        public static bool SafeCreateEmptyDirectory(string path)
        {
            // パスの文字列が空でないかチェック
            if (string.IsNullOrEmpty(path))
            {
                return false;
            }

            DirectoryInfo directoryInfo;

            // ディレクトリ情報を取得
            try
            {
                directoryInfo = new DirectoryInfo(path);
            }
            catch
            {
                return false;
            }

            // ディレクトリが存在したとき、ディレクトリ内のファイルを全て削除する
            if (directoryInfo.Exists)
            {
                foreach (FileSystemInfo fileInfo in directoryInfo.GetFileSystemInfos())
                {
                    try
                    {
                        // 読み取り専用のファイルを書き換え可能にする
                        fileInfo.Attributes = FileAttributes.Normal;

                        // ファイルを削除する
                        fileInfo.Delete();
                    }
                    catch
                    {
                    }
                }

                return true;
            }

            // ディレクトリを作成
            try
            {
                directoryInfo.Create();
            }
            catch // (Exception e)
            {
                // Logger.LogException("Console", LogLevels.Warning, e);
            }

            // 途中で例外が発生しても、最終的にディレクトリが存在すれば成功とする
            // （並列実行時に、同時に別プロセスから作成されることがある）
            bool result = Directory.Exists(path);

            return result;
        }

        /// <summary>
        /// 例外が発生しないよう、安全にディレクトリを削除します。
        /// </summary>
        /// <param name="path">ディレクトリパス</param>
        /// <param name="recursive">サブディレクトリ、およびファイルを削除する場合はtrue。それ以外の場合はfalse。</param>
        /// <returns>ディレクトリを削除したときはtrue、それ以外はfalseを返します。</returns>
        public static bool SafeDeleteDirectory(string path, bool recursive = false)
        {
            // パスの文字列が空でないかチェック
            if (string.IsNullOrEmpty(path))
            {
                return false;
            }

            // ディレクトリが存在するかチェック
            if (Directory.Exists(path) == false)
            {
                return true;
            }

            // ディレクトリを削除
            try
            {
                Directory.Delete(path, recursive);
            }
            catch
            {
            }

            // 途中で例外が発生しても、最終的にディレクトリが存在しなければ成功とする
            // （並列実行時に、同時に別プロセスから削除されることがある）
            bool result = (Directory.Exists(path) == false);

            return result;
        }

        /// <summary>
        /// 保持期限切れのファイルとディレクトリを削除します。
        /// </summary>
        /// <param name="path">ディレクトリのパス</param>
        /// <returns>ファイルとディレクトリの保持期限</returns>
        public static void DeleteExpiredFileSystemEntries(string path, TimeSpan span)
        {
            DeleteExpiredDirectories(path, span);

            DeleteExpiredFiles(path, span);
        }

        /// <summary>
        /// 保持期限切れのファイルを削除します。
        /// </summary>
        /// <param name="path">ディレクトリのパス</param>
        /// <returns>ファイルの保持期限</returns>
        public static void DeleteExpiredFiles(string path, TimeSpan span)
        {
            // ディレクトリの有無をチェック
            if (Directory.Exists(path) == false)
            {
                return;
            }

            // ファイルを列挙して保持期限切れのファイルを削除する
            try
            {
                foreach (string file in Directory.EnumerateFiles(path))
                {
                    try
                    {
                        // ファイルのタイムスタンプを取得
                        DateTime lastWriteTime = File.GetLastWriteTimeUtc(file);
                        TimeSpan elapsed = DateTime.Now.ToUniversalTime() - lastWriteTime;

                        // 期限切れのディレクトリを削除
                        if (elapsed > span)
                        {
                            File.Delete(file);
                        }
                    }
                    catch
                    {
                    }
                }
            }
            catch
            {
            }
        }

        /// <summary>
        /// 保持期限切れのディレクトリを削除します。
        /// </summary>
        /// <param name="path">ディレクトリのパス</param>
        /// <returns>ディレクトリの保持期限</returns>
        public static void DeleteExpiredDirectories(string path, TimeSpan span)
        {
            // ディレクトリの有無をチェック
            if (Directory.Exists(path) == false)
            {
                return;
            }

            // ディレクトリを列挙して保持期限切れのディレクトリを削除する
            try
            {
                foreach (string dir in Directory.EnumerateDirectories(path))
                {
                    try
                    {
                        // ディレクトリのタイムスタンプを取得
                        DateTime lastWriteTime = Directory.GetLastWriteTimeUtc(dir);
                        TimeSpan elapsed = DateTime.Now.ToUniversalTime() - lastWriteTime;

                        // 期限切れのディレクトリを削除
                        if (elapsed > span)
                        {
                            Directory.Delete(dir, true);
                        }
                    }
                    catch
                    {
                    }
                }
            }
            catch
            {
            }
        }

        /// <summary>
        /// File.WriteAllText() の拡張版です。
        /// ファイルの書き込みが成功するまで指定した回数だけ書き込みを試行します。
        /// </summary>
        /// <param name="path">書き込み先のファイル</param>
        /// <param name="contents">ファイルに書き込む文字列</param>
        /// <param name="encoding">文字列に適用するエンコーディング</param>
        /// <param name="retryCount">試行回数</param>
        /// <param name="interval">試行間隔 (ms)</param>
        /// <returns>ファイルへの書き込みが成功したときはtrue、それ以外はfalseを返します。</returns>
        public static bool WriteAllTextEx(string path, string contents, Encoding encoding, int retryCount, int interval)
        {
            bool result = false;

            for (int i = 0; i < retryCount; ++i)
            {
                try
                {
                    File.WriteAllText(path, contents, encoding);
                    result = true;
                }
                catch
                {
                }

                if (result)
                {
                    break;
                }

                Thread.Sleep(interval);
            }

            return result;
        }

        /// <summary>
        /// 例外が発生しないよう、安全にテンポラリディレクトリを作成します。
        /// このメソッドで作成したテンポラリディレクトリはアプリケーション終了時に
        /// DeleteDemporaryDirectories()を呼び出すことで削除されます。
        /// </summary>
        /// <param name="path">ディレクトリパス</param>
        /// <param name="usage">使用法</param>
        /// <returns>ディレクトリを作成したときはtrue、それ以外はfalseを返します。</returns>
        public static bool SafeCreateTemporaryDirectory(string path, TemporaryDirectoryUsage usage)
        {
            // テンポラリディレクトリを作成
            bool resCreate = SafeCreateDirectory(path);

            if (resCreate == false)
            {
                return false;
            }

            // テンポラリディレクトリ情報を更新
            if (TemporaryDirectories.Any(x => x.Path == path) == false)
            {
                TemporaryDirectoryInfo info = new TemporaryDirectoryInfo()
                {
                    Path = path,
                    Usage = usage
                };

                TemporaryDirectories.Add(info);
            }

            return true;
        }

        /// <summary>
        /// SafeCreateTemporaryDirectory()で作成したテンポラリディレクトリを一括で削除します。
        /// </summary>
        public static void SafeDeleteTemporaryDirectories()
        {
            // テンポラリディレクトリを作成したのとは逆の順番に削除
            foreach (TemporaryDirectoryInfo info in TemporaryDirectories.AsEnumerable().Reverse())
            {
                switch (info.Usage)
                {
                    case TemporaryDirectoryUsage.ForMyProcess:
                    {
                        SafeDeleteDirectory(info.Path, true);
                    } break;

                    case TemporaryDirectoryUsage.ForGlobal:
                    {
                        bool isEmpty = false;

                        try
                        {
                            isEmpty = (Directory.EnumerateFileSystemEntries(info.Path).Any() == false);
                        }
                        catch // (Exception e)
                        {
                            // Logger.LogException("Console", LogLevels.Warning, e);
                        }

                        if (isEmpty)
                        {
                            SafeDeleteDirectory(info.Path);
                        }
                    } break;
                }
            }
        }

        /// <summary>
        /// テンポラリディレクトリの使用法です。
        /// </summary>
        public enum TemporaryDirectoryUsage
        {
            /// <summary>
            /// 手動管理するテンポラリディレクトリです。
            /// SafeDeleteTemporaryDirectories() で削除されません。
            /// </summary>
            Manual,

            /// <summary>
            /// 今のプロセス専用のテンポラリディレクトリです。
            /// ディレクトリに内容があるときも SafeDeleteTemporaryDirectories() で削除されます。
            /// </summary>
            ForMyProcess,

            /// <summary>
            /// 全てのプロセスで共有するディレクトリです。
            /// ディレクトリが空のとき、SafeDeleteTemporaryDirectories() で削除されます。
            /// </summary>
            ForGlobal
        }

        /// <summary>
        /// テンポラリディレクトリ情報です。
        /// </summary>
        public struct TemporaryDirectoryInfo
        {
            /// <summary>
            /// パスを取得または設定します。
            /// </summary>
            public string Path { get; set; }

            /// <summary>
            /// 使用法を取得または設定します。
            /// </summary>
            public TemporaryDirectoryUsage Usage { get; set; }
        }
    }
}
