﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Windows.Forms;

namespace EffectCombiner.Core
{
    public abstract class FileManagerBase
    {
        protected abstract string CloseDialogMessage { get; }
        protected abstract string CloseDialogTitle { get; }
        protected abstract void OnClose();

        protected abstract string DialogBoxFilters { get; }
        protected abstract string OpenDialogTitle { get; }
        protected abstract string SaveDialogTitle { get; }

        protected abstract void OnLoad(string filename);
        protected abstract void OnSave(string filename);

        private bool isModified;
        public bool IsModified
        {
            get { return isModified; }
            set
            {
                if (isModified == value)
                    return;

                isModified = value;
                OnIsModifiedChanged();
            }
        }

        public event EventHandler IsModifiedChanged;

        protected virtual void OnIsModifiedChanged()
        {
            var handler = IsModifiedChanged;
            if (handler != null)
                handler(this, EventArgs.Empty);
        }

        private string filename;
        public string Filename
        {
            get { return filename; }
            private set
            {
                if (filename != value)
                {
                    filename = value;
                    OnFilenameChanged();
                }
            }
        }

        public event EventHandler FilenameChanged;

        protected virtual void OnFilenameChanged()
        {
            var handler = FilenameChanged;
            if (handler != null)
                handler(this, EventArgs.Empty);
        }

        private bool InternalClose()
        {
            var result = true;

            if (IsModified)
            {
                var dlgResult = MessageBox.Show(CloseDialogMessage, CloseDialogTitle,
                    MessageBoxButtons.YesNoCancel, MessageBoxIcon.Question);
                if (dlgResult == DialogResult.Cancel)
                    return false;

                if (dlgResult == DialogResult.Yes)
                    result = Filename != null ? InternalSave() : InternalSaveAs();
            }

            if (result)
            {
                OnClose();
                IsModified = false;
                Filename = null;

                var handler = Closed;
                if (handler != null)
                    handler(this, EventArgs.Empty);
            }

            return result;
        }

        private bool InternalOpen()
        {
            if (IsModified)
            {
                if (InternalClose() == false)
                    return false;
            }

            var dlg = new OpenFileDialog
            {
                Title = OpenDialogTitle,
                Filter = DialogBoxFilters,
                Multiselect = false,
                ShowHelp = true,
            };

            if (dlg.ShowDialog() != DialogResult.OK)
                return false;

            InternalOpenWithoutConfirm(dlg.FileName);

            return true;
        }

        private bool InternalOpen(string fileName)
        {
            if (IsModified)
            {
                if (InternalClose() == false)
                    return false;
            }

            InternalOpenWithoutConfirm(fileName);

            return true;
        }

        private void InternalOpenWithoutConfirm(string fileName)
        {
            OnLoad(fileName);

            Filename = fileName;
            IsModified = false;

            var handler = Opened;
            if (handler != null)
                handler(this, EventArgs.Empty);
        }

        public event EventHandler Closed;
        public event EventHandler Opened;
        public event EventHandler Saved;

        private bool InternalSave()
        {
            if (IsModified == false)
                return true;

            if (Filename == null)
                return InternalSaveAs();

            OnSave(Filename);

            IsModified = false;

            var handler = Saved;
            if (handler != null)
                handler(this, EventArgs.Empty);

            return true;
        }

        private bool InternalSaveAs()
        {
            var dlg = new SaveFileDialog
            {
                Title = SaveDialogTitle,
                Filter = DialogBoxFilters,
                ShowHelp = true,
            };

            if (dlg.ShowDialog() != DialogResult.OK)
                return false;

            OnSave(dlg.FileName);

            Filename = dlg.FileName;
            IsModified = false;

            var handler = Saved;
            if (handler != null)
                handler(this, EventArgs.Empty);

            return true;
        }

        private bool InternalApplicationClose()
        {
            return InternalClose();
        }

        // ================================================================================

        public void Close()
        {
            InternalClose();
        }

        public void Open()
        {
            InternalOpen();
        }

        public void Open(string fileName)
        {
            InternalOpen(fileName);
        }

        public void Save()
        {
            InternalSave();
        }

        public void SaveAs()
        {
            InternalSaveAs();
        }

        public virtual void ApplicationClose(out bool cancel)
        {
            cancel = true;
            if (InternalApplicationClose())
                cancel = false;
        }
    }
}
