﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Xml.Serialization;
using System.IO;
using System.ComponentModel;
using EffectCombiner.Core;

namespace EffectCombiner.Primitives
{
    public class DesactivablePath
    {
        [XmlAttribute]
        public bool IsActive = true;

        [XmlAttribute]
        public string Path;
    }

    public class ShaderGeneration
    {
        public bool WriteShaderFile = true;
        public string OutputShaderFilename;
        public DesactivablePath[] LinkDefinitionsShaderFileNames = new DesactivablePath[0];
        public bool ShowShaderCodeWindow;
        public bool TryCompileShader = true;
        public bool AllowShaderGenerationOnNotProperlyConnectedGraph = true;
        public string MainFunctionName = "main";
    }

    public class DefinitionPathInfo
    {
        [XmlAttribute]
        public bool IsActive = true;

        [XmlAttribute]
        public string Path;

        [XmlAttribute]
        public bool IsRecursive;

        [XmlAttribute]
        public string Extensions;
    }

    public class DefinitionPathInfos
    {
        public DefinitionPathInfo[] Paths = new DefinitionPathInfo[0];

        [XmlAttribute]
        public bool ReloadDefinitionsOnPathChange = true;
    }

    public class PlugSnappingSettings
    {
        [XmlAttribute]
        public bool IsActive = true;

        [XmlAttribute]
        public double MinimumDistance = 50.0;
    }

    public class WindowSettings
    {
        [XmlAttribute]
        public int Left = -1;

        [XmlAttribute]
        public int Top = -1;

        [XmlAttribute]
        public int Width = -1;

        [XmlAttribute]
        public int Height = -1;

        [XmlAttribute]
        public bool FullScreen;
    }

    public class EnvironmentSettings
    {
        public WindowSettings MainWindow = new WindowSettings();
        public WindowSettings OptionsWindow = new WindowSettings();
        public WindowSettings EventsWindow = new WindowSettings();
        public WindowSettings OperationManagerWindow = new WindowSettings();

        public bool CompletelyIncludeToSelect = true;
        public bool DrawTextShadow;

        public PlugSnappingSettings PlugSnapping = new PlugSnappingSettings();

        public string PreSaveCommandPath;
        public string PostSaveCommandPath;
        public string PreOpenCommandPath;

        public string ProjectSettingFilePath;
    }

    /// <summary>
    /// プロジェクト設定です。
    /// </summary>
    public class ProjectSettings
    {
        /// <summary>
        /// 定義ファイルパスリスト
        /// </summary>
        public DefinitionPathInfo[] DefinitionPaths = new DefinitionPathInfo[0];

        /// <summary>
        /// 保存前コマンドパス
        /// </summary>
        public string PreSaveCommandPath;

        /// <summary>
        /// 保存後コマンドパス
        /// </summary>
        public string PostSaveCommandPath;

        /// <summary>
        /// オープン前コマンドパス
        /// </summary>
        public string PreOpenCommandPath;
    }

    public class LanguageSettings
    {
        public string CurrentLanguage;
    }

    public class Options
    {
        public ShaderGeneration ShaderGeneration = new ShaderGeneration();
        public DefinitionPathInfos DefinitionPaths = new DefinitionPathInfos();
        public EnvironmentSettings EnvironmentSettings = new EnvironmentSettings();
        public LanguageSettings LanguageSettings = new LanguageSettings();

        [XmlIgnore]
        public ProjectSettings ProjectSettings = new ProjectSettings();

        public string[] RecentProjects = new string[0];

        /// <summary>
        /// 保存前コマンドパスを取得します。
        /// このパスはプロジェクト設定とユーザー設定をマージした結果です。
        /// </summary>
        public string PreSaveCommandPath
        {
            get
            {
                if (string.IsNullOrEmpty(this.EnvironmentSettings.PreSaveCommandPath) == false)
                {
                    string basePath = AppDomain.CurrentDomain.BaseDirectory;
                    string path = PathUtility.ToAbsolutePath(this.EnvironmentSettings.PreSaveCommandPath, basePath);

                    return path;
                }
                else
                {
                    string basePath = string.IsNullOrEmpty(this.EnvironmentSettings.ProjectSettingFilePath) ?
                        string.Empty : Path.GetDirectoryName(this.EnvironmentSettings.ProjectSettingFilePath);
                    string path = PathUtility.ToAbsolutePath(this.ProjectSettings.PreSaveCommandPath, basePath);

                    return path;
                }
            }
        }

        /// <summary>
        /// 保存後コマンドパスを取得します。
        /// このパスはプロジェクト設定とユーザー設定をマージした結果です。
        /// </summary>
        public string PostSaveCommandPath
        {
            get
            {
                if (string.IsNullOrEmpty(this.EnvironmentSettings.PostSaveCommandPath) == false)
                {
                    string basePath = AppDomain.CurrentDomain.BaseDirectory;
                    string path = PathUtility.ToAbsolutePath(this.EnvironmentSettings.PostSaveCommandPath, basePath);

                    return path;
                }
                else
                {
                    string basePath = string.IsNullOrEmpty(this.EnvironmentSettings.ProjectSettingFilePath) ?
                        string.Empty : Path.GetDirectoryName(this.EnvironmentSettings.ProjectSettingFilePath);
                    string path = PathUtility.ToAbsolutePath(this.ProjectSettings.PostSaveCommandPath, basePath);

                    return path;
                }
            }
        }

        /// <summary>
        /// オープン前コマンドパスを取得します。
        /// このパスはプロジェクト設定とユーザー設定をマージした結果です。
        /// </summary>
        public string PreOpenCommandPath
        {
            get
            {
                if (string.IsNullOrEmpty(this.EnvironmentSettings.PreOpenCommandPath) == false)
                {
                    string basePath = AppDomain.CurrentDomain.BaseDirectory;
                    string path = PathUtility.ToAbsolutePath(this.EnvironmentSettings.PreOpenCommandPath, basePath);

                    return path;
                }
                else
                {
                    string basePath = string.IsNullOrEmpty(this.EnvironmentSettings.ProjectSettingFilePath) ?
                        string.Empty : Path.GetDirectoryName(this.EnvironmentSettings.ProjectSettingFilePath);
                    string path = PathUtility.ToAbsolutePath(this.ProjectSettings.PreOpenCommandPath, basePath);

                    return path;
                }
            }
        }

        /// <summary>
        /// 定義ファイルパスリストを取得します。
        /// このリストはプロジェクト設定とユーザー設定をマージした結果です。
        /// </summary>
        public PathLookupInfo[] GetPathLookupInfo()
        {
            DefinitionPathInfo[] projectDefinitionPaths = null;

            if (this.ProjectSettings != null)
            {
                projectDefinitionPaths = this.ProjectSettings.DefinitionPaths;
            }

            if (projectDefinitionPaths == null)
            {
                projectDefinitionPaths = new DefinitionPathInfo[0];
            }

            DefinitionPathInfo[] userDefinitionPaths = null;

            if (this.DefinitionPaths != null)
            {
                userDefinitionPaths = this.DefinitionPaths.Paths;
            }

            if (userDefinitionPaths == null)
            {
                userDefinitionPaths = new DefinitionPathInfo[0];
            }

            int capacity = projectDefinitionPaths.Length + userDefinitionPaths.Length;
            List<PathLookupInfo> lookupInfoList = new List<PathLookupInfo>(capacity);

            foreach (DefinitionPathInfo pathInfo in projectDefinitionPaths)
            {
                if (pathInfo.IsActive && string.IsNullOrWhiteSpace(pathInfo.Path) == false)
                {
                    string basePath = Path.GetDirectoryName(this.EnvironmentSettings.ProjectSettingFilePath);
                    string path = PathUtility.ToAbsolutePath(pathInfo.Path.Trim(), basePath);

                    PathLookupInfo lookupInfo = new PathLookupInfo(
                        path,
                        pathInfo.IsRecursive,
                        CoreUtility.Split(pathInfo.Extensions, ',', ';', ' '));

                    lookupInfoList.Add(lookupInfo);
                }
            }

            foreach (DefinitionPathInfo pathInfo in userDefinitionPaths)
            {
                if (pathInfo.IsActive && string.IsNullOrWhiteSpace(pathInfo.Path) == false)
                {
                    string basePath = AppDomain.CurrentDomain.BaseDirectory;
                    string path = PathUtility.ToAbsolutePath(pathInfo.Path.Trim(), AppDomain.CurrentDomain.BaseDirectory);

                    PathLookupInfo lookupInfo = new PathLookupInfo(
                        path,
                        pathInfo.IsRecursive,
                        CoreUtility.Split(pathInfo.Extensions, ',', ';', ' '));

                    lookupInfoList.Add(lookupInfo);
                }
            }

            return lookupInfoList.ToArray();
        }

        public void PostProcessValues()
        {
            // check plug snapping distance boundaries

            var minDistance = Globals.VisualResources.PlugRadius * 1.5;
            var maxDistance = Globals.VisualResources.PlugRadius * 10.0;

            if (EnvironmentSettings.PlugSnapping.MinimumDistance < minDistance)
                EnvironmentSettings.PlugSnapping.MinimumDistance = minDistance;
            else if (EnvironmentSettings.PlugSnapping.MinimumDistance > maxDistance)
                EnvironmentSettings.PlugSnapping.MinimumDistance = maxDistance;

            // add more checking here...
        }
    }
}
